/*
 * threshold.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "game.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --who=[who]\n"
	   "  --abstraction=[abstraction]\n"
	   "  --threshold=[threshold]\n"
	   "  --x=[input]\n"
	   "  --output=[output]\n"
	   "  --output-lifted=[output]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction", required_argument, 0, 6},
      {"x", required_argument, 0, 8},
      {"who", required_argument, 0, 9},
      {"output", required_argument, 0, 10},
      {"output-lifted", required_argument, 0, 11},
      {"threshold", required_argument, 0, 12},
      {NULL},
    };
    int ch, who;
    game_t game;
    abstraction_t abstraction, abstraction_full;
    sequences_t sequences, sequences_full;
    double * sigma, * sigma_lifted, threshold;
    const char * game_path;
    const char * abstraction_path;
    const char * x_path, * output_path, * output_lifted_path;
    FILE * stream;

    game_path = NULL;
    abstraction_path = "null";
    x_path = output_path = output_lifted_path = NULL;
    who = -1;
    threshold = -1;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_path = optarg; break;
      case 8: x_path = optarg; break;
      case 9: verify(sscanf(optarg, "%d", &who) == 1); break;
      case 10: output_path = optarg; break;
      case 11: output_lifted_path = optarg; break;
      case 12: verify(sscanf(optarg, "%lf", &threshold) == 1); break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);
    verify(x_path);
    verify(valid_player(who));
    verify(threshold > 0);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_path, &game, who, &abstraction);
    null_abstraction(&game, who, &abstraction_full);

    build_sequences(&abstraction, &sequences);
    build_sequences(&abstraction_full, &sequences_full);
    
    sigma        = new_strategy(&sequences);
    sigma_lifted = new_strategy(&sequences_full);

    stream = open_stream(x_path, "rt");
    read_strategy(&sequences, stream, sigma);
    close_stream(x_path, stream);
    normalize_strategy(&sequences, sigma, 1.);

    threshold_strategy(&sequences, sigma, threshold);

    if (output_path) {
      stream = open_stream(output_path, "wt");
      write_strategy(&sequences, sigma, stream);
      close_stream(output_path, stream);
    }

    lift_strategy(&sequences, &sequences_full, sigma, sigma_lifted);
    if (output_lifted_path) {
      stream = open_stream(output_lifted_path, "wt");
      write_strategy(&sequences_full, sigma_lifted, stream);
      close_stream(output_lifted_path, stream);
    }

    xfree(sigma_lifted);
    xfree(sigma);

    free_sequences(&sequences_full);
    free_sequences(&sequences);
    
    free_abstraction(&abstraction_full);
    free_abstraction(&abstraction);

    free_game(&game);
  }
  return 0;
}
