/*
 * strategy.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _STRATEGY_H_
#define _STRATEGY_H_

#ifdef BLAS_HEADER
#include BLAS_HEADER
#endif /* BLAS_HEADER */

#include <stdio.h>
#include <string.h>

#include "attributes.h"

#include "sequence_form.h"
#include "util.h"
#include "verify.h"

void read_strategy(sequences_t * restrict sequences, FILE * restrict stream, 
		   double * restrict sigma) NONNULL(1,2,3);
void write_strategy(sequences_t * restrict sequences, const double * restrict sigma, 
		    FILE * restrict stream) NONNULL(1,2);
void write_strategy_verbose(sequences_t * restrict sequences, const double * restrict sigma, 
		    FILE * restrict stream) NONNULL(1,2);
int is_strategy(sequences_t * restrict sequences, const double * restrict sigma) NONNULL(1,2) PURE;
void normalize_strategy(sequences_t * restrict sequences, double * restrict sigma, 
			double alpha) NONNULL(1,2);
void lift_strategy(sequences_t * restrict from, sequences_t * restrict to,
		   const double * restrict sigma, double * restrict sigma2) NONNULL(1,2,3,4);
void flatten_payoffs(sequences_t * restrict from, sequences_t * restrict to,
		     const double * restrict payoffs, double * restrict payoffs2) NONNULL(1,2,3,4);
void purify_strategy(sequences_t * restrict sequences, double * restrict sigma) NONNULL(1,2);
void threshold_strategy(sequences_t * restrict sequences, double * restrict sigma, double threshold) NONNULL(1,2);
void average_strategy(sequences_t * restrict sequences, const double * restrict new, double * restrict average, int t);
double * sampled_payoffs(sequence_form_t * sequence_form, int who, const double * restrict sigma,
			 double * restrict payoffs, double * restrict chance, int * restrict q) NONNULL(1,3,5,6);

static inline double * new_strategy(sequences_t * restrict sequences) NONNULL(1);
static inline double * new_uniform_strategy(sequences_t * restrict sequences) NONNULL(1);
static inline void uniform_strategy(sequences_t * restrict sequences, double * restrict sigma) NONNULL(1,2);
static inline void zero_strategy(sequences_t * restrict sequences, 
				 double * restrict sigma) NONNULL(1);
static inline void copy_strategy(sequences_t * restrict sequences,
				 const double * restrict from, double * restrict to) NONNULL(1,2,3);
static inline double * compute_payoffs(sequence_form_t * restrict sequence_form, int who, 
				       const double * restrict sigma, 
				       double * restrict payoffs) NONNULL(1,3);
static inline double compute_ev(sequence_form_t * restrict sequence_form, int who, 
				const double * restrict sigma,
				const double * restrict payoffs) NONNULL(1,3,4);

static inline double * new_strategy(sequences_t * restrict sequences) {
  return xcalloc(sequences_n(sequences), sizeof(double));
}

static inline double * new_uniform_strategy(sequences_t * restrict sequences) {
  double * sigma;
  sigma = new_strategy(sequences);
  normalize_strategy(sequences, sigma, 1.);
  return sigma;
}

static inline void uniform_strategy(sequences_t * restrict sequences, double * restrict sigma) {
  assert(sequences);
  assert(sigma);
  zero_strategy(sequences, sigma);
  normalize_strategy(sequences, sigma, 1.);
}

static inline void zero_strategy(sequences_t * restrict sequences, double * restrict sigma) {
  memset(sigma, 0, sequences_n(sequences)*sizeof(double));
}

static inline void copy_strategy(sequences_t * restrict sequences,
				 const double * restrict from, double * restrict to) {
  memcpy(to, from, sequences_n(sequences)*sizeof(double));
}

static inline double * compute_payoffs(sequence_form_t * restrict sequence_form, int who, 
				       const double * restrict sigma, double * restrict payoffs) {
  assert(sequence_form);
  assert(valid_player(who));
  assert(sigma);
  
  if (!payoffs) {
    payoffs = new_strategy(sequence_form_sequences(sequence_form, who));
  }

  int m  = sequences_n(sequence_form_sequences(sequence_form, 0));
  int n  = sequences_n(sequence_form_sequences(sequence_form, 1));
#ifdef BLAS_HEADER
  cblas_dgemv(CblasRowMajor, who?CblasTrans:CblasNoTrans, 
	      m, n, who?-1.:1., sequence_form->A, sequences_n(sequence_form_sequences(sequence_form, 1)), 
	      sigma, 1, 0., payoffs, 1);
#else
  if (who) {
    /* transposed, negated */
    zero_strategy(sequence_form_sequences(sequence_form, who), payoffs);
    for(int k=0, i=0; i<m; ++i) {
      for(int j=0; j<n; ++j, ++k) {
	payoffs[j] -= sequence_form->A[k]*sigma[i];
      }
    }   
  } else {
    for(int k=0, i=0; i<m; ++i) {
      payoffs[i] = 0;
      for(int j=0; j<n; ++j, ++k) {
	payoffs[i] += sequence_form->A[k]*sigma[j];
      }
    }
  }
#endif /* BLAS_HEADER */

  return payoffs;
}

static inline double compute_ev(sequence_form_t * restrict sequence_form, int who, 
				const double * restrict sigma, const double * restrict payoffs) {
  assert(sigma);
  assert(payoffs);

#ifdef BLAS_HEADER
  return cblas_ddot(sequences_n(sequence_form_sequences(sequence_form, who)),
		    sigma, 1, payoffs, 1);
#else
  int n      = sequences_n(sequence_form_sequences(sequence_form, who));
  double dot = 0;
  for(int i=0; i<n; ++i) {
    dot += sigma[i]*payoffs[i];
  }
  return dot;
#endif /* BLAS_HEADER */
}

#endif /* _STRATEGY_H_ */
