/*
 * solve_regret.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>
#include <time.h>

#include "abstraction.h"
#include "best_response.h"
#include "game.h"
#include "player.h"
#include "lp_eqm.h"
#include "rng.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --x-abstraction1=[abstraction]\n"
	   "  --x-abstraction2=[abstraction]\n"
	   "  --y-abstraction1=[abstraction]\n"
	   "  --y-abstraction2=[abstraction]\n"
	   "  --x=[output]\n"
	   "  --y=[output]\n"
	   "  --x-lifted=[output]\n"
	   "  --y-lifted=[output]\n"
	   "  --x-full=[output]\n"
	   "  --y-full=[output]\n"
	   "  --player1=[player]\n"
	   "  --player2=[player]\n"
	   "  --params1=[params]\n"
	   "  --params2=[params]\n"
	   "  --T=[time steps]\n"
	   "  --sample-chance\n"
	   "  --seed=[seed]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"x-abstraction1", required_argument, 0, 2},
      {"x-abstraction2", required_argument, 0, 3},
      {"y-abstraction1", required_argument, 0, 4},
      {"y-abstraction2", required_argument, 0, 5},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"x", required_argument, 0, 8},
      {"y", required_argument, 0, 9},
      {"x-lifted", required_argument, 0, 10},
      {"y-lifted", required_argument, 0, 11},
      {"player1", required_argument, 0, 12},
      {"player2", required_argument, 0, 13},
      {"params1", required_argument, 0, 14},
      {"params2", required_argument, 0, 15},
      {"T", required_argument, 0, 16},
      {"sample-chance", no_argument, 0, 17},
      {"seed", required_argument, 0, 18},
      {"x-null", required_argument, 0, 19},
      {"y-null", required_argument, 0, 20},
      {NULL},
    };
    int ch, t, T, j, seed, sample_chance, * q;
    game_t game;
    abstraction_t abstraction_x[2], abstraction_y[2], abstraction_original[2], abstraction_null[2];
    sequences_t sequences_x[2], sequences_y[2], sequences_original[2], sequences_null[2];
    sequence_form_t sequence_form_x, sequence_form_y, sequence_form_original;
    const double * sigma_x, * sigma_y;
    double * sigma_avg_x, * sigma_avg_y, * sigma_avg_x_lifted, * sigma_avg_y_lifted;
    double * sigma_x_lifted, * sigma_y_lifted, * sigma_avg_x_null, * sigma_avg_y_null;
    double * payoffs_x, * payoffs_y, * payoffs_x_flattened, * payoffs_y_flattened, * chance;
    const char * game_path;
    const char * x_path, * y_path, * x_path_lifted, * y_path_lifted, * x_path_null, * y_path_null;
    const char * abstraction_x_name[2], * abstraction_y_name[2], * abstraction_original_name[2];
    const char * player_type[2], * params[2];
    player_t player[2];
    double ev_t, br0, br1, eps;
    FILE * stream;
    rng_t * rng;
    
    game_path = NULL;
    abstraction_x_name[0] = abstraction_x_name[1] = NULL;
    abstraction_y_name[0] = abstraction_y_name[1] = NULL;
    abstraction_original_name[0] = abstraction_original_name[1] = "null";
    x_path = y_path = NULL;
    x_path_lifted = y_path_lifted = x_path_null = y_path_null = NULL;
    player_type[0] = player_type[1] = params[0] = params[1] = NULL;
    T = -1;
    sample_chance = 0;
    seed = time(NULL);
    
    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 2: abstraction_x_name[0] = optarg; break;
      case 3: abstraction_x_name[1] = optarg; break;
      case 4: abstraction_y_name[0] = optarg; break;
      case 5: abstraction_y_name[1] = optarg; break;
      case 6: abstraction_original_name[0] = optarg; break;
      case 7: abstraction_original_name[1] = optarg; break;
      case 8: x_path = optarg; break;
      case 9: y_path = optarg; break;
      case 10: x_path_lifted = optarg; break;
      case 11: y_path_lifted = optarg; break;
      case 12: player_type[0] = optarg; break;
      case 13: player_type[1] = optarg; break;
      case 14: params[0] = optarg; break;
      case 15: params[1] = optarg; break;
      case 16: verify(sscanf(optarg, "%d", &T) == 1); break;
      case 17: sample_chance = 1; break;
      case 18: verify(sscanf(optarg, "%d", &seed) == 1); break;
      case 19: x_path_null = optarg; break;
      case 20: y_path_null = optarg; break;
      default: verify("unknown argument");
      }
    }

    verify(game_path);
    verify(player_type[0]);
    verify(player_type[1]);
    verify(T >= 0);

    if (!abstraction_x_name[0]) {
      abstraction_x_name[0] = abstraction_original_name[0];
    }
    if (!abstraction_x_name[1]) {
      abstraction_x_name[1] = abstraction_original_name[1];
    }
    if (!abstraction_y_name[0]) {
      abstraction_y_name[0] = abstraction_original_name[0];
    }
    if (!abstraction_y_name[1]) {
      abstraction_y_name[1] = abstraction_original_name[1];
    }

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_original_name[0], &game, 0, &abstraction_original[0]);
    load_abstraction(abstraction_original_name[1], &game, 1, &abstraction_original[1]);
    load_abstraction(abstraction_x_name[0], &game, 0, &abstraction_x[0]);
    load_abstraction(abstraction_x_name[1], &game, 1, &abstraction_x[1]);
    load_abstraction(abstraction_y_name[0], &game, 0, &abstraction_y[0]);
    load_abstraction(abstraction_y_name[1], &game, 1, &abstraction_y[1]);
    null_abstraction(&game, 0, &abstraction_null[0]);
    null_abstraction(&game, 1, &abstraction_null[1]);
    verify(is_coarser(&abstraction_x[0], &abstraction_original[0]));
    verify(is_coarser(&abstraction_x[1], &abstraction_original[1]));
    verify(is_coarser(&abstraction_y[0], &abstraction_original[0]));
    verify(is_coarser(&abstraction_y[1], &abstraction_original[1]));

    build_sequences(&abstraction_original[0], &sequences_original[0]);
    build_sequences(&abstraction_original[1], &sequences_original[1]);
    build_sequences(&abstraction_x[0], &sequences_x[0]);
    build_sequences(&abstraction_x[1], &sequences_x[1]);
    build_sequences(&abstraction_y[0], &sequences_y[0]);
    build_sequences(&abstraction_y[1], &sequences_y[1]);
    build_sequences(&abstraction_null[0], &sequences_null[0]);
    build_sequences(&abstraction_null[1], &sequences_null[1]);
    
    build_sequence_form(&sequences_original[0], &sequences_original[1], &sequence_form_original);
    build_sequence_form(&sequences_x[0], &sequences_x[1], &sequence_form_x);
    build_sequence_form(&sequences_y[0], &sequences_y[1], &sequence_form_y);
    
    sigma_avg_x         = new_uniform_strategy(&sequences_x[0]);
    sigma_avg_y         = new_uniform_strategy(&sequences_y[1]);
    sigma_x_lifted      = new_strategy(&sequences_original[0]);
    sigma_y_lifted      = new_strategy(&sequences_original[1]);
    sigma_avg_x_lifted  = new_strategy(&sequences_original[0]);
    sigma_avg_y_lifted  = new_strategy(&sequences_original[1]);
    sigma_avg_x_null    = new_strategy(&sequences_null[0]);
    sigma_avg_y_null    = new_strategy(&sequences_null[1]);
    payoffs_x           = new_strategy(&sequences_original[0]);
    payoffs_y           = new_strategy(&sequences_original[1]);
    payoffs_x_flattened = new_strategy(&sequences_x[0]);
    payoffs_y_flattened = new_strategy(&sequences_y[1]);
        
    create_player(player_type[0], &sequence_form_x, 0, T, params[0], &player[0]);
    create_player(player_type[1], &sequence_form_y, 1, T, params[1], &player[1]);

    rng = xmalloc(sizeof(rng_t));
    rng_init_gen_rand(rng, seed);

    q      = xmalloc(sizeof(int)*total_histories(&game));
    chance = xmalloc(sizeof(double)*max_chance_depth(&game));
    
    ev_t = 0.;
    for(t=1; t<=T; ++t) {
      sigma_x = get_strategy(&player[0]);
      sigma_y = get_strategy(&player[1]);
	
      assert(is_strategy(&sequences_x[0], sigma_x));
      assert(is_strategy(&sequences_y[1], sigma_y));

      average_strategy(&sequences_x[0], sigma_x, sigma_avg_x, t);
      average_strategy(&sequences_y[1], sigma_y, sigma_avg_y, t);

      lift_strategy(&sequences_x[0], &sequences_original[0], sigma_avg_x, sigma_avg_x_lifted);
      lift_strategy(&sequences_y[1], &sequences_original[1], sigma_avg_y, sigma_avg_y_lifted);

      compute_payoffs(&sequence_form_original, 0, sigma_avg_y_lifted, payoffs_x);
      compute_payoffs(&sequence_form_original, 1, sigma_avg_x_lifted, payoffs_y);
      
      ev_t = compute_ev(&sequence_form_original, 0, sigma_avg_x_lifted, payoffs_x);
      br0  = best_response(&sequences_original[0], payoffs_x);
      br1  = best_response(&sequences_original[1], payoffs_y);
      eps  = br0 - ev_t;
      if (eps < br1 + ev_t) {
	eps = br1 + ev_t;
      }
	
      printf("t=%d ev=%g br0=%g br1=%g eps=%g\n", t, ev_t, br0, br1, eps);
      
      if (t < T) {
	lift_strategy(&sequences_x[0], &sequences_original[0], sigma_x, sigma_x_lifted);
	lift_strategy(&sequences_y[1], &sequences_original[1], sigma_y, sigma_y_lifted);

	if (!sample_chance) {
	  compute_payoffs(&sequence_form_original, 0, sigma_y_lifted, payoffs_x);
	  compute_payoffs(&sequence_form_original, 1, sigma_x_lifted, payoffs_y);
	} else {
	  for(j=0; j<max_chance_depth(&game); ++j) {
	    chance[j] = rng_genrand_close_open(rng);
	  }
	  sampled_payoffs(&sequence_form_original, 0, sigma_y_lifted, payoffs_x, chance, q);
	  sampled_payoffs(&sequence_form_original, 1, sigma_x_lifted, payoffs_y, chance, q);
	}
		
	flatten_payoffs(&sequences_original[0], &sequences_x[0], payoffs_x, payoffs_x_flattened);
	flatten_payoffs(&sequences_original[1], &sequences_y[1], payoffs_y, payoffs_y_flattened);

	update_full_strategy(&player[0], payoffs_x_flattened);
	update_full_strategy(&player[1], payoffs_y_flattened);
      }
    }

    printf("game's value = %g\n", ev_t);

    free_player(&player[1]);
    free_player(&player[0]);
    
    if (x_path) {
      stream = open_stream(x_path, "wt");
      write_strategy(&sequences_x[0], sigma_avg_x, stream);
      close_stream(x_path, stream);
    }
    if (y_path) {
      stream = open_stream(y_path, "wt");
      write_strategy(&sequences_y[1], sigma_avg_y, stream);
      close_stream(y_path, stream);
    }
    
    lift_strategy(&sequences_x[0], &sequences_original[0], sigma_avg_x, sigma_avg_x_lifted);
    lift_strategy(&sequences_y[1], &sequences_original[1], sigma_avg_y, sigma_avg_y_lifted);
    if (x_path_lifted) {
      stream = open_stream(x_path_lifted, "wt");
      write_strategy(&sequences_original[0], sigma_avg_x_lifted, stream);
      close_stream(x_path_lifted, stream);
    }
    if (y_path_lifted) {
      stream = open_stream(y_path_lifted, "wt");
      write_strategy(&sequences_original[1], sigma_avg_y_lifted, stream);
      close_stream(y_path_lifted, stream);
    }

    lift_strategy(&sequences_x[0], &sequences_null[0], sigma_avg_x, sigma_avg_x_null);
    lift_strategy(&sequences_y[1], &sequences_null[1], sigma_avg_y, sigma_avg_y_null);
    if (x_path_null) {
      stream = open_stream(x_path_null, "wt");
      write_strategy(&sequences_null[0], sigma_avg_x_null, stream);
      close_stream(x_path_null, stream);
    }
    if (y_path_null) {
      stream = open_stream(y_path_null, "wt");
      write_strategy(&sequences_null[1], sigma_avg_y_null, stream);
      close_stream(y_path_null, stream);
    }
    
    xfree(chance);
    xfree(q);
    
    xfree(payoffs_y_flattened);
    xfree(payoffs_x_flattened);
    xfree(payoffs_y);
    xfree(payoffs_x);
    xfree(sigma_avg_y_null);
    xfree(sigma_avg_x_null);
    xfree(sigma_y_lifted);
    xfree(sigma_x_lifted);
    xfree(sigma_avg_y_lifted);
    xfree(sigma_avg_x_lifted);
    xfree(sigma_avg_y);
    xfree(sigma_avg_x);

    free_sequence_form(&sequence_form_y);
    free_sequence_form(&sequence_form_x);
    free_sequence_form(&sequence_form_original);

    free_sequences(&sequences_null[0]);
    free_sequences(&sequences_null[1]);
    free_sequences(&sequences_y[1]);
    free_sequences(&sequences_y[0]);
    free_sequences(&sequences_x[1]);
    free_sequences(&sequences_x[0]);
    free_sequences(&sequences_original[1]);
    free_sequences(&sequences_original[0]);

    free_abstraction(&abstraction_null[0]);
    free_abstraction(&abstraction_null[1]);
    free_abstraction(&abstraction_y[1]);
    free_abstraction(&abstraction_y[0]);
    free_abstraction(&abstraction_x[1]);
    free_abstraction(&abstraction_x[0]);
    free_abstraction(&abstraction_original[1]);
    free_abstraction(&abstraction_original[0]);

    free_game(&game);
  }
  return 0;
}
