/*
 * solve_eqm_lp.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "game.h"
#include "lp_eqm.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --x=[output]\n"
	   "  --y=[output]\n"
	   "  --x-lifted=[output]\n"
	   "  --y-lifted=[output]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"x", required_argument, 0, 8},
      {"y", required_argument, 0, 9},
      {"x-lifted", required_argument, 0, 10},
      {"y-lifted", required_argument, 0, 11},
      {NULL},
    };
    int ch;
    game_t game;
    abstraction_t abstraction_abstract[2], abstraction_null[2];
    sequences_t sequences_abstract[2], sequences_null[2];
    sequence_form_t sequence_form_abstract;
    double * sigma_x, * sigma_y;
    double * sigma_x_lifted, * sigma_y_lifted;
    const char * game_path;
    const char * abstraction_abstract_name[2];
    const char * x_path, * y_path, * x_path_lifted, * y_path_lifted;
    FILE * stream;
    double v;

    game_path = NULL;
    abstraction_abstract_name[0] = abstraction_abstract_name[1] = "null";
    x_path = y_path = NULL;
    x_path_lifted = y_path_lifted = NULL;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_abstract_name[0] = optarg; break;
      case 7: abstraction_abstract_name[1] = optarg; break;
      case 8: x_path = optarg; break;
      case 9: y_path = optarg; break;
      case 10: x_path_lifted = optarg; break;
      case 11: y_path_lifted = optarg; break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_abstract_name[0], &game, 0, &abstraction_abstract[0]);
    load_abstraction(abstraction_abstract_name[1], &game, 1, &abstraction_abstract[1]);
    null_abstraction(&game, 0, &abstraction_null[0]);
    null_abstraction(&game, 1, &abstraction_null[1]);

    build_sequences(&abstraction_abstract[0], &sequences_abstract[0]);
    build_sequences(&abstraction_abstract[1], &sequences_abstract[1]);
    build_sequences(&abstraction_null[0], &sequences_null[0]);
    build_sequences(&abstraction_null[1], &sequences_null[1]);
    
    build_sequence_form(&sequences_abstract[0], &sequences_abstract[1], &sequence_form_abstract);
    
    sigma_x             = new_strategy(&sequences_abstract[0]);
    sigma_y             = new_strategy(&sequences_abstract[1]);
    sigma_x_lifted      = new_strategy(&sequences_null[0]);
    sigma_y_lifted      = new_strategy(&sequences_null[1]);
    
    v = lp_eqm(&sequence_form_abstract, sigma_x, sigma_y, NULL, NULL);
    printf("game's value = %g\n", v);
    
    if (x_path) {
      stream = open_stream(x_path, "wt");
      write_strategy(&sequences_abstract[0], sigma_x, stream);
      close_stream(x_path, stream);
    }
    if (y_path) {
      stream = open_stream(y_path, "wt");
      write_strategy(&sequences_abstract[1], sigma_y, stream);
      close_stream(y_path, stream);
    }
    
    lift_strategy(&sequences_abstract[0], &sequences_null[0], sigma_x, sigma_x_lifted);
    lift_strategy(&sequences_abstract[1], &sequences_null[1], sigma_y, sigma_y_lifted);
    if (x_path_lifted) {
      stream = open_stream(x_path_lifted, "wt");
      write_strategy(&sequences_null[0], sigma_x_lifted, stream);
      close_stream(x_path_lifted, stream);
    }
    if (y_path_lifted) {
      stream = open_stream(y_path_lifted, "wt");
      write_strategy(&sequences_null[1], sigma_y_lifted, stream);
      close_stream(y_path_lifted, stream);
    }
    
    xfree(sigma_y_lifted);
    xfree(sigma_x_lifted);
    xfree(sigma_y);
    xfree(sigma_x);
    
    free_sequence_form(&sequence_form_abstract);

    free_sequences(&sequences_null[1]);
    free_sequences(&sequences_null[0]);
    free_sequences(&sequences_abstract[1]);
    free_sequences(&sequences_abstract[0]);
    
    free_abstraction(&abstraction_null[1]);
    free_abstraction(&abstraction_null[0]);
    free_abstraction(&abstraction_abstract[1]);
    free_abstraction(&abstraction_abstract[0]);

    free_game(&game);
  }
  return 0;
}
