/*
 * sequence_form.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _SEQUENCE_FORM_H_
#define _SEQUENCE_FORM_H_

#include "attributes.h"

#include "abstraction.h"
#include "game.h"
#include "util.h"
#include "verify.h"

#define EMPTY_SEQUENCE 0

typedef struct {
  int n;
  int * first_sequence, * first_info_set, * info_set, * parent_sequence, * info_set_order;
  int * parent_info_set, * to_sequence;
  abstraction_t * abstraction;
} sequences_t;

typedef struct {
  sequences_t * sequences[2];
  double * A, max_abs_A;
} sequence_form_t;

void build_sequences(abstraction_t * restrict abstraction, 
		     sequences_t * restrict sequences) NONNULL(1,2);
void free_sequences(sequences_t * restrict sequences) NONNULL(1);

void build_sequence_form(sequences_t * restrict sequences0,
			 sequences_t * restrict sequences1,
			 sequence_form_t * restrict sequence_from) NONNULL(1,2,3);
void free_sequence_form(sequence_form_t * restrict sequence_form) NONNULL(1);

static inline abstraction_t * sequences_abstraction(sequences_t * restrict sequences) NONNULL(1) PURE;
static inline game_t * sequences_game(sequences_t * restrict sequences) NONNULL(1) PURE;
static inline int sequences_n(sequences_t * restrict sequences) NONNULL(1) PURE;
static inline int valid_sequence(sequences_t * restrict sequences, int sequence) NONNULL(1) PURE;
static inline int first_sequence(sequences_t * restrict sequences,
				 int info_set) NONNULL(1) PURE;
static inline int last_sequence(sequences_t * restrict sequences,
				int info_set) NONNULL(1) PURE;
static inline int ith_sequence(sequences_t * restrict sequences,
			       int info_set, int i) NONNULL(1) PURE;
static inline int num_sequences(sequences_t * restrict sequences,
				int info_set) NONNULL(1) PURE;
static inline int parent_sequence(sequences_t * restrict sequences, 
				  int info_set) NONNULL(1) PURE;
static inline int parent_info_set(sequences_t * restrict sequences,
				  int sequence) NONNULL(1) PURE;
static inline int num_info_sets(sequences_t * restrict sequences,
				int sequence) NONNULL(1) PURE;
static inline int ith_info_set(sequences_t * restrict sequences,
			       int sequence, int i) NONNULL(1) PURE;
static inline int info_set_order(sequences_t * restrict sequences, 
				 int i) NONNULL(1) PURE;
static inline int sequence_from_history(sequences_t * restrict sequences,
					int history) NONNULL(1) PURE;

static inline sequences_t * sequence_form_sequences(sequence_form_t * restrict sequence_form,
						    int who) NONNULL(1) PURE;
static inline int sequence_form_index(sequence_form_t * restrict sequence_form,
				      int u, int v) NONNULL(1) PURE;
static inline double sequence_form_A(sequence_form_t * restrict sequence_form,
				     int u, int v) NONNULL(1) PURE;
static inline double sequence_form_max_abs_A(sequence_form_t * restrict sequence_form) NONNULL(1) PURE;

static inline abstraction_t * sequences_abstraction(sequences_t * restrict sequences) {
  assert(sequences);
  return sequences->abstraction;
}

static inline game_t * sequences_game(sequences_t * restrict sequences) {
  assert(sequences);
  return abstraction_game(sequences->abstraction);
}

static inline int sequences_n(sequences_t * restrict sequences) {
  assert(sequences);
  return sequences->n;
}

static inline int valid_sequence(sequences_t * restrict sequences, int sequence) {
  return sequence >= 0 && sequence < sequences_n(sequences);
}

static inline int first_sequence(sequences_t * restrict sequences,
					   int info_set) {
  assert(valid_abstract_info_set(sequences_abstraction(sequences), info_set));
  return sequences->first_sequence[info_set];
}

static inline int last_sequence(sequences_t * restrict sequences,
					  int info_set) {
  assert(valid_abstract_info_set(sequences_abstraction(sequences), info_set));
  return sequences->first_sequence[info_set+1]-1;
}

static inline int ith_sequence(sequences_t * restrict sequences,
			       int info_set, int i) {
  assert(i >= 0 && i < num_sequences(sequences, info_set));
  return sequences->first_sequence[info_set]+i;
}

static inline int num_sequences(sequences_t * restrict sequences,
				int info_set) {
  assert(valid_abstract_info_set(sequences_abstraction(sequences), info_set));
  return sequences->first_sequence[info_set+1]-sequences->first_sequence[info_set];
}

static inline int parent_sequence(sequences_t * restrict sequences, 
				  int info_set) {
  assert(valid_abstract_info_set(sequences_abstraction(sequences), info_set));
  return sequences->parent_sequence[info_set];
}

static inline int num_info_sets(sequences_t * restrict sequences,
				int sequence) {
  assert(valid_sequence(sequences, sequence));
  return sequences->first_info_set[sequence+1] - sequences->first_info_set[sequence];
}

static inline int parent_info_set(sequences_t * restrict sequences,
				  int sequence) {
  assert(valid_sequence(sequences, sequence));
  assert(sequence != EMPTY_SEQUENCE);
  return sequences->parent_info_set[sequence];
}

static inline int ith_info_set(sequences_t * restrict sequences,
			       int sequence, int i) {
  assert(valid_sequence(sequences, sequence));
  return sequences->info_set[sequences->first_info_set[sequence]+i];
}

static inline int info_set_order(sequences_t * restrict sequences, 
				 int i) {
  assert(valid_abstract_info_set(sequences_abstraction(sequences), i));
  return sequences->info_set_order[i];
}

static inline int sequence_from_history(sequences_t * restrict sequences,
					int history) {
  assert(valid_history(sequences_game(sequences), history));
  return sequences->to_sequence[history];
}

static inline sequences_t * sequence_form_sequences(sequence_form_t * restrict sequence_form,
						    int who) {
  assert(sequence_form);
  assert(valid_player(who));
  return sequence_form->sequences[who];
}

static inline int sequence_form_index(sequence_form_t * restrict sequence_form,
				      int u, int v) {
  assert(sequence_form);
  assert(valid_sequence(sequence_form->sequences[0], u));
  assert(valid_sequence(sequence_form->sequences[1], v));
  return sequences_n(sequence_form_sequences(sequence_form, 1))*u+v;
}

static inline double sequence_form_A(sequence_form_t * restrict sequence_form,
				     int u, int v) {
  assert(sequence_form);
  return sequence_form->A[sequence_form_index(sequence_form, u, v)];
}

static inline double sequence_form_max_abs_A(sequence_form_t * restrict sequence_form) {
  assert(sequence_form);
  return sequence_form->max_abs_A;
}

#endif /* _SEQUENCE_FORM_H_ */
