/*
 * sequence_form.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <string.h>

#include "abstraction.h"
#include "game.h"
#include "util.h"
#include "verify.h"

#include "sequence_form.h"

void build_sequences(abstraction_t * restrict abstraction, 
		     sequences_t * restrict sequences) {
  int * q, i, j, k, n, m, id, u, v;
  game_t * game;

  assert(abstraction);
  assert(sequences);

  sequences->abstraction = abstraction;
  game = sequences_game(sequences);

  sequences->first_sequence  = xmalloc((abstract_info_sets(abstraction)+1)*
				       sizeof(int));
  sequences->n = 1;
  for(i=0; i<abstract_info_sets(abstraction); ++i) {
    sequences->first_sequence[i] = sequences->n;
    sequences->n += num_actions_from_info_set(abstraction, i);
  }
  sequences->first_sequence[i] = sequences->n;

  sequences->first_info_set  = xcalloc(sequences->n+1, sizeof(int));
  sequences->info_set        = xmalloc(abstract_info_sets(abstraction)*sizeof(int));
  sequences->parent_sequence = xmalloc(abstract_info_sets(abstraction)*sizeof(int));
  sequences->info_set_order  = xmalloc(abstract_info_sets(abstraction)*sizeof(int));
  sequences->parent_info_set = xmalloc(sequences->n*sizeof(int));
  sequences->to_sequence     = xcalloc(total_histories(game), sizeof(int));
  
  for(i=0; i<abstract_info_sets(abstraction); ++i) {
    for(j=first_sequence(sequences, i); j<=last_sequence(sequences, i); ++j) {
      sequences->parent_info_set[j] = i;
    }
  }

  memset(sequences->parent_sequence, -1, abstract_info_sets(abstraction)*sizeof(int));

  q = xmalloc(total_histories(game)*sizeof(int));
  for(q[j=1,i=n=0]=game_root(game); i!=j;) {
    u = q[i++];
   
    if (chance_history(game, u) || (player_history(game, u) && 
				    who(game, u) != abstraction_who(abstraction))) {
      for(k=first_action(game, u); k<=last_action(game, u); ++k) {
	v = do_action(game, k);
	sequences->to_sequence[v] = sequences->to_sequence[u];
	q[j++] = v;
      }
    } else if (player_history(game, u)) {
      id = abstract_info_set(abstraction, u);

      if (sequences->parent_sequence[id] == -1) {
	sequences->parent_sequence[id] = sequences->to_sequence[u];
	sequences->info_set_order[n++] = id;
      } else {
	verify(sequences->parent_sequence[id] == sequences->to_sequence[u]);
      }

      for(m=0, k=first_action(game, u); k<=last_action(game, u); ++k, ++m) {
	v = do_action(game, k);
	sequences->to_sequence[v] = sequences->first_sequence[id]+m;
	q[j++] = v;
      }
    } else {
      assert(terminal_history(game, u));
    }
  }
  verify(n == abstract_info_sets(abstraction));
  xfree(q);

  for(i=0; i<abstract_info_sets(abstraction); ++i) {
    ++sequences->first_info_set[sequences->parent_sequence[i]];
  }
  for(n=0, i=0; i<sequences->n; ++i) {
    j = sequences->first_info_set[i];
    sequences->first_info_set[i] = n;
    n += j;
  }
  sequences->first_info_set[i] = n;
  for(i=0; i<abstract_info_sets(abstraction); ++i) {
    sequences->info_set[sequences->first_info_set[sequences->parent_sequence[i]]++] = i;
  }
  for(i=sequences->n; i>0; --i) {
    sequences->first_info_set[i] = sequences->first_info_set[i-1];
  }
  sequences->first_info_set[0] = 0;
}

void free_sequences(sequences_t * restrict sequences) {
  assert(sequences);

  xfree(sequences->to_sequence);
  xfree(sequences->parent_info_set);
  xfree(sequences->info_set_order);
  xfree(sequences->parent_sequence);
  xfree(sequences->info_set);
  xfree(sequences->first_info_set);
  xfree(sequences->first_sequence);
}

void build_sequence_form(sequences_t * restrict sequences0,
			 sequences_t * restrict sequences1,
			 sequence_form_t * restrict sequence_form) {
  int i, u, v;
  assert(sequences0);
  assert(sequences1);
  assert(sequence_form);

  assert(sequences_game(sequences0) == sequences_game(sequences1));
  assert(abstraction_who(sequences_abstraction(sequences0)) == 0);
  assert(abstraction_who(sequences_abstraction(sequences1)) == 1);

  sequence_form->sequences[0] = sequences0;
  sequence_form->sequences[1] = sequences1;
  sequence_form->A            = xcalloc(sequences_n(sequences0)*sequences_n(sequences1),
					sizeof(double));
  for(i=first_terminal_history(sequences_game(sequences0)); 
      i<=last_terminal_history(sequences_game(sequences1)); ++i) {
    u = sequences0->to_sequence[i];
    v = sequences1->to_sequence[i];
    sequence_form->A[sequence_form_index(sequence_form, u, v)] += 
      pi_chance(sequences_game(sequences0), i)*
      value(sequences_game(sequences0), i);
  }

  sequence_form->max_abs_A = sequence_form->A[0];
  for(u=0; u<sequences_n(sequences0); ++u) {
    for(v=0; v<sequences_n(sequences1); ++v) {
      if (sequence_form->A[sequence_form_index(sequence_form, u, v)] > sequence_form->max_abs_A) {
	sequence_form->max_abs_A = sequence_form->A[sequence_form_index(sequence_form, u, v)];
      }
      if (-sequence_form->A[sequence_form_index(sequence_form, u, v)] > sequence_form->max_abs_A) {
	sequence_form->max_abs_A = -sequence_form->A[sequence_form_index(sequence_form, u, v)];
      }
    }
  }
}

void free_sequence_form(sequence_form_t * restrict sequence_form) {
  assert(sequence_form);
  xfree(sequence_form->A);
}


