/** 
 * @file rng.h 
 *
 * @brief double precision SIMD oriented Fast Mersenne Twister(rng)
 * pseudorandom number generator based on IEEE 754 format.
 *
 * @author Mutsuo Saito (Hiroshima University)
 * @author Makoto Matsumoto (Hiroshima University)
 *
 * Copyright (C) 2007, 2008 Mutsuo Saito, Makoto Matsumoto and
 * Hiroshima University. All rights reserved.
 *
 * The new BSD License is applied to this software.
 * see LICENSE.txt
 *
 * @note We assume that your system has inttypes.h.  If your system
 * doesn't have inttypes.h, you have to typedef uint32_t and uint64_t,
 * and you have to define PRIu64 and PRIx64 in this file as follows:
 * @verbatim
 typedef unsigned int uint32_t
 typedef unsigned long long uint64_t  
 #define PRIu64 "llu"
 #define PRIx64 "llx"
@endverbatim
 * uint32_t must be exactly 32-bit unsigned integer type (no more, no
 * less), and uint64_t must be exactly 64-bit unsigned integer type.
 * PRIu64 and PRIx64 are used for printf function to print 64-bit
 * unsigned int and 64-bit unsigned int in hexadecimal format.
 */

#ifndef RNG_H
#define RNG_H

#include <stdio.h>
#include <sys/time.h>

#include "util.h"
#include "verify.h"

#define RNG_MEXP 19937
/*-----------------
  BASIC DEFINITIONS
  -----------------*/
/* Mersenne Exponent. The period of the sequence 
 *  is a multiple of 2^RNG_MEXP-1.
 * #define RNG_MEXP 19937 */
/** RNG generator has an internal state array of 128-bit integers,
 * and N is its size. */
#define RNG_N ((RNG_MEXP - 128) / 104 + 1)
/** N32 is the size of internal state array when regarded as an array
 * of 32-bit integers.*/
#define RNG_N32 (RNG_N * 4)
/** N64 is the size of internal state array when regarded as an array
 * of 64-bit integers.*/
#define RNG_N64 (RNG_N * 2)

#if !defined(RNG_BIG_ENDIAN)
#  if defined(__BYTE_ORDER) && defined(__BIG_ENDIAN)
#    if __BYTE_ORDER == __BIG_ENDIAN
#      define RNG_BIG_ENDIAN 1
#    endif
#  elif defined(_BYTE_ORDER) && defined(_BIG_ENDIAN)
#    if _BYTE_ORDER == _BIG_ENDIAN
#      define RNG_BIG_ENDIAN 1
#    endif
#  elif defined(__BYTE_ORDER__) && defined(__BIG_ENDIAN__)
#    if __BYTE_ORDER__ == __BIG_ENDIAN__
#      define RNG_BIG_ENDIAN 1
#    endif
#  elif defined(BYTE_ORDER) && defined(BIG_ENDIAN)
#    if BYTE_ORDER == BIG_ENDIAN
#      define RNG_BIG_ENDIAN 1
#    endif
#  elif defined(__BIG_ENDIAN) || defined(_BIG_ENDIAN) \
    || defined(__BIG_ENDIAN__) || defined(BIG_ENDIAN)
#      define RNG_BIG_ENDIAN 1
#  endif
#endif

#if defined(RNG_BIG_ENDIAN) && defined(__amd64)
#  undef RNG_BIG_ENDIAN
#endif

#if defined(__STDC_VERSION__) && (__STDC_VERSION__ >= 199901L)
#  include <inttypes.h>
#elif defined(_MSC_VER) || defined(__BORLANDC__)
#  if !defined(RNG_UINT32_DEFINED) && !defined(SFMT_UINT32_DEFINED)
typedef unsigned int uint32_t;
typedef unsigned __int64 uint64_t;
#    define UINT64_C(v) (v ## ui64)
#    define RNG_UINT32_DEFINED
#    if !defined(inline)
#      define inline __inline
#    endif
#  endif
#else
#  include <inttypes.h>
#  if !defined(inline)
#    if defined(__GNUC__)
#      define inline __inline__
#    else
#      define inline
#    endif
#  endif
#endif

#ifndef PRIu64
#  if defined(_MSC_VER) || defined(__BORLANDC__)
#    define PRIu64 "I64u"
#    define PRIx64 "I64x"
#  else
#    define PRIu64 "llu"
#    define PRIx64 "llx"
#  endif
#endif

#ifndef UINT64_C
#  define UINT64_C(v) (v ## ULL) 
#endif

/*------------------------------------------
  128-bit SIMD like data type for standard C
  ------------------------------------------*/
#if defined(HAVE_ALTIVEC)
#  if !defined(__APPLE__)
#    include <altivec.h>
#  endif
/** 128-bit data structure */
union W128_T {
    vector unsigned int s;
    uint64_t u[2];
    uint32_t u32[4];
    double d[2];
};

#elif defined(HAVE_SSE2)
#  include <emmintrin.h>

/** 128-bit data structure */
union W128_T {
    __m128i si;
    __m128d sd;
    uint64_t u[2];
    uint32_t u32[4];
    double d[2];
};
#else  /* standard C */
/** 128-bit data structure */
union W128_T {
    uint64_t u[2];
    uint32_t u32[4];
    double d[2];
};
#endif

/** 128-bit data type */
typedef union W128_T w128_t;

/** the 128-bit internal state array */
struct RNG_T {
    w128_t status[RNG_N + 1];
    int idx;
};
typedef struct RNG_T rng_t;

/** rng internal state vector */
extern rng_t rng_global_data;
/** rng mexp for check */
extern const int rng_global_mexp;

void rng_gen_rand_all(rng_t * restrict rng);
void rng_fill_array_open_close(rng_t * restrict rng, double array[restrict], int size);
void rng_fill_array_close_open(rng_t * restrict rng, double array[restrict], int size);
void rng_fill_array_open_open(rng_t * restrict rng, double array[restrict], int size);
void rng_fill_array_close1_open2(rng_t * restrict rng, double array[restrict], int size);
void rng_chk_init_gen_rand(rng_t * restrict rng, uint32_t seed, int mexp);
void rng_chk_init_by_array(rng_t * restrict rng, uint32_t init_key[restrict],
			     int key_length, int mexp);
const char *rng_get_idstring(void);
int rng_get_min_array_size(void);

#define INLINE static inline
#define RNG_PRE_INLINE INLINE
#define RNG_PST_INLINE __attribute__((always_inline))

RNG_PRE_INLINE rng_t * rng_global() RNG_PST_INLINE;
RNG_PRE_INLINE uint32_t rng_genrand_uint32(rng_t * restrict rng) RNG_PST_INLINE;
RNG_PRE_INLINE double rng_genrand_close1_open2(rng_t * restrict rng)
    RNG_PST_INLINE;
RNG_PRE_INLINE double rng_genrand_close_open(rng_t * restrict rng)
    RNG_PST_INLINE;
RNG_PRE_INLINE double rng_genrand_open_close(rng_t * restrict rng)
    RNG_PST_INLINE;
RNG_PRE_INLINE double rng_genrand_open_open(rng_t * restrict rng)
    RNG_PST_INLINE;
RNG_PRE_INLINE uint32_t rng_gv_genrand_uint32(void) RNG_PST_INLINE;
RNG_PRE_INLINE double rng_gv_genrand_close1_open2(void) RNG_PST_INLINE;
RNG_PRE_INLINE double rng_gv_genrand_close_open(void) RNG_PST_INLINE;
RNG_PRE_INLINE double rng_gv_genrand_open_close(void) RNG_PST_INLINE;
RNG_PRE_INLINE double rng_gv_genrand_open_open(void) RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_fill_array_open_close(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_fill_array_close_open(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_fill_array_open_open(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_fill_array_close1_open2(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_init_gen_rand(uint32_t seed) RNG_PST_INLINE;
RNG_PRE_INLINE void rng_gv_init_by_array(uint32_t init_key[restrict],
					     int key_length) RNG_PST_INLINE;
RNG_PRE_INLINE void rng_init_gen_rand(rng_t *rng, uint32_t seed)
    RNG_PST_INLINE;
RNG_PRE_INLINE void rng_init_by_array(rng_t *rng, uint32_t init_key[restrict],
					  int key_length) RNG_PST_INLINE;

RNG_PRE_INLINE uint32_t rng_seed_time() RNG_PST_INLINE;
RNG_PRE_INLINE rng_t * new_rng() MALLOC RNG_PST_INLINE;
RNG_PRE_INLINE void free_rng(rng_t * restrict rng) RNG_PST_INLINE;

INLINE rng_t * rng_global() {
  return &rng_global_data;
}

INLINE uint32_t rng_seed_time() {
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_usec;
}

INLINE rng_t * new_rng() {
  return xmalloc(sizeof(rng_t));
}

INLINE void free_rng(rng_t * restrict rng) {
  assert(rng);
  xfree(rng);
}

/**
 * This function generates and returns unsigned 32-bit integer.
 * This is slower than SFMT, only for convenience usage.
 * rng_init_gen_rand() or rng_init_by_array() must be called
 * before this function.
 * @param rng rng internal state date
 * @return double precision floating point pseudorandom number
 */
INLINE uint32_t rng_genrand_uint32(rng_t * restrict rng) {
    uint32_t r;
    uint64_t *psfmt64 = &rng->status[0].u[0];

    if (rng->idx >= RNG_N64) {
	rng_gen_rand_all(rng);
	rng->idx = 0;
    }
    r = psfmt64[rng->idx++] & 0xffffffffU;
    return r;
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [1, 2).  This is
 * the primitive and faster than generating numbers in other ranges.
 * rng_init_gen_rand() or rng_init_by_array() must be called
 * before this function.
 * @param rng rng internal state date
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_genrand_close1_open2(rng_t * restrict rng) {
    double r;
    double *psfmt64 = &rng->status[0].d[0];

    if (rng->idx >= RNG_N64) {
	rng_gen_rand_all(rng);
	rng->idx = 0;
    }
    r = psfmt64[rng->idx++];
    return r;
}

/**
 * This function generates and returns unsigned 32-bit integer.
 * This is slower than SFMT, only for convenience usage.
 * rng_gv_init_gen_rand() or rng_gv_init_by_array() must be called
 * before this function.  This function uses \b global variables.
 * @return double precision floating point pseudorandom number
 */
INLINE uint32_t rng_gv_genrand_uint32(void) {
    return rng_genrand_uint32(&rng_global_data);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [1, 2).
 * rng_gv_init_gen_rand() or rng_gv_init_by_array() must be called
 * before this function. This function uses \b global variables.
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_gv_genrand_close1_open2(void) {
    return rng_genrand_close1_open2(&rng_global_data);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [0, 1).
 * rng_init_gen_rand() or rng_init_by_array() must be called
 * before this function.
 * @param rng rng internal state date
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_genrand_close_open(rng_t * restrict rng) {
    return rng_genrand_close1_open2(rng) - 1.0;
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [0, 1).
 * rng_gv_init_gen_rand() or rng_gv_init_by_array() must be called
 * before this function. This function uses \b global variables.
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_gv_genrand_close_open(void) {
    return rng_gv_genrand_close1_open2() - 1.0;
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1].
 * rng_init_gen_rand() or rng_init_by_array() must be called
 * before this function. 
 * @param rng rng internal state date
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_genrand_open_close(rng_t * restrict rng) {
    return 2.0 - rng_genrand_close1_open2(rng);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1].
 * rng_gv_init_gen_rand() or rng_gv_init_by_array() must be called
 * before this function. This function uses \b global variables.
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_gv_genrand_open_close(void) {
    return 2.0 - rng_gv_genrand_close1_open2();
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1).
 * rng_init_gen_rand() or rng_init_by_array() must be called
 * before this function.
 * @param rng rng internal state date
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_genrand_open_open(rng_t * restrict rng) {
    double *rng64 = &rng->status[0].d[0];
    union {
	double d;
	uint64_t u;
    } r;

    if (rng->idx >= RNG_N64) {
	rng_gen_rand_all(rng);
	rng->idx = 0;
    }
    r.d = rng64[rng->idx++];
    r.u |= 1;
    return r.d - 1.0;
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1).
 * rng_gv_init_gen_rand() or rng_gv_init_by_array() must be called
 * before this function. This function uses \b global variables.
 * @return double precision floating point pseudorandom number
 */
INLINE double rng_gv_genrand_open_open(void) {
    return rng_genrand_open_open(&rng_global_data);
}

/**
 * This function generates double precision floating point
 * pseudorandom numbers which distribute in the range [1, 2) to the
 * specified array[] by one call. This function is the same as
 * rng_fill_array_close1_open2() except that this function uses
 * \b global variables.
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_fill_array_close1_open2()
 */
INLINE void rng_gv_fill_array_close1_open2(double array[restrict], int size) {
    rng_fill_array_close1_open2(&rng_global_data, array, size);
}

/**
 * This function generates double precision floating point
 * pseudorandom numbers which distribute in the range (0, 1] to the
 * specified array[] by one call. This function is the same as
 * rng_gv_fill_array_close1_open2() except the distribution range.
 * This function uses \b global variables.
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_fill_array_close1_open2() and \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void rng_gv_fill_array_open_close(double array[restrict], int size) {
    rng_fill_array_open_close(&rng_global_data, array, size);
}

/**
 * This function generates double precision floating point
 * pseudorandom numbers which distribute in the range [0, 1) to the
 * specified array[] by one call. This function is the same as
 * rng_gv_fill_array_close1_open2() except the distribution range.
 * This function uses \b global variables.
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_fill_array_close1_open2() \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void rng_gv_fill_array_close_open(double array[restrict], int size) {
    rng_fill_array_close_open(&rng_global_data, array, size);
}

/**
 * This function generates double precision floating point
 * pseudorandom numbers which distribute in the range (0, 1) to the
 * specified array[] by one call. This function is the same as
 * rng_gv_fill_array_close1_open2() except the distribution range.
 * This function uses \b global variables.
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_fill_array_close1_open2() \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void rng_gv_fill_array_open_open(double array[restrict], int size) {
    rng_fill_array_open_open(&rng_global_data, array, size);
}

/**
 * This function initializes the internal state array with a 32-bit
 * integer seed.
 * @param rng rng state vector.
 * @param seed a 32-bit integer used as the seed.
 */
INLINE void rng_init_gen_rand(rng_t * restrict rng, uint32_t seed) {
    rng_chk_init_gen_rand(rng, seed, RNG_MEXP);
}

/**
 * This function initializes the internal state array with a 32-bit
 * integer seed. This function uses \b global variables.
 * @param seed a 32-bit integer used as the seed.
 * see also \sa rng_init_gen_rand()
 */
INLINE void rng_gv_init_gen_rand(uint32_t seed) {
    rng_init_gen_rand(&rng_global_data, seed);
}

/**
 * This function initializes the internal state array,
 * with an array of 32-bit integers used as the seeds.
 * @param rng rng state vector
 * @param init_key the array of 32-bit integers, used as a seed.
 * @param key_length the length of init_key.
 */
INLINE void rng_init_by_array(rng_t *rng, uint32_t init_key[restrict],
				       int key_length) {
    rng_chk_init_by_array(rng, init_key, key_length, RNG_MEXP);
}

/**
 * This function initializes the internal state array,
 * with an array of 32-bit integers used as the seeds.
 * This function uses \b global variables.
 * @param init_key the array of 32-bit integers, used as a seed.
 * @param key_length the length of init_key.
 * see also \sa rng_init_by_array()
 */
INLINE void rng_gv_init_by_array(uint32_t init_key[restrict], int key_length) {
    rng_init_by_array(&rng_global_data, init_key, key_length);
}

#if !defined(RNG_DO_NOT_USE_OLD_NAMES)
RNG_PRE_INLINE const char *get_idstring(void) RNG_PST_INLINE;
RNG_PRE_INLINE int get_min_array_size(void) RNG_PST_INLINE;
RNG_PRE_INLINE void init_gen_rand(uint32_t seed) RNG_PST_INLINE;
RNG_PRE_INLINE void init_by_array(uint32_t init_key[restrict], int key_length)
    RNG_PST_INLINE;
RNG_PRE_INLINE double genrand_close1_open2(void) RNG_PST_INLINE;
RNG_PRE_INLINE double genrand_close_open(void) RNG_PST_INLINE;
RNG_PRE_INLINE double genrand_open_close(void) RNG_PST_INLINE;
RNG_PRE_INLINE double genrand_open_open(void) RNG_PST_INLINE;
RNG_PRE_INLINE void fill_array_open_close(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void fill_array_close_open(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void fill_array_open_open(double array[restrict], int size)
    RNG_PST_INLINE;
RNG_PRE_INLINE void fill_array_close1_open2(double array[restrict], int size)
    RNG_PST_INLINE;

/**
 * This function is just the same as rng_get_idstring().
 * @return id string.
 * see also \sa rng_get_idstring()
 */
INLINE const char *get_idstring(void) {
    return rng_get_idstring();
}

/**
 * This function is just the same as rng_get_min_array_size().
 * @return minimum size of array used for fill_array functions.
 * see also \sa rng_get_min_array_size()
 */
INLINE int get_min_array_size(void) {
    return rng_get_min_array_size();
}

/**
 * This function is just the same as rng_gv_init_gen_rand().
 * @param seed a 32-bit integer used as the seed.
 * see also \sa rng_gv_init_gen_rand(), \sa rng_init_gen_rand().
 */
INLINE void init_gen_rand(uint32_t seed) {
    rng_gv_init_gen_rand(seed);
}

/**
 * This function is just the same as rng_gv_init_by_array().
 * @param init_key the array of 32-bit integers, used as a seed.
 * @param key_length the length of init_key.
 * see also \sa rng_gv_init_by_array(), \sa rng_init_by_array().
 */
INLINE void init_by_array(uint32_t init_key[restrict], int key_length) {
    rng_gv_init_by_array(init_key, key_length);
}

/**
 * This function is just the same as rng_gv_genrand_close1_open2().
 * @return double precision floating point number.
 * see also \sa rng_genrand_close1_open2() \sa
 * rng_gv_genrand_close1_open2()
 */
INLINE double genrand_close1_open2(void) {
    return rng_gv_genrand_close1_open2();
}

/**
 * This function is just the same as rng_gv_genrand_close_open().
 * @return double precision floating point number.
 * see also \sa rng_genrand_close_open() \sa
 * rng_gv_genrand_close_open()
 */
INLINE double genrand_close_open(void) {
    return rng_gv_genrand_close_open();
}

/**
 * This function is just the same as rng_gv_genrand_open_close().
 * @return double precision floating point number.
 * see also \sa rng_genrand_open_close() \sa
 * rng_gv_genrand_open_close()
 */
INLINE double genrand_open_close(void) {
    return rng_gv_genrand_open_close();
}

/**
 * This function is just the same as rng_gv_genrand_open_open().
 * @return double precision floating point number.
 * see also \sa rng_genrand_open_open() \sa
 * rng_gv_genrand_open_open()
 */
INLINE double genrand_open_open(void) {
    return rng_gv_genrand_open_open();
}

/**
 * This function is juset the same as rng_gv_fill_array_open_close().
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_gv_fill_array_open_close(), \sa
 * rng_fill_array_close1_open2(), \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void fill_array_open_close(double array[restrict], int size) {
    rng_gv_fill_array_open_close(array, size);
}

/**
 * This function is juset the same as rng_gv_fill_array_close_open().
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_gv_fill_array_close_open(), \sa
 * rng_fill_array_close1_open2(), \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void fill_array_close_open(double array[restrict], int size) {
    rng_gv_fill_array_close_open(array, size);
}

/**
 * This function is juset the same as rng_gv_fill_array_open_open().
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_gv_fill_array_open_open(), \sa
 * rng_fill_array_close1_open2(), \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void fill_array_open_open(double array[restrict], int size) {
    rng_gv_fill_array_open_open(array, size);
}

/**
 * This function is juset the same as rng_gv_fill_array_close1_open2().
 * @param array an array where pseudorandom numbers are filled
 * by this function.
 * @param size the number of pseudorandom numbers to be generated.
 * see also \sa rng_fill_array_close1_open2(), \sa
 * rng_gv_fill_array_close1_open2()
 */
INLINE void fill_array_close1_open2(double array[restrict], int size) {
    rng_gv_fill_array_close1_open2(array, size);
}
#endif /* RNG_DO_NOT_USE_OLD_NAMES */

#endif /* RNG_H */
