/*
 * player.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _PLAYER_H_
#define _PLAYER_H_

#include "sequence_form.h"

/* create a player */
typedef void * (*create_player_func)(sequence_form_t * restrict sequence_form, int who, int T, 
				     const char * restrict params);

/* free a player */
typedef void (*free_player_func)(void * restrict player);

/* get the strategy */
typedef const double * (*get_strategy_func)(void * restrict player);

/* update the strategy - full information */
typedef void (*update_full_strategy_func)(void * restrict player, double * restrict payoffs);

/* update the strategy - bandit setting */
typedef void (*update_bandit_strategy_func)(void * restrict player, int terminal);

typedef struct {
  create_player_func create_player;
  free_player_func free_player;
  get_strategy_func get_strategy;
  update_full_strategy_func update_full_strategy;
  update_bandit_strategy_func update_bandit_strategy;
  void * data;
} player_t;

void create_player(const char * restrict player_name, sequence_form_t * restrict sequence_form,
		   int who, int T, const char * params, player_t * restrict player);

static inline void free_player(player_t * restrict player) NONNULL(1);
static inline const double * get_strategy(player_t * restrict player) NONNULL(1) PURE;
static inline void update_full_strategy(player_t * restrict player, 
					double * restrict payoffs) NONNULL(1,2);

static inline void free_player(player_t * restrict player) {
  assert(player);
  player->free_player(player->data);
}

static inline const double * get_strategy(player_t * restrict player) {
  assert(player);
  return player->get_strategy(player->data);
}

static inline void update_full_strategy(player_t * restrict player, 
					double * restrict payoffs) {
  assert(payoffs);
  player->update_full_strategy(player->data, payoffs);
}

#endif /* _PLAYER_H_ */
