/*
 * lp_eqm.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <glpk.h>

#include "abstraction.h"
#include "game.h"
#include "sequence_form.h"
#include "util.h"
#include "verify.h"

#include "lp_eqm.h"

double lp_eqm(sequence_form_t * restrict sequence_form,
	      double * restrict sigma1, double * restrict sigma2,
	      double * restrict ev2, double * restrict ev1) {
  abstraction_t * restrict abstraction[2];
  sequences_t * restrict sequences[2];
  glp_prob * lp;
  glp_smcp parm;
  double v, * element;
  int i, j, n, m, * row, * col;

  sequences[0]   = sequence_form_sequences(sequence_form, 0);
  sequences[1]   = sequence_form_sequences(sequence_form, 1);
  abstraction[0] = sequences_abstraction(sequences[0]);
  abstraction[1] = sequences_abstraction(sequences[1]);
  
  n = lp_E_size(sequences[0])+lp_E_size(sequences[1])+lp_A_size(sequence_form);

  element = xmalloc(sizeof(double)*n);
  row = xmalloc(sizeof(int)*n);
  col = xmalloc(sizeof(int)*n);
  
  m = 0;
  /* F^T */
  m += lp_E(sequences[1], 1, 1, col, row, element);
  /* -A^T */
  m += lp_A(sequence_form, 2+abstract_info_sets(abstraction[1]), 1, 1, col+m, row+m, element+m);
  /* E */
  m += lp_E(sequences[0], 1+sequences_n(sequences[1]), 2+abstract_info_sets(abstraction[1]), row+m, col+m, element+m);
  verify(m == n);

  lp = glp_create_prob();
  glp_add_rows(lp, sequences_n(sequences[1])+1+abstract_info_sets(abstraction[0]));
  glp_add_cols(lp, sequences_n(sequences[0])+1+abstract_info_sets(abstraction[1]));
  for(i=1; i<=sequences_n(sequences[1]); ++i) {
    glp_set_row_bnds(lp, i, GLP_UP, 0., 0.);
  }
  glp_set_row_bnds(lp, i++, GLP_FX, 1., 1.);
  for(j=0; j<abstract_info_sets(abstraction[0]); ++i, ++j) {
    glp_set_row_bnds(lp, i, GLP_FX, 0., 0.);
  }
  for(i=1; i<=1+abstract_info_sets(abstraction[1]); ++i) {
    glp_set_col_bnds(lp, i, GLP_FR, 0., 0.);
  }
  for(j=0; j<sequences_n(sequences[0]); ++i, ++j) {
    glp_set_col_bnds(lp, i, GLP_LO, 0., 0.);
  }

  glp_load_matrix(lp, n, row-1, col-1, element-1);

  glp_set_obj_dir(lp, GLP_MAX);
  glp_set_obj_coef(lp, 1, 1.);

  glp_init_smcp(&parm);
  parm.msg_lev = GLP_MSG_OFF;
  glp_simplex(lp, &parm);
  
  v = glp_get_obj_val(lp);
  if (sigma1) {
    for(i=2+abstract_info_sets(abstraction[1]), j=0; j<sequences_n(sequences[0]); ++i, ++j) {
      sigma1[j] = glp_get_col_prim(lp, i);
    }
  }
  if (sigma2) {
    for(i=1; i<=sequences_n(sequences[1]); ++i) {
      sigma2[i-1] = glp_get_row_dual(lp, i);
    }
  }
  if (ev2) {
    for(i=1; i<=1+abstract_info_sets(abstraction[1]); ++i) {
      ev2[i-1] = glp_get_col_prim(lp, i);
    }
  }
  if (ev1) {
    for(j=1+sequences_n(sequences[1]), i=0; i<abstract_info_sets(abstraction[0]); ++i, ++j) {
      ev1[i] = glp_get_row_dual(lp, j);
    }
  }

  glp_delete_prob(lp);

  xfree(col);
  xfree(row);
  xfree(element);

  return v;
}

int lp_E(sequences_t * restrict sequences, int row_offset, int column_offset, int * restrict row,
	 int * restrict col, double * restrict element) {
  int i, j, m;
  
  m = 0;

  element[m] = 1.;
  row[m]     = row_offset;
  col[m]     = column_offset; ++m;
  for(i=0; i<abstract_info_sets(sequences_abstraction(sequences)); ++i) {
    element[m] = -1.;
    row[m]     = 1+row_offset+i;
    col[m]     = column_offset+parent_sequence(sequences, i); ++m;
    for(j=first_sequence(sequences, i); j<=last_sequence(sequences, i); ++j) {
      element[m] = 1.;
      row[m]     = 1+row_offset+i;
      col[m]     = column_offset+j; ++m;
    }
  }

  return m;
}

int lp_A(sequence_form_t * restrict sequence_form, int row_offset, int column_offset,
	 int negate, int * restrict row, int * restrict col, double * restrict element) {
  sequences_t * restrict sequences[2];
  int i, j, m;

  assert(sequence_form);
  assert(row);
  assert(col);
  assert(element);

  sequences[0] = sequence_form_sequences(sequence_form, 0);
  sequences[1] = sequence_form_sequences(sequence_form, 1);

  m = 0;
  for(i=0; i<sequences_n(sequences[0]); ++i) {
    for(j=0; j<sequences_n(sequences[1]); ++j) {
      element[m] = negate?-sequence_form_A(sequence_form, i, j):sequence_form_A(sequence_form, i, j);
      row[m]     = row_offset+i;
      col[m]     = column_offset+j; ++m;
    }
  }

  return m;
}
