/*
 * game_info.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "game.h"
#include "sequence_form.h"
#include "verify.h"
#include "util.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arguments:\n"
	   "  -g --game=[game file]\n"
	   "  -c --chance\n"
	   "  -p --player\n"
	   "  -z --terminal\n"
	   "  -a --actions\n"
	   "  -i --info-sets\n"
	   "  -x --abstraction1=[abstraction]\n"
	   "  -y --abstraction2=[abstraction]\n"
	   "  -s --sequences\n"
	   "  -A --payoff-matrix\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 'g'},
      {"chance", no_argument, 0, 'c'},
      {"player", no_argument, 0, 'p'},
      {"terminal", no_argument, 0, 'z'},
      {"actions", no_argument, 0, 'a'},
      {"info-sets", no_argument, 0, 'i'},
      {"abstraction1", required_argument, 0, 'x'},
      {"abstraction2", required_argument, 0, 'y'},
      {"sequences", no_argument, 0, 's'},
      {"payoff-matrix", no_argument, 0, 'A'},
      {NULL},
    };
    const char * game_path, * abstraction1, * abstraction2;
    FILE * stream;
    game_t game;
    abstraction_t abstraction[2];
    sequences_t sequences[2];
    sequence_form_t sequence_form;
    int ch, chance, player, terminal, actions, info_sets, seq, A, i, j, k, u;
    
    game_path    = NULL;
    chance       = 0;
    player       = 0;
    terminal     = 0;
    actions      = 0;
    info_sets    = 0;
    seq          = 0;
    A            = 0;
    abstraction1 = "null";
    abstraction2 = "null";

    while((ch = getopt_long(argc, argv, "g:cpzaisx:y:A", opts, NULL)) != -1) {
      switch(ch) {
      case 'g': game_path = optarg; break;
      case 'c': chance = 1; break;
      case 'p': player = 1; break;
      case 'z': terminal = 1; break;
      case 'a': actions = 1; break;
      case 'i': info_sets = 1; break;
      case 'x': abstraction1 = optarg; break;
      case 'y': abstraction2 = optarg; break;
      case 's': seq = 1; break;
      case 'A': A = 1; break;
      default: verify("unknown argument");
      }
    }

    verify(game_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction1, &game, 0, &abstraction[0]);
    build_sequences(&abstraction[0], &sequences[0]);
    load_abstraction(abstraction2, &game, 1, &abstraction[1]);
    build_sequences(&abstraction[1], &sequences[1]);
    build_sequence_form(&sequences[0], &sequences[1], &sequence_form);

    printf("game name:     %s\n"
	   "histories:     %d\n"
	   "  chance:      %d\n"
	   "  player:      %d (%d %d)\n"
	   "  terminal:    %d\n"
	   "utility:       %d %d %d\n"
	   "info sets:     %d (%d %d)\n"
	   "actions:       %d (%d %d %d)\n"
	   "root:          %d\n"
	   "abstraction x: %s\n"
	   "  sequences:   %d\n"
	   "  info sets:   %d\n"
	   "abstraction y: %s\n"
	   "  sequences:   %d\n"
	   "  info sets:   %d\n"
	   , game_name(&game), total_histories(&game), total_chance_histories(&game),
	   total_player_histories(&game, BOTH), total_player_histories(&game, 0),
	   total_player_histories(&game, 1), total_terminal_histories(&game),
	   min_u(&game), max_u(&game), delta_u(&game),
	   total_info_sets(&game, BOTH), total_info_sets(&game, 0), total_info_sets(&game, 1),
	   total_actions(&game), total_chance_actions(&game),
	   total_player_actions(&game, 0), total_player_actions(&game, 1),
	   game_root(&game), 
	   abstraction1, sequences_n(&sequences[0]), abstract_info_sets(&abstraction[0]),
	   abstraction2, sequences_n(&sequences[1]), abstract_info_sets(&abstraction[1]));

    if (chance) {
      for(i=first_chance_history(&game); i<=last_chance_history(&game); ++i) {
	printf("%d '%s' parent: '%s' actions: %d%s\n", i, 
	       history_name(&game, i),
	       parent(&game, i)==-1?"null":history_name(&game, parent(&game, i)),
	       num_actions(&game, i), game_root(&game)==i?" (root)":"");
	printf("  player: c pi_chance: %g sequence: %d, %d\n", pi_chance(&game, i),
	       sequence_from_history(&sequences[0], i),
	       sequence_from_history(&sequences[1], i));
	if (actions) {
	  for(j=first_action(&game, i); j<=last_action(&game, i); ++j) {
	    u = do_action(&game, j);
	    printf("  '%s' -> '%s' pi: %g\n", action_name(&game, j), history_name(&game, u),
		   sigma_chance(&game, j));
	  }
	}
      }
    }

    if (player) {
      for(i=first_player_history(&game); i<=last_player_history(&game); ++i) {
	printf("%d '%s' parent: '%s' actions: %d%s\n", i, 
	       history_name(&game, i), 
	       parent(&game, i)==-1?"null":history_name(&game, parent(&game, i)), 
	       num_actions(&game, i), game_root(&game)==i?" (root)":"");
	printf("  player: %d info set: %d abstract info set: %d pi_chance: %g sequence: %d, %d\n",
	       who(&game, i), info_set(&game, i), 
	       abstract_info_set(&abstraction[who(&game, i)], i), pi_chance(&game, i),
	       sequence_from_history(&sequences[0], i),
	       sequence_from_history(&sequences[1], i));
	if (actions) {
	  for(j=first_action(&game, i); j<=last_action(&game, i); ++j) {
	    u = do_action(&game, j);
	    printf("  '%s' -> '%s' sequence: %d\n", action_name(&game, j), history_name(&game, u),
		   first_sequence(&sequences[who(&game, i)],
					    abstract_info_set(&abstraction[who(&game, i)], i))+
		   j-first_action(&game, i));
	  }
	}
      }
    }

    if (terminal) {
      for(i=first_terminal_history(&game); i<=last_terminal_history(&game); ++i) {
	printf("%d '%s' parent: '%s' pi_chance: %g sequence: %d, %d value: %d\n", i, 
	       history_name(&game, i),
	       parent(&game, i)==-1?"null":history_name(&game, parent(&game, i)), 
	       pi_chance(&game, i), sequence_from_history(&sequences[0], i),
	       sequence_from_history(&sequences[1], i), value(&game, i));
      }
    }

    if (info_sets) {
      for(i=0; i<2; ++i) {
	for(j=0; j<abstract_info_sets(&abstraction[i]); ++j) {
	  printf("info set %d, %d histories (%d):", i, j, total_histories_in_info_set(&abstraction[i], j));
	  for(k=0; k<total_histories_in_info_set(&abstraction[i], j); ++k) {
	    printf(" '%s'", history_name(&game, ith_history_in_info_set(&abstraction[i], j, k)));
	  }
	  printf("\n");
	  printf("  info sets (%d):", num_members(&abstraction[i], j));
	  for(k=0; k<num_members(&abstraction[i], j); ++k) {
	    printf(" %d", ith_member(abstraction, j, k));
	  }
	  printf("\n");
	  if (seq) {
	    printf("  parent: %d first_sequence: %d last_sequence: %d n: %d\n",
		   parent_sequence(&sequences[i], j), 
		   first_sequence(&sequences[i], j),
		   last_sequence(&sequences[i], j),
		   num_sequences(&sequences[i], j));
	  }
	}
      }
    }

    if (seq) {
      for(i=0; i<2; ++i) {
	printf("topological order %d:", i);
	for(j=0; j<abstract_info_sets(&abstraction[i]); ++j) {
	  printf(" %d", info_set_order(&sequences[i], j));
	}
	printf("\n");
      }
      for(i=0; i<2; ++i) {
	for(j=0; j<sequences_n(&sequences[i]); ++j) {
	  printf("sequence %d, %d: n: %d:", i, j, 
		 num_info_sets(&sequences[i], j));
	  for(k=0; k<num_info_sets(&sequences[i], j); ++k) {
	    printf(" %d", ith_info_set(&sequences[i], j, k));
	  }
	  printf("\n");
	}
      }
    }

    if (A) {
      printf("A = [...\n");
      for(i=0; i<sequences_n(&sequences[0]); ++i) {
	for(j=0; j<sequences_n(&sequences[1]); ++j) {
	  printf(j?" %g":"%g", sequence_form_A(&sequence_form, i, j));
	}
	printf("; ...\n");
      }
      printf("]\n");
    }

    free_sequence_form(&sequence_form);
    for(i=0; i<2; ++i) {
      free_sequences(&sequences[i]);
      free_abstraction(&abstraction[i]);
    }
    free_game(&game);
  }

  return 0;
}
