/*
 * game.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _GAME_H_
#define _GAME_H_

#include <stdio.h>

#include "attributes.h"
#include "verify.h"

#define BOTH  -1

typedef struct {
  char * game_name;
  int chance, player, terminal, root, actions, chance_actions;
  char ** name;
  int * parent;
  int * first_action, * num_actions, * action;
  int * action_history;
  char ** action_name;
  double * sigma_chance, * pi_chance;
  int * who, * info_set, * chance_depth;
  int * value;
  int min_u, max_u, max_chance_depth;
  int player_histories[2], player_actions[2], info_sets[2];
} game_t;

void read_game(FILE * restrict stream, game_t * restrict game) NONNULL(1,2);
void free_game(game_t * restrict game) NONNULL(1);

static inline int valid_player(int who) CONST;
static inline int opponent(int who) CONST;
static inline int value_multipler(int who) CONST;

static inline int valid_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int chance_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int player_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int internal_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int terminal_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int first_chance_history(game_t * restrict game) NONNULL(1) PURE;
static inline int last_chance_history(game_t * restrict game) NONNULL(1) PURE;
static inline int first_player_history(game_t * restrict game) NONNULL(1) PURE;
static inline int last_player_history(game_t * restrict game) NONNULL(1) PURE;
static inline int first_internal_history(game_t * restrict game) NONNULL(1) PURE;
static inline int last_internal_history(game_t * restrict game) NONNULL(1) PURE;
static inline int first_terminal_history(game_t * restrict game) NONNULL(1) PURE;
static inline int last_terminal_history(game_t * restrict game) NONNULL(1) PURE;
static inline int total_chance_histories(game_t * restrict game) NONNULL(1) PURE;
static inline int total_player_histories(game_t * restrict game, int who) NONNULL(1) PURE;
static inline int total_internal_histories(game_t * restrict game) NONNULL(1) PURE;
static inline int total_terminal_histories(game_t * restrict game) NONNULL(1) PURE;
static inline int total_histories(game_t * restrict game) NONNULL(1) PURE;
static inline int total_actions(game_t * restrict game) NONNULL(1) PURE;
static inline int total_chance_actions(game_t * restrict game) NONNULL(1) PURE;
static inline int total_player_actions(game_t * restrict game, int who) NONNULL(1) PURE;
static inline int valid_info_set(game_t * restrict game, int who, int info_set) NONNULL(1) PURE;
static inline int total_info_sets(game_t * restrict game, int who) NONNULL(1) PURE;
static inline int valid_action(game_t * restrict game, int action) NONNULL(1) PURE;
static inline int chance_action(game_t * restrict game, int action) NONNULL(1) PURE;
static inline int player_action(game_t * restrict game, int action) NONNULL(1) PURE;

static inline const char * game_name(game_t * restrict game) NONNULL(1) PURE;
static inline int game_root(game_t * restrict game) NONNULL(1) PURE;
static inline const char * history_name(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int parent(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int num_actions(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int first_action(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int last_action(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int ith_action(game_t * restrict game, int history, int i) NONNULL(1) PURE;
static inline int do_action(game_t * restrict game, int action) NONNULL(1) PURE;
static inline int action_to_history(game_t * restrict game, int action) NONNULL(1) PURE;
static inline const char * action_name(game_t * restrict game, int action) NONNULL(1) PURE;
static inline int chance_depth(game_t * restrict game, int history) NONNULL(1) PURE;
static inline double sigma_chance(game_t * restrict game, int action) NONNULL(1) PURE;
static inline const double * sigma_chance_history(game_t * restrict game, int history) NONNULL(1) PURE;
static inline double pi_chance(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int who(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int info_set(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int value(game_t * restrict game, int history) NONNULL(1) PURE;
static inline int value_for(game_t * restrict game, int history, int who) NONNULL(1) PURE;
static inline int min_u(game_t * restrict game) NONNULL(1) PURE;
static inline int max_u(game_t * restrict game) NONNULL(1) PURE;
static inline int delta_u(game_t * restrict game) NONNULL(1) PURE;
static inline int max_chance_depth(game_t * restrict game) NONNULL(1) PURE;

static inline int valid_player(int who) {
  return who == 0 || who == 1;
}

static inline int opponent(int who) {
  assert(valid_player(who));
  return 1-who;
}

static inline int value_multipler(int who) {
  assert(valid_player(who));
  return 1-2*who;
}

static inline int valid_history(game_t * restrict game, int history) {
  assert(game);
  return history >= 0 && history < total_histories(game);
}

static inline int chance_history(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return history < game->chance;
}

static inline int player_history(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return history >= game->chance && history < game->chance+game->player;
}

static inline int internal_history(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return history < game->chance+game->player;
}

static inline int terminal_history(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return history >= game->chance+game->player;
}

static inline int first_chance_history(game_t * restrict game) {
  assert(game);
  return 0;
}

static inline int last_chance_history(game_t * restrict game) {
  assert(game);
  return game->chance-1;
}

static inline int first_player_history(game_t * restrict game) {
  assert(game);
  return game->chance;
}

static inline int last_player_history(game_t * restrict game) {
  assert(game);
  return game->chance+game->player-1;
}

static inline int first_internal_history(game_t * restrict game) {
  assert(game);
  return 0;
}

static inline int last_internal_history(game_t * restrict game) {
  return game->chance+game->player-1;
}

static inline int first_terminal_history(game_t * restrict game) {
  return game->player+game->chance;
}

static inline int last_terminal_history(game_t * restrict game) {
  return game->player+game->chance+game->terminal-1;
}

static inline int total_chance_histories(game_t * restrict game) {
  assert(game);
  return game->chance;
}

static inline int total_player_histories(game_t * restrict game, int who) {
  assert(game);
  
  if (who == BOTH) {
    return game->player;
  } else {
    assert(valid_player(who));
    return game->player_histories[who];
  }
}

static inline int total_internal_histories(game_t * restrict game) {
  assert(game);
  return game->chance + game->player;
}

static inline int total_terminal_histories(game_t * restrict game) {
  assert(game);
  return game->terminal;
}

static inline int total_histories(game_t * restrict game) {
  assert(game);
  return game->chance + game->player + game->terminal;
}

static inline int total_actions(game_t * restrict game) {
  assert(game);
  return game->actions;
}

static inline int total_chance_actions(game_t * restrict game) {
  assert(game);
  return game->chance_actions;
}

static inline int total_player_actions(game_t * restrict game, int who) {
  assert(game);
  
  if (who == BOTH) {
    return game->player_actions[0]+game->player_actions[1];
  } else {
    assert(valid_player(who));
    return game->player_actions[who];
  }
}

static inline int valid_info_set(game_t * restrict game, int who, int info_set) {
  assert(game);
  assert(valid_player(who));
  return info_set >= 0 && info_set < game->info_sets[who];
}

static inline int total_info_sets(game_t * restrict game, int who) {
  assert(game);
  
  if (who == BOTH) {
    return game->info_sets[0]+game->info_sets[1];
  } else {
    assert(valid_player(who));
    return game->info_sets[who];
  }
}

static inline int valid_action(game_t * restrict game, int action) {
  assert(game);
  return action >= 0 && action < game->actions;
}

static inline int chance_action(game_t * restrict game, int action) {
  assert(valid_action(game, action));
  return action < game->chance_actions;
}

static inline int player_action(game_t * restrict game, int action) {
  assert(valid_action(game, action));
  return action >= game->chance_actions;
}

static inline const char * game_name(game_t * restrict game) {
  assert(game);
  return game->game_name;
}

static inline int game_root(game_t * restrict game) {
  assert(game);
  return game->root;
}

static inline const char * history_name(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return game->name[history];
}

static inline int parent(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return game->parent[history];
}

static inline int num_actions(game_t * restrict game, int history) {
  assert(internal_history(game, history));
  return game->num_actions[history];
}

static inline int first_action(game_t * restrict game, int history) {
  assert(internal_history(game, history));
  return game->first_action[history];
}

static inline int last_action(game_t * restrict game, int history) {
  assert(internal_history(game, history));
  return game->first_action[history]+game->num_actions[history]-1;
}

static inline int ith_action(game_t * restrict game, int history, int i) {
  assert(i >= 0 && i < num_actions(game, history));
  return first_action(game, history)+i;
}

static inline int do_action(game_t * restrict game, int action) {
  assert(valid_action(game, action));
  return game->action[action];
}

static inline int action_to_history(game_t * restrict game, int action) {
  assert(valid_action(game, action));
  return game->action_history[action];
}

static inline const char * action_name(game_t * restrict game, int action) {
  assert(valid_action(game, action));
  return game->action_name[action];
}

static inline int chance_depth(game_t * restrict game, int history) {
  assert(chance_history(game, history));
  return game->chance_depth[history];
}

static inline double sigma_chance(game_t * restrict game, int action) {
  assert(chance_action(game, action));
  return game->sigma_chance[action];
}

static inline const double * sigma_chance_history(game_t * restrict game, int history) {
  assert(chance_history(game, history));
  return game->sigma_chance+first_action(game, history);
}

static inline double pi_chance(game_t * restrict game, int history) {
  assert(valid_history(game, history));
  return game->pi_chance[history];
}

static inline int who(game_t * restrict game, int history) {
  assert(player_history(game, history));
  return game->who[history-game->chance];
}

static inline int info_set(game_t * restrict game, int history) {
  assert(player_history(game, history));
  return game->info_set[history-game->chance];
}

static inline int value(game_t * restrict game, int history) {
  assert(terminal_history(game, history));
  return game->value[history-game->chance-game->player];
}

static inline int value_for(game_t * restrict game, int history, int who) {
  assert(terminal_history(game, history));
  assert(valid_player(who));
  return value_multipler(who)*game->value[history-game->chance-game->player];
}

static inline int min_u(game_t * restrict game) {
  assert(game);
  return game->min_u;
}

static inline int max_u(game_t * restrict game) {
  assert(game);
  return game->max_u;
}

static inline int delta_u(game_t * restrict game) {
  assert(game);
  return game->max_u - game->min_u;
}

static inline int max_chance_depth(game_t * restrict game) {
  assert(game);
  return game->max_chance_depth;
}

#endif /* _GAME_H_ */
