/*
 * game.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <stdio.h>
#include <string.h>

#include "util.h"
#include "verify.h"

#include "game.h"

void read_game(FILE * restrict stream, game_t * restrict game) {
  char tmp[64];
  int i, j, k, n, m1, m2, id;
  int * q, * chance_depth;

  assert(stream);
  assert(game);

  verify(remove_comment(stream));
  verify(fscanf(stream, "'%[^']'", tmp) == 1);
  game->game_name = xstrdup(tmp);

  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->chance) == 1);
  verify(game->chance >= 0);

  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->player) == 1);
  verify(game->player >= 0);
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->terminal) == 1);
  verify(game->terminal > 0);
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->root) == 1);
  verify(internal_history(game, game->root));

  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->actions) == 1);
  verify(game->actions >= 0);

  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->chance_actions) == 1);
  verify(game->chance_actions >= 0);
  verify(game->chance_actions <= game->actions);
  verify(game->actions == total_histories(game)-1);

  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->info_sets[0]) == 1);
  verify(game->info_sets[0] >= 0);
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &game->info_sets[1]) == 1);
  verify(game->info_sets[1] >= 0);

  game->player_histories[0] = game->player_histories[1] = 0;
  game->player_actions[0]   = game->player_actions[1]   = 0;

  game->name           = xcalloc(total_histories(game), sizeof(char*));
  game->parent         = xmalloc(total_histories(game)*sizeof(int));
  game->first_action   = xcalloc(game->chance+game->player, sizeof(int));
  game->num_actions    = xcalloc(game->chance+game->player, sizeof(int));
  game->action         = xcalloc(game->actions, sizeof(int));
  game->action_name    = xcalloc(game->actions, sizeof(char*));
  game->action_history = xcalloc(game->actions, sizeof(int));
  game->sigma_chance   = xcalloc(game->chance_actions, sizeof(double));
  game->who            = xcalloc(game->player, sizeof(int));
  game->info_set       = xcalloc(game->player, sizeof(int));
  game->value          = xcalloc(game->terminal, sizeof(int));
  game->pi_chance      = xcalloc(total_histories(game), sizeof(double));

  memset(game->parent, -1, total_histories(game)*sizeof(int));

  game->min_u = 1<<30;
  game->max_u = -(1<<30);
 
  m1 = 0;
  m2 = game->chance_actions;
  for(k=0; k<total_histories(game); ++k) {
    verify(remove_comment(stream));
    verify(fscanf(stream, "%d", &id) == 1);
    verify(!game->name[id]);
    i = id;
    if (chance_history(game, id)) {
      verify(remove_comment(stream));
      verify(fscanf(stream, "'%[^']'", tmp) == 1);
      game->name[i] = xstrdup(tmp);
      
      verify(remove_comment(stream));
      verify(fscanf(stream, "%d", &n) == 1);
      verify(n > 0);
      verify(n+m1 <= game->chance_actions);
      
      game->first_action[i] = m1;
      game->num_actions[i]  = n;
      for(j=0; j<n; ++j) {
	game->action_history[m1+j] = i;

	verify(remove_comment(stream));
	verify(fscanf(stream, "'%[^']'", tmp) == 1);
	game->action_name[m1+j] = xstrdup(tmp);
	
	verify(remove_comment(stream));
	verify(fscanf(stream, "%d", &game->action[m1+j]));
	verify(valid_history(game, game->action[m1+j]));
	verify(game->parent[game->action[m1+j]] == -1);
	game->parent[game->action[m1+j]] = i;
	
	verify(remove_comment(stream));
	verify(fscanf(stream, "%lf", &game->sigma_chance[m1+j]));
	verify(game->sigma_chance[m1+j] > 0);
      }
      normalize(game->sigma_chance+m1, n, 1.);
      m1 += n;
    } else if (player_history(game, id)) {
      id -= game->chance;

      verify(remove_comment(stream));
      verify(fscanf(stream, "'%[^']'", tmp) == 1);
      game->name[i] = xstrdup(tmp);

      verify(remove_comment(stream));
      verify(fscanf(stream, "%d", &game->who[id]) == 1);
      verify(game->who[id] == 0 || game->who[id] == 1);

      verify(remove_comment(stream));
      verify(fscanf(stream, "%d", &game->info_set[id]) == 1);
      verify(valid_info_set(game, game->who[id], game->info_set[id]));

      verify(remove_comment(stream));
      verify(fscanf(stream, "%d", &n) == 1);
      verify(n > 0);
      verify(n+m2 <= game->actions);
    
      ++game->player_histories[game->who[id]];
      game->player_actions[game->who[id]] += n;
      game->first_action[i] = m2;
      game->num_actions[i]  = n;
      for(j=0; j<n; ++j) {
	game->action_history[m2+j] = i;

	verify(remove_comment(stream));
	verify(fscanf(stream, "'%[^']'", tmp) == 1);
	game->action_name[m2+j] = xstrdup(tmp);
	
	verify(remove_comment(stream));
	verify(fscanf(stream, "%d", &game->action[m2+j]));
	verify(valid_history(game, game->action[m2+j]));
	verify(game->parent[game->action[m2+j]] == -1);
	game->parent[game->action[m2+j]] = i;
      }
      m2 += n;
    } else {
      verify(terminal_history(game, id));
      id -= game->chance+game->player;

      verify(remove_comment(stream));
      verify(fscanf(stream, "'%[^']'", tmp) == 1);
      game->name[i] = xstrdup(tmp);

      verify(remove_comment(stream));
      verify(fscanf(stream, "%d", &game->value[id]) == 1);

      game->min_u = mini(game->min_u, game->value[id]);
      game->max_u = maxi(game->max_u, game->value[id]);
    }
  }

  verify(game->parent[game->root] == -1);
  verify(m1 == game->chance_actions);
  verify(m2 == game->actions);
  game->first_action[game->chance+game->player] = m2;

  game->chance_depth     = xmalloc(sizeof(int)*game->chance);
  game->max_chance_depth = 0;
  q            = xmalloc(sizeof(int)*total_histories(game));
  chance_depth = xmalloc(sizeof(int)*total_histories(game));
  game->pi_chance[game_root(game)] = 1.;
  for(q[j=1,chance_depth[i=0]=0]=game_root(game); i!=j; ++i) {
    id = q[i];
    if (chance_history(game, id)) {
      game->chance_depth[id] = chance_depth[i];
      for(k=first_action(game, id); k<=last_action(game, id); ++k) {
	n = do_action(game, k);
	game->pi_chance[n] = game->pi_chance[id]*sigma_chance(game, k);
	chance_depth[j]    = chance_depth[i]+1;
	if (game->max_chance_depth < chance_depth[j]) {
	  game->max_chance_depth = chance_depth[j];
	}
	q[j++] = n;
      }
    } else if (player_history(game, id)) {
      for(k=first_action(game, id); k<=last_action(game, id); ++k) {
	n = do_action(game, k);
	game->pi_chance[n] = game->pi_chance[id];
	chance_depth[j]    = chance_depth[i];
	q[j++]             = n;
      }
    }
  }
  xfree(q);
}

void free_game(game_t * restrict game) {
  int i;

  assert(game);

  for(i=0; i<total_histories(game); ++i) {
    xfree(game->name[i]);
  }
  for(i=0; i<total_actions(game); ++i) {
    xfree(game->action_name[i]);
  }
  xfree(game->pi_chance);
  xfree(game->value);
  xfree(game->who);
  xfree(game->sigma_chance);
  xfree(game->action_history);
  xfree(game->action_name);
  xfree(game->action);
  xfree(game->num_actions);
  xfree(game->first_action);
  xfree(game->action);
  xfree(game->name);
  xfree(game->game_name);
}
