/*
 * extract.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "best_response.h"
#include "game.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --A=[output]\n"
	   "  --E=[output]\n"
	   "  --F=[output]\n"
	   "  --x=[output]\n"
	   "  --y=[output]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"A", required_argument, 0, 8},
      {"E", required_argument, 0, 9},
      {"F", required_argument, 0, 10},
      {"x", required_argument, 0, 11},
      {"y", required_argument, 0, 12},
      {NULL},
    };
    int ch, i, j;
    game_t game;
    abstraction_t abstraction[2];
    sequences_t sequences[2];
    sequence_form_t sequence_form;
    double * sigma_x, * sigma_y;
    const char * game_path;
    const char * abstraction_name[2];
    const char * A_path, * E_path,  * F_path, * x_path, * y_path;
    FILE * stream;

    game_path = NULL;
    abstraction_name[0] = abstraction_name[1] = "null";
    A_path = E_path = F_path = x_path = y_path = NULL;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_name[0] = optarg; break;
      case 7: abstraction_name[1] = optarg; break;
      case 8: A_path = optarg; break;
      case 9: E_path = optarg; break;
      case 10: F_path = optarg; break;
      case 11: x_path = optarg; break;
      case 12: y_path = optarg; break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_name[0], &game, 0, &abstraction[0]);
    load_abstraction(abstraction_name[1], &game, 1, &abstraction[1]);

    build_sequences(&abstraction[0], &sequences[0]);
    build_sequences(&abstraction[1], &sequences[1]);
    
    build_sequence_form(&sequences[0], &sequences[1], &sequence_form);
    
    sigma_x = new_uniform_strategy(&sequences[0]);
    sigma_y = new_uniform_strategy(&sequences[1]);

    if (A_path) {
      stream = open_stream(A_path, "wt");
      for(i=0; i<sequences_n(&sequences[0]); ++i) {
	for(j=0; j<sequences_n(&sequences[1]); ++j) {
	  fprintf(stream, j?" %g":"%g", sequence_form_A(&sequence_form, i, j));
	}
	fprintf(stream, "\n");
      }
      close_stream(A_path, stream);
    }

    if (E_path) {
      stream = open_stream(E_path, "wt");
      fprintf(stream, "1 1 1\n");
      for(i=0; i<abstract_info_sets(&abstraction[0]); ++i) {
	fprintf(stream, "%d %d -1\n", 2+i, parent_sequence(&sequences[0], i)+1);
	for(j=first_sequence(&sequences[0], i); j<=last_sequence(&sequences[0], i); ++j) {
	  fprintf(stream, "%d %d 1\n", 2+i, j+1);
	}
      }
      close_stream(E_path, stream);
    }

    if (F_path) {
      stream = open_stream(F_path, "wt");
      fprintf(stream, "1 1 1\n");
      for(i=0; i<abstract_info_sets(&abstraction[1]); ++i) {
	fprintf(stream, "%d %d -1\n", 2+i, parent_sequence(&sequences[1], i)+1);
	for(j=first_sequence(&sequences[1], i); j<=last_sequence(&sequences[1], i); ++j) {
	  fprintf(stream, "%d %d 1\n", 2+i, j+1);
	}
      }
      close_stream(F_path, stream);
    }

    if (x_path) {
      stream = open_stream(x_path, "wt");
      for(i=0; i<sequences_n(&sequences[0]); ++i) {
	fprintf(stream, "%g\n", sigma_x[i]);
      }
      close_stream(x_path, stream);
    }

    if (y_path) {
      stream = open_stream(y_path, "wt");
      for(i=0; i<sequences_n(&sequences[1]); ++i) {
	fprintf(stream, "%g\n", sigma_y[i]);
      }
      close_stream(y_path, stream);
    }

    xfree(sigma_y);
    xfree(sigma_x);

    free_sequence_form(&sequence_form);

    free_sequences(&sequences[1]);
    free_sequences(&sequences[0]);
    
    free_abstraction(&abstraction[1]);
    free_abstraction(&abstraction[0]);

    free_game(&game);
  }
  return 0;
}
