/*
 * exploitability.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "best_response.h"
#include "game.h"
#include "lp_eqm.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --strategy=[input]\n"
	   "  --who=[player]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"strategy", required_argument, 0, 8},
      {"who", required_argument, 0, 10},
      {NULL},
    };
    int ch, who;
    game_t game;
    abstraction_t abstraction_original[2];
    sequences_t sequences_original[2];
    sequence_form_t sequence_form_original;
    double * sigma, * payoffs;
    const char * game_path;
    const char * abstraction_original_name[2];
    const char * input_path;
    FILE * stream;
    double u, v;

    game_path = NULL;
    abstraction_original_name[0] = abstraction_original_name[1] = "null";
    input_path = NULL;
    who        = -1;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_original_name[0] = optarg; break;
      case 7: abstraction_original_name[1] = optarg; break;
      case 8: input_path = optarg; break;
      case 10: verify(sscanf(optarg, "%d", &who) == 1); break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);
    verify(valid_player(who));
    verify(input_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_original_name[0], &game, 0, &abstraction_original[0]);
    load_abstraction(abstraction_original_name[1], &game, 1, &abstraction_original[1]);

    build_sequences(&abstraction_original[0], &sequences_original[0]);
    build_sequences(&abstraction_original[1], &sequences_original[1]);
    
    build_sequence_form(&sequences_original[0], &sequences_original[1], &sequence_form_original);
    
    u = lp_eqm(&sequence_form_original, NULL, NULL, NULL, NULL);
    if (who == 0) {
      u = -u;
    }

    sigma        = new_strategy(&sequences_original[who]);
    payoffs      = new_strategy(&sequences_original[opponent(who)]);

    stream = open_stream(input_path, "rt");
    read_strategy(&sequences_original[who], stream, sigma);
    close_stream(input_path, stream);
    
    compute_payoffs(&sequence_form_original, opponent(who), sigma, payoffs);

    v = best_response(&sequences_original[opponent(who)], payoffs);
    printf("# exploitability = %g\n", v-u);

    xfree(payoffs);
    xfree(sigma);

    free_sequence_form(&sequence_form_original);

    free_sequences(&sequences_original[1]);
    free_sequences(&sequences_original[0]);
    
    free_abstraction(&abstraction_original[1]);
    free_abstraction(&abstraction_original[0]);

    free_game(&game);
  }
  return 0;
}
