/*
 * ev.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "best_response.h"
#include "game.h"
#include "lp_eqm.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --x=[input]\n"
	   "  --y=[input]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"x", required_argument, 0, 8},
      {"y", required_argument, 0, 9},
      {NULL},
    };
    int ch;
    game_t game;
    abstraction_t abstraction_original[2];
    sequences_t sequences_original[2];
    sequence_form_t sequence_form_original;
    double * sigma_x, * sigma_y, * payoffs;
    const char * game_path;
    const char * abstraction_original_name[2];
    const char * x_path, * y_path;
    FILE * stream;
    double u;

    game_path = NULL;
    abstraction_original_name[0] = abstraction_original_name[1] = "null";
    x_path = y_path = NULL;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_original_name[0] = optarg; break;
      case 7: abstraction_original_name[1] = optarg; break;
      case 8: x_path = optarg; break;
      case 9: y_path = optarg; break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);
    verify(x_path);
    verify(y_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_original_name[0], &game, 0, &abstraction_original[0]);
    load_abstraction(abstraction_original_name[1], &game, 1, &abstraction_original[1]);

    build_sequences(&abstraction_original[0], &sequences_original[0]);
    build_sequences(&abstraction_original[1], &sequences_original[1]);
    
    build_sequence_form(&sequences_original[0], &sequences_original[1], &sequence_form_original);

    sigma_x      = new_strategy(&sequences_original[0]);
    sigma_y      = new_strategy(&sequences_original[1]);
    payoffs      = new_strategy(&sequences_original[1]);

    stream = open_stream(x_path, "rt");
    read_strategy(&sequences_original[0], stream, sigma_x);
    close_stream(x_path, stream);
    normalize_strategy(&sequences_original[0], sigma_x, 1.);

    stream = open_stream(y_path, "rt");
    read_strategy(&sequences_original[1], stream, sigma_y);
    close_stream(y_path, stream);
    normalize_strategy(&sequences_original[1], sigma_y, 1.);
    
    compute_payoffs(&sequence_form_original, 1, sigma_x, payoffs);
    u = compute_ev(&sequence_form_original, 1, sigma_y, payoffs);
    printf("# ev = %g\n", -u);

    xfree(payoffs);
    xfree(sigma_y);
    xfree(sigma_x);

    free_sequence_form(&sequence_form_original);

    free_sequences(&sequences_original[1]);
    free_sequences(&sequences_original[0]);
    
    free_abstraction(&abstraction_original[1]);
    free_abstraction(&abstraction_original[0]);

    free_game(&game);
  }
  return 0;
}
