/*
 * best_response_main.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <getopt.h>
#include <stdio.h>

#include "abstraction.h"
#include "best_response.h"
#include "game.h"
#include "sequence_form.h"
#include "strategy.h"
#include "util.h"
#include "verify.h"

int main(int argc, char ** argv) {
  if (argc == 1) {
    printf("usage: %s [arguments]\n"
	   "\n"
	   "arugments:\n"
	   "  --game=[game]\n"
	   "  --abstraction1=[abstraction]\n"
	   "  --abstraction2=[abstraction]\n"
	   "  --strategy=[input]\n"
	   "  --output=[output]\n"
	   "  --output-lifted=[output]\n"
	   "  --who=[player]\n"
	   , argv[0]);
  } else {
    struct option opts[] = {
      {"game", required_argument, 0, 1},
      {"abstraction1", required_argument, 0, 6},
      {"abstraction2", required_argument, 0, 7},
      {"strategy", required_argument, 0, 8},
      {"output", required_argument, 0, 9},
      {"who", required_argument, 0, 10},
      {"output-lifted", required_argument, 0, 11},
      {NULL},
    };
    int ch, who;
    game_t game;
    abstraction_t abstraction_original[2], abstraction_null;
    sequences_t sequences_original[2], sequences_null;
    sequence_form_t sequence_form_original;
    double * sigma, * sigma_lifted, * payoffs;
    const char * game_path;
    const char * abstraction_original_name[2];
    const char * input_path, * output_path, * output_path_lifted;
    FILE * stream;
    double v;

    game_path = NULL;
    abstraction_original_name[0] = abstraction_original_name[1] = "null";
    input_path         = NULL;
    output_path        = NULL;
    output_path_lifted = NULL;

    while((ch = getopt_long(argc, argv, "", opts, NULL)) != -1) {
      switch(ch) {
      case 1: game_path = optarg; break;
      case 6: abstraction_original_name[0] = optarg; break;
      case 7: abstraction_original_name[1] = optarg; break;
      case 8: input_path = optarg; break;
      case 9: output_path = optarg; break;
      case 10: verify(sscanf(optarg, "%d", &who) == 1); break;
      case 11: output_path_lifted = optarg; break;
      default: verify("unknown argument");
      }
    }
    
    verify(game_path);
    verify(valid_player(who));
    verify(input_path);

    stream = open_stream(game_path, "rt");
    read_game(stream, &game);
    close_stream(game_path, stream);

    load_abstraction(abstraction_original_name[0], &game, 0, &abstraction_original[0]);
    load_abstraction(abstraction_original_name[1], &game, 1, &abstraction_original[1]);
    null_abstraction(&game, opponent(who), &abstraction_null); 

    build_sequences(&abstraction_original[0], &sequences_original[0]);
    build_sequences(&abstraction_original[1], &sequences_original[1]);
    build_sequences(&abstraction_null, &sequences_null);
    
    build_sequence_form(&sequences_original[0], &sequences_original[1], &sequence_form_original);
    
    sigma        = new_strategy(&sequences_original[who]);
    sigma_lifted = new_strategy(&sequences_null);
    payoffs      = new_strategy(&sequences_original[opponent(who)]);

    stream = open_stream(input_path, "rt");
    read_strategy(&sequences_original[who], stream, sigma);
    close_stream(input_path, stream);
    
    compute_payoffs(&sequence_form_original, opponent(who), sigma, payoffs);

    v = best_response(&sequences_original[opponent(who)], payoffs);
    printf("# best response value = %g\n", v);

    if (output_path) {
      stream = open_stream(output_path, "wt");
      write_strategy(&sequences_original[opponent(who)], payoffs, stream);
      close_stream(output_path, stream);
    }

    lift_strategy(&sequences_original[opponent(who)], &sequences_null, payoffs, sigma_lifted);
    if (output_path_lifted) {
      stream = open_stream(output_path_lifted, "wt");
      write_strategy(&sequences_null, sigma_lifted, stream);
      close_stream(output_path_lifted, stream);
    }

    xfree(payoffs);
    xfree(sigma_lifted);
    xfree(sigma);

    free_sequence_form(&sequence_form_original);

    free_sequences(&sequences_null);
    free_sequences(&sequences_original[1]);
    free_sequences(&sequences_original[0]);
    
    free_abstraction(&abstraction_null);
    free_abstraction(&abstraction_original[1]);
    free_abstraction(&abstraction_original[0]);

    free_game(&game);
  }
  return 0;
}
