/*
 * abstraction.h
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#ifndef _ABSTRACTION_H_
#define _ABSTRACTION_H_

#include <stdio.h>

#include "attributes.h"

#include "game.h"
#include "verify.h"

typedef struct {
  char * name;
  int who, info_sets;
  int * map;
  int * first_history, * history;
  int * first_member, * member;
  game_t * game;
} abstraction_t;

void load_abstraction(const char * restrict path, game_t * restrict game, int who, 
		      abstraction_t * restrict abstraction) NONNULL(1,2);
void read_abstraction(FILE * restrict stream, game_t * restrict game, 
		      abstraction_t * restrict abstraction) NONNULL(1,2);
void null_abstraction(game_t * restrict game, int who, 
		      abstraction_t * restrict abstraction) NONNULL(1,3);
void free_abstraction(abstraction_t * restrict abstraction) NONNULL(1);

int is_coarser(abstraction_t * restrict a, abstraction_t * restrict b) NONNULL(1,2) PURE;

static inline const char * abstraction_name(abstraction_t * restrict abstraction) NONNULL(1) PURE;
static inline int abstraction_who(abstraction_t * restrict abstraction) NONNULL(1) PURE;
static inline int abstract_info_sets(abstraction_t * restrict abstraction) NONNULL(1) PURE;
static inline int abstraction_map(abstraction_t * restrict abstraction, int info_set) NONNULL(1) PURE;
static inline int abstract_info_set(abstraction_t * restrict abstraction, 
				    int history) NONNULL(1) PURE;
static inline game_t * abstraction_game(abstraction_t * restrict abstraction) NONNULL(1) PURE;

static inline int valid_abstract_info_set(abstraction_t * restrict abstraction,
					  int info_set) NONNULL(1) PURE;
static inline int total_histories_in_info_set(abstraction_t * restrict abstraction, 
					      int info_set) NONNULL(1) PURE;
static inline int ith_history_in_info_set(abstraction_t * restrict abstraction,
					  int info_set, int i) NONNULL(1) PURE;
static inline int num_actions_from_info_set(abstraction_t * restrict abstraction, 
					    int info_set) NONNULL(1) PURE;
static inline const char * action_name_from_info_set(abstraction_t * restrict abstraction, 
						     int info_set) NONNULL(1) PURE;
static inline int ith_member(abstraction_t * restrict abstraction,
			     int info_set, int i) NONNULL(1) PURE;
static inline int num_members(abstraction_t * restrict abstraction,
			      int info_set) NONNULL(1) PURE;

static inline const char * abstraction_name(abstraction_t * restrict abstraction) {
  assert(abstraction);
  return abstraction->name;
}

static inline int abstraction_who(abstraction_t * restrict abstraction) {
  assert(abstraction);
  return abstraction->who;
}

static inline int abstract_info_sets(abstraction_t * restrict abstraction) {
  assert(abstraction);
  return abstraction->info_sets;
}

static inline int abstraction_map(abstraction_t * restrict abstraction, int info_set) {
  assert(abstraction);
  assert(valid_info_set(abstraction->game, abstraction->who, info_set));
  return abstraction->map?abstraction->map[info_set]:info_set;
}

static inline int abstract_info_set(abstraction_t * restrict abstraction, 
				    int history) {
  assert(abstraction);
  assert(who(abstraction_game(abstraction), history) == abstraction_who(abstraction));
  return abstraction_map(abstraction, info_set(abstraction_game(abstraction), history));
}

static inline game_t * abstraction_game(abstraction_t * restrict abstraction) {
  assert(abstraction);
  return abstraction->game;
}

static inline int valid_abstract_info_set(abstraction_t * restrict abstraction,
					  int info_set) {
  return info_set >= 0 && info_set < abstract_info_sets(abstraction);
}

static inline int total_histories_in_info_set(abstraction_t * restrict abstraction,
					      int info_set) {
  assert(valid_abstract_info_set(abstraction, info_set));
  return abstraction->first_history[info_set+1] - abstraction->first_history[info_set];
}

static inline int ith_history_in_info_set(abstraction_t * restrict abstraction,
					  int info_set, int i) {
  assert(i >= 0);
  assert(i < total_histories_in_info_set(abstraction, info_set));
  return abstraction->history[abstraction->first_history[info_set]+i];
}

static inline int num_actions_from_info_set(abstraction_t * restrict abstraction,
					    int info_set) {
  return num_actions(abstraction_game(abstraction), ith_history_in_info_set(abstraction, info_set, 0));
}

static inline const char * action_name_from_info_set(abstraction_t * restrict abstraction,
						     int info_set) {
  return action_name(abstraction_game(abstraction), ith_history_in_info_set(abstraction, info_set, 0));
}

static inline int ith_member(abstraction_t * restrict abstraction,
			     int info_set, int i) {
  assert(i >= 0 && i < num_members(abstraction, info_set));
  return abstraction->member[abstraction->first_member[info_set]+i];
}

static inline int num_members(abstraction_t * restrict abstraction,
			      int info_set) {
  assert(valid_abstract_info_set(abstraction, info_set));
  return abstraction->first_member[info_set+1] - abstraction->first_member[info_set];
}

#endif /* _ABSTRACTION_H_ */
