/*
 * abstraction.c
 * author: Kevin Waugh (waugh@cs.cmu.edu)
 */

#include <stdio.h>
#include <string.h>

#include "game.h"
#include "util.h"
#include "verify.h"

#include "abstraction.h"

static void use_abstraction(abstraction_t * restrict abstraction) NONNULL(1);

void load_abstraction(const char * restrict path, game_t * restrict game, int who,
		      abstraction_t * restrict abstraction) {
  FILE * stream;

  assert(path);
  assert(valid_player(who));
  
  if (!strcmp(path, "null")) {
    null_abstraction(game, who, abstraction);
  } else {
    stream = open_stream(path, "rt");
    read_abstraction(stream, game, abstraction);
    close_stream(path, stream);
    verify(abstraction->who == who);
  }
}

void read_abstraction(FILE * restrict stream, game_t * restrict game, 
		      abstraction_t * restrict abstraction) {
  char tmp[64];
  int i, from, to;
  
  assert(stream);
  assert(game);
  assert(abstraction);

  verify(remove_comment(stream));
  verify(fscanf(stream, "'%[^']'", tmp) == 1);
  abstraction->name = xstrdup(tmp);
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "'%[^']'", tmp) == 1);
  verify(!strcmp(tmp, game_name(game)));
  abstraction->game = game;
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &abstraction->who) == 1);
  verify(valid_player(abstraction->who));
  
  verify(remove_comment(stream));
  verify(fscanf(stream, "%d", &abstraction->info_sets) == 1);
  verify(abstraction->info_sets >= 0);
  verify(abstraction->info_sets <= total_info_sets(game, abstraction->who));

  abstraction->map = xmalloc(total_info_sets(game, abstraction->who)*sizeof(int));
  memset(abstraction->map, -1, total_info_sets(game, abstraction->who)*sizeof(int));
  for(i=0; i<total_info_sets(game, abstraction->who); ++i) {
    verify(remove_comment(stream));
    verify(fscanf(stream, "%d", &from) == 1);
    verify(valid_info_set(game, abstraction->who, from));
    verify(abstraction->map[from] == -1);

    verify(remove_comment(stream));
    verify(fscanf(stream, "%d", &to) == 1);
    verify(to >= 0 && to < abstraction->info_sets);
    
    abstraction->map[from] = to;
  }

  use_abstraction(abstraction);
}

void null_abstraction(game_t * restrict game, int who, 
		      abstraction_t * restrict abstraction) {
  assert(game);
  assert(valid_player(who));
  assert(abstraction);

  abstraction->name      = xstrdup("null");
  abstraction->who       = who;
  abstraction->info_sets = total_info_sets(game, who);
  abstraction->map       = NULL;
  abstraction->game      = game;

  use_abstraction(abstraction);
}

void free_abstraction(abstraction_t * restrict abstraction) {
  assert(abstraction);

  xfree(abstraction->member);
  xfree(abstraction->first_member);
  xfree(abstraction->history);
  xfree(abstraction->first_history);
  xfree(abstraction->map);
  xfree(abstraction->name);
}

int is_coarser(abstraction_t * restrict a, abstraction_t * restrict b) {
  int i, j, k, ii;

  assert(a);
  assert(b);
  assert(abstraction_game(a) == abstraction_game(b));
  assert(abstraction_who(a) == abstraction_who(b));

  for(i=0; i<abstract_info_sets(b); ++i) {
    k = ith_member(b, i, 0);
    for(j=1; j<num_members(b, i); ++j) {
      ii = ith_member(b, i, j);
      if (abstraction_map(a, ii) != abstraction_map(a, k)) {
	return 0;
      }
    }
  }

  return 1;
}

static void use_abstraction(abstraction_t * restrict abstraction) {
  int i, id, n, j, k, u, v, m1, m2;
  game_t * game;

  game = abstraction_game(abstraction);
  
  abstraction->first_history = xcalloc(abstract_info_sets(abstraction)+1, sizeof(int));
  abstraction->history       = xcalloc(total_player_histories(game, abstraction_who(abstraction)),
				       sizeof(int));

  for(i=first_player_history(game); i<=last_player_history(game); ++i) {
    if (who(game, i) == abstraction_who(abstraction)) {
      id = abstract_info_set(abstraction, i);
      ++abstraction->first_history[id];
    }
  }
  for(i=n=0; i<=abstract_info_sets(abstraction); ++i) {
    j = abstraction->first_history[i];
    abstraction->first_history[i] = n;
    n += j;
  }
  for(i=first_player_history(game); i<=last_player_history(game); ++i) {
    if (who(game, i) == abstraction_who(abstraction)) {
      id = abstract_info_set(abstraction, i);
      abstraction->history[abstraction->first_history[id]++] = i;
    }
  }
  for(i=abstract_info_sets(abstraction); i>0; --i) {
    abstraction->first_history[i] = abstraction->first_history[i-1];
  }
  abstraction->first_history[0] = 0;

  for(j=0; j<abstract_info_sets(abstraction); ++j) {
    u = ith_history_in_info_set(abstraction, j, 0);
    for(k=1; k<total_histories_in_info_set(abstraction, j); ++k) {
      v = ith_history_in_info_set(abstraction, j, k);
      verify(num_actions(game, u) == num_actions(game, v));
      for(m1=first_action(game, u), m2=first_action(game, v); m1<=last_action(game, u); ++m1, ++m2) {
	verify(!strcmp(action_name(game, m1), action_name(game, m2)));
      }
    }
  }

  abstraction->first_member = xcalloc(abstract_info_sets(abstraction)+1, sizeof(int));
  abstraction->member       = xcalloc(total_info_sets(game, abstraction_who(abstraction)), 
				      sizeof(int));
  for(i=0; i<total_info_sets(game, abstraction_who(abstraction)); ++i) {
    ++abstraction->first_member[abstraction_map(abstraction, i)];
  }
  for(i=n=0; i<abstract_info_sets(abstraction); ++i) {
    j = abstraction->first_member[i];
    abstraction->first_member[i] = n;
    n += j;
  }
  for(i=0; i<total_info_sets(game, abstraction_who(abstraction)); ++i) {
    abstraction->member[abstraction->first_member[abstraction_map(abstraction, i)]++] = i;
  }
  for(i=abstract_info_sets(abstraction); i>0; --i) {
    abstraction->first_member[i] = abstraction->first_member[i-1];
  }
  abstraction->first_member[0] = 0;
}
