/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2009-2011, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of KTH Speech Music and Hearing nor the names of its contributors may be
      used to endorse or promote products derived from this software without specific prior written
      permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#define _POSIX_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <math.h>
#include <sndfile.h>
#include "filename.h"
#include "stopwatch.h"
#include "windowpair.h"
#include "sutils.h"
#include "msf.h"

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#define NPOLICYMAX 10

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

extern char *  optarg;
extern int     optind;
extern int     opterr;
extern int     optopt;

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static char *  progName;
 
/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void PRINTUSAGE()
{
	fprintf( stderr, "\n" );
	fprintf( stderr, "Usage: %s\n", progName );
	fprintf( stderr, "\t[ -tint <seconds> | -tint1 <seconds> ... -tint9 <seconds> ]\n" );
	fprintf( stderr, "\t[ -text <seconds> | -text1 <seconds> ... -text9 <seconds> ]\n" );
	fprintf( stderr, "\t[ -tsep <seconds> ] -tsep1 <seconds> ... -tsep9 <seconds> ]\n" );
	fprintf( stderr, "\t[ -winShape <shape> | -winShape1 <shape> ... -winShape9 <shape> ]\n" );
	fprintf( stderr, "\t[ -tfra <seconds> ]\n" );
	fprintf( stderr, "\t[ -ljustifiedmean <seconds> | -rjustifiedmean <seconds>\n" );
	fprintf( stderr, "\t\t| -centeredmean <seconds> | -forgetting <alpha> ]\n" );
	fprintf( stderr, "\t<audioInLocation>\n" );
	fprintf( stderr, "\t[ -stdout | <featOutLocation> ]\n" );
	fprintf( stderr, "\n" );

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

WinShape str2winShape
	(
		const char *  string
	)
{
	WinShape  winShape;

	if ( strcmp( optarg, "hamming_hamming" ) == 0 )
	{
		winShape = WinShape_HAMMING_HAMMING;
	}
	else if ( strcmp( optarg, "hann_hann" ) == 0 )
	{
		winShape = WinShape_HANN_HANN;
	}
	else if ( strcmp( optarg, "hamming_hann" ) == 0 )
	{
		winShape = WinShape_HAMMING_HANN;
	}
	else
	{
		PRINTUSAGE();
		fprintf
			(
				stderr,
				"%s: winShape must be on of \
					{ hamming_hamming, hann_hann, hamming_hann }\n",
				progName
			);
		exit( -1 );
	}

	return winShape;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

double str2postProcessMethod
	(
		double        lJustifiedMean,
		double        rJustifiedMean,
		double        centeredMean,
		double        forgettingAlpha,
		const char *  label,
		char *        string
	)
{
	double retVal;

	if
		(
			( ! isnan( lJustifiedMean ) )
		||
			( ! isnan( rJustifiedMean ) )
		||
			( ! isnan( centeredMean ) )
		||
			( ! isnan( forgettingAlpha ) )
		)
	{
		PRINTUSAGE();
		fprintf
			(
				stderr,
				"%s: cannot specify %s,",
				progName,
				label
			);

		if ( ! isnan( lJustifiedMean ) )
		{
			fprintf( stderr, "ljustifiedmean already specified.\n" );
		}
		else if ( ! isnan( rJustifiedMean ) )
		{
			fprintf( stderr, "rjustifiedmean already specified.\n" );
		}
		else if ( ! isnan( centeredMean ) )
		{
			fprintf( stderr, "centeredmean already specified.\n" );
		}
		else /* if ( ! isnan( forgettingAlpha ) ) */
		{
			fprintf( stderr, "forgetting already specified.\n" );
		}

		exit( -1 );
	}

	retVal = str2dbl( string );
	if ( retVal <= 0.0 )
	{
		PRINTUSAGE();
		fprintf( stderr, "%s: %s value cannot be non-positive.\n", progName, label );
		exit( -1 );
	}

	return retVal;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

int  policyCOUNTDBL
	(
		const char *  label,
		double        tab[ NPOLICYMAX ],
		double        defaultValue
	)
{
	int  nPolicies;
	int  policyIdx;

	nPolicies = 0;
	for ( policyIdx = 1; policyIdx < NPOLICYMAX; policyIdx ++ )
	{
		if ( isnan( tab[ policyIdx ] ) )
		{
			break;
		}
		else
		{
			if ( tab[ policyIdx ] <= 0.0 )
			{
				PRINTUSAGE();
				fprintf
					(
						stderr,
						"%s: %s%d value cannot be non-positive.\n",
						progName,
						label,
						policyIdx 
					);
				exit( -1  );
			}
			nPolicies ++;
		}
	}
	for ( ; policyIdx < NPOLICYMAX; policyIdx ++ )
	{
		if ( ! isnan( tab[ policyIdx ] ) )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: policies beyond %d for %s are defined\n",
					progName,
					policyIdx + 1,
					label
				);
			exit( -1 );
		}
	}

	if ( nPolicies > 0 )
	{
		if ( ! isnan( tab[ 0 ] ) )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: both %s and %s<n> options invoked\n",
					progName,
					label,
					label
				);
			exit( -1 );
		}

		for ( policyIdx = 1; policyIdx <= nPolicies; policyIdx ++ )
		{
			tab[ policyIdx - 1 ] = tab[ policyIdx ];
		}
	}
	else if ( ! isnan( tab[ 0 ] ) )
	{
		if ( tab[ 0 ] <= 0.0 )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: %s value cannot be non-positive.\n",
					progName,
					label
				);
			exit( -1 );
		}

		nPolicies = 1;
	}
	else if ( nPolicies == 0 )
	{
		tab[ 0 ] = defaultValue;
		nPolicies = 1;
	}

	return nPolicies;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

int  policyCOUNTWINSHAPE
	(
		WinShape  tab[ NPOLICYMAX ],
		WinShape  defaultValue
	)
{
	int  nPolicies;
	int  policyIdx;

	nPolicies = 0;
	for ( policyIdx = 1; policyIdx < NPOLICYMAX; policyIdx ++ )
	{
		if ( tab[ policyIdx ] == WinShape_UNDEFINED )
		{
			break;
		}
		else
		{
			nPolicies ++;
		}
	}
	for ( ; policyIdx < NPOLICYMAX; policyIdx ++ )
	{
		if ( tab[ policyIdx ] != WinShape_UNDEFINED )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: policies beyond %d for winShape are defined\n",
					progName,
					policyIdx + 1
				);
			exit( -1 );
		}
	}

	if ( nPolicies > 0 )
	{
		if ( tab[ 0 ] != WinShape_UNDEFINED )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: both winShape and winShape<n> options invoked\n",
					progName
				);
			exit( -1 );
		}

		for ( policyIdx = 1; policyIdx <= nPolicies; policyIdx ++ )
		{
			tab[ policyIdx - 1 ] = tab[ policyIdx ];
		}
	}
	else if ( tab[ 0 ] != WinShape_UNDEFINED )
	{
		nPolicies = 1;
	}
	else if ( nPolicies == 0 )
	{
		tab[ 0 ] = defaultValue;
		nPolicies = 1;
	}

	return nPolicies;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

int policyMERGECOUNTS
	(
		const int  nPolicies_tint,
		const int  nPolicies_text,
		const int  nPolicies_tsep,
		const int  nPolicies_winShape
	)
{
	int  nPolicies;

	/*
	 * Get the maximum number of policies for any parameter.
	 */

	nPolicies = -1;
	if ( nPolicies_tint > nPolicies )
	{
		nPolicies = nPolicies_tint;
	}
	if ( nPolicies_text > nPolicies )
	{
		nPolicies = nPolicies_text;
	}
	if ( nPolicies_tsep > nPolicies )
	{
		nPolicies = nPolicies_tsep;
	}
	if ( nPolicies_winShape > nPolicies )
	{
		nPolicies = nPolicies_winShape;
	}

	/*
	 * Check that any parameter with more than one policy has the same number of policies as
	 *  the maximum.
	 */

	if ( nPolicies_tint > 1 )
	{
		if ( nPolicies_tint != nPolicies )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: number of tint policies is not equal to %d\n",
					progName,
					nPolicies
				);
			exit( -1 );
		}
	}
	if ( nPolicies_text > 1 )
	{
		if ( nPolicies_text != nPolicies )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: number of text policies is not equal to %d\n",
					progName,
					nPolicies
				);
			exit( -1 );
		}
	}
	if ( nPolicies_tsep > 1 )
	{
		if ( nPolicies_tsep != nPolicies )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: number of tsep policies is not equal to %d\n",
					progName,
					nPolicies
				);
			exit( -1 );
		}
	}
	if ( nPolicies_winShape > 1 )
	{
		if ( nPolicies_winShape != nPolicies )
		{
			PRINTUSAGE();
			fprintf
				(
					stderr,
					"%s: number of winShape policies is not equal to %d\n",
					progName,
					nPolicies
				);
			exit( -1 );
		}
	}

	return nPolicies;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

int main( int argc, char * argv[] )
{
	int            policyIdx;
	double         tint[ NPOLICYMAX ];
	double         text[ NPOLICYMAX ];
	double         tsep[ NPOLICYMAX ];
	WinShape       winShape[ NPOLICYMAX ];
	double         tfra;
	double         fs;
	double         lJustifiedMean;
	double         rJustifiedMean;
	double         centeredMean;
	double         forgettingAlpha;
	int            postProcessFlag;
	int            stdoutFlag;
	int            nPolicies;
	MsfComputer *  msfComputerPtrTab[ NPOLICYMAX ];
	int            Tsize[ NPOLICYMAX ];
	int            halfTsize[ NPOLICYMAX ];
	char *         inLocation;
	int            nInFiles;
	char **        inFileNameTab;
	char **        inPathNameTab;
	
	progName = argv[ 0 ];

	{
		int  argIdx;

		printf( "Command line: %s", progName );
		for ( argIdx = 1; argIdx < argc; argIdx ++ )
		{
			printf( " %s", argv[ argIdx ] );
		}
		printf( "\n" );
	}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Initialize command-line-modifiable parameters with default values.
	 */

	for ( policyIdx = 0; policyIdx < NPOLICYMAX; policyIdx ++ )
	{
		tint[ policyIdx ] = NAN;
		text[ policyIdx ] = NAN;
		tsep[ policyIdx ] = NAN;
		winShape[ policyIdx ] = WinShape_UNDEFINED;

		msfComputerPtrTab[ policyIdx ] = (MsfComputer *) NULL;
	}

	tfra = default_tfra;
	fs = default_fs;

	lJustifiedMean = NAN;
	rJustifiedMean = NAN;
	centeredMean = NAN;
	forgettingAlpha = NAN;
	postProcessFlag = 0;

	stdoutFlag = 0;

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Parse command line to modify parameters, if applicable.
	 */

	while ( 1 )
	{
		int                   optChar;
		int                   option_index;
		static struct option  long_options[] =
			{
				{ "tint",       1, 0,  1 },
				{ "tint1",      1, 0,  2 },
				{ "tint2",      1, 0,  3 },
				{ "tint3",      1, 0,  4 },
				{ "tint4",      1, 0,  5 },
				{ "tint5",      1, 0,  6 },
				{ "tint6",      1, 0,  7 },
				{ "tint7",      1, 0,  8 },
				{ "tint8",      1, 0,  9 },
				{ "tint9",      1, 0, 10 },
				
				{ "text",       1, 0, 11 },
				{ "text1",      1, 0, 12 },
				{ "text2",      1, 0, 13 },
				{ "text3",      1, 0, 14 },
				{ "text4",      1, 0, 15 },
				{ "text5",      1, 0, 16 },
				{ "text6",      1, 0, 17 },
				{ "text7",      1, 0, 18 },
				{ "text8",      1, 0, 19 },
				{ "text9",      1, 0, 20 },

				{ "tsep",       1, 0, 21 },
				{ "tsep1",      1, 0, 22 },
				{ "tsep2",      1, 0, 23 },
				{ "tsep3",      1, 0, 24 },
				{ "tsep4",      1, 0, 25 },
				{ "tsep5",      1, 0, 26 },
				{ "tsep6",      1, 0, 27 },
				{ "tsep7",      1, 0, 28 },
				{ "tsep8",      1, 0, 29 },
				{ "tsep9",      1, 0, 30 },

				{ "winShape",   1, 0, 31 },
				{ "winShape1",  1, 0, 32 },
				{ "winShape2",  1, 0, 33 },
				{ "winShape3",  1, 0, 34 },
				{ "winShape4",  1, 0, 35 },
				{ "winShape5",  1, 0, 36 },
				{ "winShape6",  1, 0, 37 },
				{ "winShape7",  1, 0, 38 },
				{ "winShape8",  1, 0, 39 },
				{ "winShape9",  1, 0, 40 },

				{ "tfra",       1, 0, 41 },

				{ "ljustifiedmean", 1, 0, 42 },
				{ "rjustifiedmean", 1, 0, 43 },
				{ "centeredmean",   1, 0, 44 },
				{ "forgetting",     1, 0, 45 },

				{ "stdout",         0, 0, 46 },

				{ 0, 0, 0, 0 }
			};

		optChar = getopt_long_only( argc, argv, "", long_options, &option_index );
		if ( optChar == -1 )
		{
			break;
		}

		switch( optChar )
		{
		case  1: /* tint */
			tint[0] = str2dbl( optarg );
			break;
		case  2: /* tint(1) */
			tint[1] = str2dbl( optarg );
			break;
		case  3: /* tint(2) */
			tint[2] = str2dbl( optarg );
			break;
		case  4: /* tint(3) */
			tint[3] = str2dbl( optarg );
			break;
		case  5: /* tint(4) */
			tint[4] = str2dbl( optarg );
			break;
		case  6: /* tint(5) */
			tint[5] = str2dbl( optarg );
			break;
		case  7: /* tint(6) */
			tint[6] = str2dbl( optarg );
			break;
		case  8: /* tint(7) */
			tint[7] = str2dbl( optarg );
			break;
		case  9: /* tint(8) */
			tint[8] = str2dbl( optarg );
			break;
		case 10: /* tint(9) */
			tint[9] = str2dbl( optarg );
			break;

		case 11: /* text */
			text[0] = str2dbl( optarg );
			break;
		case 12: /* text(1) */
			text[1] = str2dbl( optarg );
			break;
		case 13: /* text(2) */
			text[2] = str2dbl( optarg );
			break;
		case 14: /* text(3) */
			text[3] = str2dbl( optarg );
			break;
		case 15: /* text(4) */
			text[4] = str2dbl( optarg );
			break;
		case 16: /* text(5) */
			text[5] = str2dbl( optarg );
			break;
		case 17: /* text(6) */
			text[6] = str2dbl( optarg );
			break;
		case 18: /* text(7) */
			text[7] = str2dbl( optarg );
			break;
		case 19: /* text(8) */
			text[8] = str2dbl( optarg );
			break;
		case 20: /* text(9) */
			text[9] = str2dbl( optarg );
			break;

		case 21: /* tsep */
			tsep[0] = str2dbl( optarg );
			break;
		case 22: /* tsep(1) */
			tsep[1] = str2dbl( optarg );
			break;
		case 23: /* tsep(2) */
			tsep[2] = str2dbl( optarg );
			break;
		case 24: /* tsep(3) */
			tsep[3] = str2dbl( optarg );
			break;
		case 25: /* tsep(4) */
			tsep[4] = str2dbl( optarg );
			break;
		case 26: /* tsep(5) */
			tsep[5] = str2dbl( optarg );
			break;
		case 27: /* tsep(6) */
			tsep[6] = str2dbl( optarg );
			break;
		case 28: /* tsep(7) */
			tsep[7] = str2dbl( optarg );
			break;
		case 29: /* tsep(8) */
			tsep[8] = str2dbl( optarg );
			break;
		case 30: /* tsep(9) */
			tsep[9] = str2dbl( optarg );
			break;

		case 31: /* winShape */
			winShape[0] = str2winShape( optarg );
			break;
		case 32: /* winShape(1) */
			winShape[1] = str2winShape( optarg );
			break;
		case 33: /* winShape(2) */
			winShape[2] = str2winShape( optarg );
			break;
		case 34: /* winShape(3) */
			winShape[3] = str2winShape( optarg );
			break;
		case 35: /* winShape(4) */
			winShape[4] = str2winShape( optarg );
			break;
		case 36: /* winShape(5) */
			winShape[5] = str2winShape( optarg );
			break;
		case 37: /* winShape(6) */
			winShape[6] = str2winShape( optarg );
			break;
		case 38: /* winShape(7) */
			winShape[7] = str2winShape( optarg );
			break;
		case 39: /* winShape(8) */
			winShape[8] = str2winShape( optarg );
			break;
		case 40: /* winShape(9) */
			winShape[9] = str2winShape( optarg );
			break;

		case 41: /* tfra */
			tfra = str2dbl( optarg );
			break;

		case 42: /* ljustifiedmean */
			lJustifiedMean = str2postProcessMethod
				(
					lJustifiedMean,
					rJustifiedMean,
					centeredMean,
					forgettingAlpha,
					"ljustifiedmean",
					optarg
				);
			postProcessFlag = 1;
			break;

		case 43: /* rjustifiedmean */
			rJustifiedMean = str2postProcessMethod
				(
					lJustifiedMean,
					rJustifiedMean,
					centeredMean,
					forgettingAlpha,
					"rjustifiedmean",
					optarg
				);
			postProcessFlag = 1;
			break;

		case 44: /* centeredmean */
			centeredMean = str2postProcessMethod
				(
					lJustifiedMean,
					rJustifiedMean,
					centeredMean,
					forgettingAlpha,
					"centeredmean",
					optarg
				);
			postProcessFlag = 1;
			break;

		case 45: /* forgetting */
			forgettingAlpha = str2postProcessMethod
				(
					lJustifiedMean,
					rJustifiedMean,
					centeredMean,
					forgettingAlpha,
					"forgetting",
					optarg
				);
			postProcessFlag = 1;
			break;

		case 46: /* stdout */
			stdoutFlag = 1;
			break;

		case '?':
		default:
			PRINTUSAGE();
			fprintf( stderr, "%s: unknown argument \"%s\"\n", progName, optarg );
			exit( -1 );
		}
	}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Validate policies.
	 */

	{
		int  nPolicies_tint;
		int  nPolicies_text;
		int  nPolicies_tsep;
		int  nPolicies_winShape;

		nPolicies_tint = policyCOUNTDBL( "tint", tint, default_tint );
		nPolicies_text = policyCOUNTDBL( "text", text, default_text );
		nPolicies_tsep = policyCOUNTDBL( "tsep", tsep, default_tsep );
		nPolicies_winShape = policyCOUNTWINSHAPE( winShape, default_winShape );

		nPolicies = policyMERGECOUNTS
			(
				nPolicies_tint,
				nPolicies_text,
				nPolicies_tsep,
				nPolicies_winShape
			);

		if ( nPolicies > 1 )
		{
			if ( nPolicies_tint == 1 )
			{
				for ( policyIdx = 1; policyIdx < nPolicies; policyIdx ++ )
				{
					tint[ policyIdx ] = tint[ 0 ];
				}
			}

			if ( nPolicies_text == 1 )
			{
				for ( policyIdx = 1; policyIdx < nPolicies; policyIdx ++ )
				{
					text[ policyIdx ] = text[ 0 ];
				}
			}

			if ( nPolicies_tsep == 1 )
			{
				for ( policyIdx = 1; policyIdx < nPolicies; policyIdx ++ )
				{
					tsep[ policyIdx ] = tsep[ 0 ];
				}
			} 

			if ( nPolicies_winShape == 1 )
			{
				for ( policyIdx = 1; policyIdx < nPolicies; policyIdx ++ )
				{
					winShape[ policyIdx ] = winShape[ 0 ];
				}
			}
		}
	}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Create trappings for subsequent MSF computation.
	 */

	for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
	{
		msfCREATE
			(
				tint[ policyIdx ],
				text[ policyIdx ],
				tsep[ policyIdx ],
				fs,
				winShape[ policyIdx ],
				&(Tsize[ policyIdx ]),
				&(msfComputerPtrTab[ policyIdx ])
			);

		assert( (Tsize[ policyIdx ] % 2) == 0 );
		halfTsize[ policyIdx ] = Tsize[ policyIdx ] / 2;
	}

	printf( "-------[ POLICIES SUMMARY ]-------\n" );
	msfPRINTTAB( msfComputerPtrTab, nPolicies );

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Get a list of input files.
	 */

	printf( "-------[ FILES TO PROCESS ]-------\n" );

	inLocation = (char *) NULL;
	nInFiles = 0;

	if ( ( argc - optind ) > 0 )
	{
		inLocation = argv[ optind ++ ];
		filenameGETLIST
			(
				inLocation,
				(char *) NULL,
				&inFileNameTab,
				&inPathNameTab,
				&nInFiles
			);
	}

	if ( nInFiles <= 0 )
	{
		printf( "\t(no files to process, exiting.)\n" );
	}
	else
	{

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

		/*
		 * Get a single output file-or-directory name.
		 */

		char *  outLocation;
		char *  featDirName;
		char *  featFileName;
		int     inFileIdx;

		outLocation = (char *) NULL;
		featDirName = (char *) NULL;
		featFileName = (char *) NULL;
		if ( ( argc - optind ) > 0 )
		{
			struct stat  statBuf;
			int          statRetVal;

			if ( stdoutFlag == 1 )
			{
				PRINTUSAGE();
				fprintf
					(
						stderr,
						"%s: both output location and stdout specified\n",
						progName
					);
				exit( -1 );
			}

			outLocation = argv[ optind ++ ];
			statRetVal = stat( outLocation, &statBuf );
			if ( statRetVal == 0 )
			{
				if ( S_ISDIR( statBuf.st_mode ) )
				{
					/* assume outLocation is a dirname */

					featDirName = outLocation;
				}
				else
				{
					/* assume outLocation is a filename */

					PRINTUSAGE();
					fprintf
						(
							stderr,
							"%s: %s already exists\n",
							progName,
							outLocation
						);
					exit( -1 );
				}
			}
			else
			{
				/* assume outLocation does not exist */

				if ( nInFiles == 1 )
				{
					featDirName = ".";
					featFileName = outLocation;
				}
				else /* if ( nInFiles > 1 ) */
				{
					PRINTUSAGE();
					fprintf
						(
							stderr,
							"%s: output dirname missing\n",
							progName
						);
					exit( -1 );
				}
			}
		}
		else if ( stdoutFlag == 1 )
		{
			/* output location is standart output */
		}
		else
		{
			/* no location provided */

			if ( nInFiles == 1 )
			{
				featDirName = ".";
			}
			else /* if ( nInFiles > 1 ) */
			{
				PRINTUSAGE();
				fprintf
					(
						stderr,
						"%s: output dirname missing\n",
						progName
					);
				exit( -1 );
			}
		}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

		for ( inFileIdx = 0; inFileIdx < nInFiles; inFileIdx ++ )
		{ 
			char *       audioPathName;
			SF_INFO      audioInfo;
			SNDFILE *    audioFPtr;
			int          nAudio;
			double       audioDuration;
			double *     audio;
			FILE *       featFPtr;
			StopWatch *  stopwatchPtr;
			double       centerSec;
			int          nFrames;
			double **    featTabTab;
			double *     centerSecTab;
			int          frameIdx;
			double       stopwatchDuration;

			audioPathName = inPathNameTab[ inFileIdx ];

			audioInfo.format = 0;
			audioFPtr = sf_open( audioPathName, SFM_READ, &audioInfo );
			if ( audioFPtr == (SNDFILE *) NULL )
			{
	
				fprintf
					(
						stderr,
						"%s COULD NOT BE OPENED, skipping.\n",
						audioPathName
					);

				printf
					(
						"%s COULD NOT BE OPENED, skipping.\n",
						audioPathName
					);
				continue;
			}

			nAudio = (int) (audioInfo.frames);
			if ( audioInfo.samplerate != (int) fs )
			{
				printf
					(
						"%s IS NOT SAMPLED AT %dHz, skipping.\n",
						audioPathName,
						(int) fs
					);
				continue;
			}
			else if ( audioInfo.channels != 1 )
			{
				printf
					(
						"%s HAS MORE THAN 1 CHANNEL, skipping.\n",
						audioPathName
					);
				continue;
			}
			else if ( audioInfo.sections != 1 )
			{
				printf
					(
						"%s HAS MORE THAN 1 SECTION, skipping.\n",
						audioPathName
					);
				continue;
			}

			audioDuration = ((double) nAudio) / fs;

			audio = (double *) malloc( nAudio * sizeof( double ) );
/* DIE IF */		assert( audio != (double *) NULL ); 
/* DIE IF */		assert( sf_read_double( audioFPtr, audio, nAudio ) == nAudio );
			sf_close( audioFPtr );

			if ( stdoutFlag == 0 ) 
			{
				printf( "%s   %6.2f%%", audioPathName, 0.0 );
				fflush( stdout );
			}

			/*
			 * Loop over frames for the audio.
			 */

			if ( postProcessFlag )
			{
				for
					(
						centerSec = 0.0, nFrames = 0;
						centerSec < audioDuration;
						centerSec += tfra, nFrames ++
					);

				featTabTab = (double **) malloc( nFrames * sizeof( double * ) );
				assert( featTabTab != (double **) NULL );

				for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
				{
					double *  featTab;

					featTab = (double *) malloc( nPolicies * sizeof( double ) );
					assert( featTab != (double *) NULL );

					featTabTab[ frameIdx ] = featTab;
				}

				centerSecTab = (double *) malloc( nFrames * sizeof( double ) );
				assert( centerSecTab != (double *) NULL );
			}
			else
			{
				nFrames = -1;
				featTabTab = (double **) NULL;
			}

			if ( stdoutFlag == 0 )
			{
				char  featsPathName[ 1024 ];

				if ( featFileName == (char *) NULL )
				{
					sprintf
						(
							featsPathName,
							"%s/%s.txt",
							featDirName,
							inFileNameTab[ inFileIdx ]
						);
				}
				else
				{
					sprintf
						(
							featsPathName,
							"%s/%s",
							featDirName,
							featFileName
						);
				}

				featFPtr = fopen( featsPathName, "w" );
/* DIE IF */			assert( featFPtr != (FILE *) NULL );
			}
			else
			{
				featFPtr = stdout;
			}

			stopwatchPtr = stopwatchSTART();
			for
				(
					centerSec = 0.0, frameIdx = 0;
					centerSec < audioDuration;
					centerSec += tfra, frameIdx ++
				)
			{
				int     centerIdx;

				centerIdx = (int) ( centerSec * fs );
				if ( ( ( centerSec * fs ) - (double) centerIdx ) > 0.5 )
				{
					centerIdx ++;
				}

				if ( postProcessFlag )
				{
					centerSecTab[ frameIdx ] = centerSec;
				}
				else
				{
					fprintf( featFPtr, "%d %g", frameIdx, centerSec );
				}

				for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
				{
					int     begIdx;
					int     endIdx;
					double  msf;

					begIdx = centerIdx - halfTsize[ policyIdx ];
					endIdx = centerIdx + halfTsize[ policyIdx ];

					if ( centerIdx >= nAudio )
					{
/* DIE IF (BIZARRELY) */			assert( 0 );
					}
					else if ( ( begIdx < 0 ) || ( endIdx >= nAudio ) )
					{
						msf = NAN;
					}
					else
					{
						int  nFrameAudio;

						nFrameAudio = endIdx - begIdx;

						msfCOMPUTE
							(
								msfComputerPtrTab[ policyIdx ],
								audio,
								begIdx,
								nFrameAudio,
								&msf
							);
					}

					if ( postProcessFlag )
					{
						featTabTab[ frameIdx ][ policyIdx ] = msf;
					}
					else
					{
						fprintf( featFPtr, " %g", msf );
					}
				}

				if ( ! postProcessFlag )
				{
					fprintf( featFPtr, "\n" );
				}

				if ( stdoutFlag == 0 )
				{
					if ( postProcessFlag )
					{
						printf( "\b\b\b\b\b\b\b%6.2f%%", 100.0 * (double) centerIdx / ( 2.0 * nAudio ) );
					}
					else
					{
						printf( "\b\b\b\b\b\b\b%6.2f%%", 100.0 * (double) centerIdx / nAudio );
					}
					fflush( stdout );
				}
			}

			stopwatchDuration = stopwatchSTOP( stopwatchPtr );

			if ( postProcessFlag )
			{
				double **  ppFeatTabTab;

				ppFeatTabTab = (double **) malloc( nFrames * sizeof( double * ) );
				assert( ppFeatTabTab != (double **) NULL );

				for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
				{
					double *  ppFeatTab;

					ppFeatTab = (double *) malloc( nPolicies * sizeof( double ) );
					assert( ppFeatTab != (double *) NULL );

					ppFeatTabTab[ frameIdx ] = ppFeatTab;
				}

				if ( ! isnan( lJustifiedMean ) )
				{
					int  T;

					T = (int) round( lJustifiedMean / tfra );

					for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
					{
						for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
						{
							int     begIdx;
							int     endIdx;
							double  value;

							begIdx = frameIdx;
							endIdx = begIdx + T;

							if ( ( begIdx < 0 ) || ( endIdx >= nFrames ) )
							{
								value = NAN;
							}
							else
							{
								int  idx;

								value = 0.0;
								for ( idx = begIdx; idx < endIdx; idx ++ )
								{
									value += featTabTab[ idx ][ policyIdx ];
								}

								value /= (1.0 + T);
							} 

							ppFeatTabTab[ frameIdx ][ policyIdx ] = value;
						}
					}
				}
				else if ( ! isnan( rJustifiedMean ) )
				{
					int  T;

					T = (int) round( rJustifiedMean / tfra );

					for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
					{
						for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
						{
							int     begIdx;
							int     endIdx;
							double  value;

							begIdx = frameIdx - T;
							endIdx = frameIdx;

							if ( ( begIdx < 0 ) || ( endIdx >= nFrames ) )
							{
								value = NAN;
							}
							else
							{
								int  idx;

								value = 0.0;
								for ( idx = begIdx; idx < endIdx; idx ++ )
								{
									value += featTabTab[ idx ][ policyIdx ];
								}

								value /= (1.0 + T);
							}

							ppFeatTabTab[ frameIdx ][ policyIdx ] = value;
						}
					}
				}
				else if ( ! isnan( centeredMean ) )
				{
					int  halfT;

					halfT = (int) round( ( centeredMean / tfra ) / 2.0 );

					for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
					{
						for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
						{
							int     begIdx;
							int     endIdx;
							double  value;

							begIdx = frameIdx - halfT;
							endIdx = frameIdx + halfT; 

							if ( ( begIdx < 0 ) || ( endIdx >= nFrames ) )
							{
								value = NAN;
							}
							else
							{
								int  idx;

								value = 0.0;
								for ( idx = begIdx; idx < endIdx; idx ++ )
								{
									value += featTabTab[ idx ][ policyIdx ];
								}

								value /= (1.0 + halfT + halfT);
							}

							ppFeatTabTab[ frameIdx ][ policyIdx ] = value;
						}
					}
				}
				else if ( ! isnan( forgettingAlpha ) )
				{
					for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
					{
						double  prevPpValue;

						prevPpValue = NAN;
						for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
						{
							double  currValue;
							double  currPpValue;

							currValue = featTabTab[ frameIdx ][ policyIdx ];

							if ( isnan( prevPpValue) && isnan( currValue ) )
							{
								currPpValue = NAN;
							}
							else if ( isnan( prevPpValue ) )
							{
								currPpValue = currValue;
							}
							else if ( isnan( currValue ) )
							{
								currPpValue = prevPpValue;
							}
							else
							{
								currPpValue = forgettingAlpha * prevPpValue
									+ ( 1.0 - forgettingAlpha ) * currValue;
							}

							ppFeatTabTab[ frameIdx ][ policyIdx ] = currPpValue;
							prevPpValue = currPpValue;
						}
					}
				}

				for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
				{
					fprintf( featFPtr, "%d %g", frameIdx, centerSecTab[ frameIdx ] );

					for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
					{
						fprintf( featFPtr, " %g", ppFeatTabTab[ frameIdx ][ policyIdx ] );
					}
					fprintf( featFPtr, "\n" );

					if ( stdoutFlag == 0 )
					{
						printf( "\b\b\b\b\b\b\b%6.2f%%", 50.0 + 100.0 * (double) frameIdx / ( 2.0 * nFrames ) );
						fflush( stdout );
					}
				}

				for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
				{
					free( (void  *) (ppFeatTabTab[ frameIdx ]) );
				}
				free( (void *) ppFeatTabTab );
			}

			if ( stdoutFlag == 0 )
			{
				printf( "\b\b\b\b\b\b\b100.00%%  " );
				printf
					(
						"(audio %.6gs | dsp+io %.6gs :: rtf = %g x realtime)\n",
						audioDuration,
						stopwatchDuration,
						( stopwatchDuration / audioDuration )
					);
			}

			if ( stdoutFlag == 0 )
			{
				fclose( featFPtr );
			}

			free( (void *) audio );

			if ( postProcessFlag )
			{
				for ( frameIdx = 0; frameIdx < nFrames; frameIdx ++ )
				{
					free( (void *) (featTabTab[ frameIdx ]) );
				}
				free( (void *) featTabTab );
				free( (void *) centerSecTab );
			}
		}

		filenameFREELIST( inFileNameTab, inPathNameTab, nInFiles );
	}

	for ( policyIdx = 0; policyIdx < nPolicies; policyIdx ++ )
	{
		msfDESTROY( &(msfComputerPtrTab[ policyIdx ]) );
	}

	return 0;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

