/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2009-2011, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of KTH Speech Music and Hearing nor the names of its contributors may be
      used to endorse or promote products derived from this software without specific prior written
      permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <stdio.h>
#include <string.h>
#define _ISOC99_SOURCE
#include <math.h>
#include <float.h>
#include "windowpair.h"
#include "mutils.h"
#include "mel.h"
#include "msf.h"

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void msfCREATE
	(
		const double    tint,
		const double    text,
		const double    tsep,
		const double    fs,
		const WinShape  winShape,
		int *           TsizePtr,
		MsfComputer **  msfComputerPtrPtr
	)
{
	int              Tint;
	int              Text;
	int              Tsep;
	double *         hL;
	double *         hR;
	int              Nh;
	int              NhPow2;
	double *         fftL;
	double *         fftR;
	int              Nfft;
	double *         magL;
	double *         magR;
	int              Nmag;
	MelFilterBank *  melPtr;
	double *         melL;
	double *         melR;
	int              Nmel;
	MsfComputer *    msfComputerPtr;

	windowpairCREATE
		(
			tint,
			text,
			tsep,
			fs,
			winShape,
			&Tint,
			&Text,
			&Tsep,
			TsizePtr,
			&hL,
			&hR,
			&Nh,
			&NhPow2,
			&fftL,
			&fftR,
			&Nfft
		);

	Nmag = NhPow2 / 2;

	magL = (double *) malloc( Nmag * sizeof( double ) );
	assert( magL != (double *) NULL );

	magR = (double *) malloc( Nmag * sizeof( double ) );
	assert( magR != (double *) NULL );

        melALLOC( &melPtr, fs, Nmag );

	Nmel = melPtr->nFilt;

        melL = (double *) malloc( Nmel * sizeof( double ) );
        assert( melL != (double *) NULL );

        melR = (double *) malloc( Nmel * sizeof( double ) );
        assert( melR != (double *) NULL );

	msfComputerPtr = (MsfComputer *) malloc( sizeof( MsfComputer ) );
	assert( msfComputerPtr != (MsfComputer *) NULL );

	msfComputerPtr->tint = tint;
	msfComputerPtr->text = text;
	msfComputerPtr->tsep = tsep;
	msfComputerPtr->fs = fs;
	msfComputerPtr->winShape = winShape;
	msfComputerPtr->Tint = Tint;
	msfComputerPtr->Text = Text;
	msfComputerPtr->Tsep = Tsep;
	msfComputerPtr->hL = hL;
	msfComputerPtr->hR = hR;
	msfComputerPtr->Nh = Nh;
	msfComputerPtr->NhPow2 = NhPow2;
	msfComputerPtr->fftL = fftL;
	msfComputerPtr->fftR = fftR;
	msfComputerPtr->Nfft = Nfft;
	msfComputerPtr->magL = magL;
	msfComputerPtr->magR = magR;
	msfComputerPtr->Nmag = Nmag;
	msfComputerPtr->melPtr = melPtr;
	msfComputerPtr->melL = melL;
	msfComputerPtr->melR = melR;
	msfComputerPtr->Nmel = Nmel;

	*msfComputerPtrPtr = msfComputerPtr;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void msfDESTROY
	(
		MsfComputer **  msfComputerPtrPtr
	)
{
	windowpairDESTROY
		(
			&((*msfComputerPtrPtr)->hL),
			&((*msfComputerPtrPtr)->hR),
			&((*msfComputerPtrPtr)->fftL),
			&((*msfComputerPtrPtr)->fftR)
		);

	free( (void *) ((*msfComputerPtrPtr)->magL) );
	free( (void *) ((*msfComputerPtrPtr)->magR) );

	melFREE( &((*msfComputerPtrPtr)->melPtr) );

	free( (void *) ((*msfComputerPtrPtr)->melL) );
	free( (void *) ((*msfComputerPtrPtr)->melR) );

	free( (void *) (*msfComputerPtrPtr) );
	*msfComputerPtrPtr = (MsfComputer *) NULL;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void msfCOMPUTE
	(
		MsfComputer *   msfComputerPtr,
		const double *  audio,
		const int       begIdx,
		const int       nAudio,
		double *        melSpectralFluxPtr
	)
{
	int     allZero;
	int     i;
	double  negLogitMelCosDist;

	/*
	 * Check if all samples are zero (degenerate case).
	 */

	allZero = 1;
	for ( i = 0; i < nAudio; i ++ )
	{
		if ( *(audio + begIdx + i) != 0.0 )
		{
			allZero = 0;
			break;
		}
	}

	if ( allZero == 1 )
	{
		negLogitMelCosDist = NAN;
	}
	else
	{
		MelFilterBank *  melPtr;
		double *         melL;
		double *         melR;

		melPtr = msfComputerPtr->melPtr;
		melL = msfComputerPtr->melL;
		melR = msfComputerPtr->melR;

		/*
		 * Computer the Mel spectra for L and R windows.
		 */

		{
			double *  fftL;
			double *  fftR;
			double *  magL;
			double *  magR;
			int       Nmag;
			int       freqIdx;
			double    dummy1;
			double    dummy2;

			fftL = msfComputerPtr->fftL;
			fftR = msfComputerPtr->fftR;
			magL = msfComputerPtr->magL;
			magR = msfComputerPtr->magR;
			Nmag = msfComputerPtr->Nmag;

			/*
			                         audio frame to analyze:
 
			                    begIdx        center       endIdx
			|                   |===============|===============|                   |
			|                   |               |               |                   |
			|<- (NhPow2-Nh)/2 ->|<-   (Nh/2)  ->|<-   (Nh/2)  ->|<- (NhPow2-Nh)/2 ->|
			|                   |                               |                   |
			|                   |  (temporal support of audio)  |                   |
			|             (also : temporal non-zero support of windows)             |
			|                                                                       |
                        |<------------------------------ NhPow2 ------------------------------->|
			|                       (temporal support of FFT)                       |
			|             (also : temporal complete support of windows)             | 

			*/

			windowpairTWOFFT
				(
					audio,
					begIdx,
					nAudio,
					msfComputerPtr->hL,
					msfComputerPtr->hR,
					msfComputerPtr->Nh,
					msfComputerPtr->NhPow2,
					fftL,
					fftR,
					msfComputerPtr->Nfft
				);

			magL[ 0 ] = fabs( fftL[ 0 ] );
			magR[ 0 ] = fabs( fftR[ 0 ] );
			for ( freqIdx = 1; freqIdx < Nmag; freqIdx ++ )
			{
				int     freqIdxTimes2;
				double  re;
				double  im;

				#undef ENERGY

				freqIdxTimes2 = freqIdx + freqIdx;

				re = fftL[ freqIdxTimes2 ];
				im = fftL[ freqIdxTimes2 + 1 ];
				magL[ freqIdx ] =
				#ifdef ENERGY
					re * re + im * im;
				#else
					sqrt( re * re + im * im );
				#endif

				re = fftR[ freqIdxTimes2 ];
				im = fftR[ freqIdxTimes2 + 1 ];
				magR[ freqIdx ] =
				#ifdef ENERGY
					re * re + im * im;
				#else
					sqrt( re * re + im * im );
				#endif
			}

			melFILTER( magL, Nmag, melPtr, melL, &dummy1, &dummy2 );
			melFILTER( magR, Nmag, melPtr, melR, &dummy1, &dummy2 );
		}

		/*
		 * Compute the cosine distance between the L and R Mel spectra.
		 */

		{
			double  logSumMelLL;
			double  logSumMelLR;
			double  logSumMelRR;
			int     melIdx;
			double  logMelCosDist;

			{
				double  logLVal;
				double  logRVal;

				logLVal = log( melL[ 0 ] );
				logRVal = log( melR[ 0 ] );

				logSumMelLL = logLVal + logLVal;
				logSumMelLR = logLVal + logRVal;
				logSumMelRR = logRVal + logRVal;
			}

			for ( melIdx = 1; melIdx < melPtr->nFilt; melIdx ++ )
			{
				double  logLVal;
				double  logRVal;

				logLVal = log( melL[ melIdx ] );
				logRVal = log( melR[ melIdx ] );

				logSumMelLL += log( 1.0 + exp( logLVal + logLVal - logSumMelLL ) );
				logSumMelLR += log( 1.0 + exp( logLVal + logRVal - logSumMelLR ) );
				logSumMelRR += log( 1.0 + exp( logRVal + logRVal - logSumMelRR ) );
			}

			logMelCosDist = logSumMelLR - 0.5 * ( logSumMelLL + logSumMelRR );
			negLogitMelCosDist = - ( logMelCosDist - log( 1.0 - exp( logMelCosDist ) ) );
		}
	}

	*melSpectralFluxPtr = negLogitMelCosDist;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void msfPRINT
	(
		const MsfComputer *  msfComputerPtr
	)
{
        printf( "tint           : %.3g\n", msfComputerPtr->tint );
        printf( "text           : %.3g\n", msfComputerPtr->text );
        printf( "tsep           : %.3g\n", msfComputerPtr->tsep );
        printf( "fs             : %g\n", msfComputerPtr->fs );
        printf( "Tint           : %d\n", msfComputerPtr->Tint );
        printf( "Text           : %d\n", msfComputerPtr->Text );
        printf( "Tsep           : %d\n", msfComputerPtr->Tsep );
        printf( "winShape       : %d\n", msfComputerPtr->winShape );
        printf( "Nh             : %d\n", msfComputerPtr->Nh );
        printf( "NhPow2         : %d\n", msfComputerPtr->NhPow2 );
        printf( "Nfft           : %d\n", msfComputerPtr->Nfft );
	printf( "Nmag           : %d\n", msfComputerPtr->Nmag );
	printf( "Nmel           : %d\n", msfComputerPtr->Nmel );

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void msfPRINTTAB
	(
		MsfComputer **  msfComputerPtrTab,
		const int       n
	)
{
	int  i;

	printf( "Parameter       " );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t#%d", (i + 1) );
	}
	printf( "\n" );

	printf( "tint           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%.3g", msfComputerPtrTab[ i ]->tint );
	} 
	printf( "\n" );

	printf( "text           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%.3g", msfComputerPtrTab[ i ]->text );
	} 
	printf( "\n" );

	printf( "tsep           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%.3g", msfComputerPtrTab[ i ]->tsep );
	} 
	printf( "\n" );

	printf( "fs             :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%g", msfComputerPtrTab[ i ]->fs );
	} 
	printf( "\n" );

	printf( "Tint           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Tint );
	} 
	printf( "\n" );

	printf( "Text           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Text );
	} 
	printf( "\n" );

	printf( "Tsep           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Tsep );
	} 
	printf( "\n" );

	printf( "winShape       :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->winShape );
	} 
	printf( "\n" );

	printf( "Nh             :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Nh );
	} 
	printf( "\n" );

	printf( "NhPow2         :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->NhPow2 );
	} 
	printf( "\n" );

	printf( "Nfft           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Nfft );
	} 
	printf( "\n" );

	printf( "Nmag           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Nmag );
	} 
	printf( "\n" );

	printf( "Nmel           :" );
	for ( i = 0; i < n; i ++ )
	{
		printf( "\t%d", msfComputerPtrTab[ i ]->Nmel );
	} 
	printf( "\n" );

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

