/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2009-2011, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of KTH Speech Music and Hearing nor the names of its contributors may be
      used to endorse or promote products derived from this software without specific prior written
      permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <assert.h>
#include <stdlib.h>
#define _ISOC99_SOURCE
#include <math.h>
#include <string.h>
#include <stdio.h>
#include "mel.h"

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

static double  freq2melFreq( double freq )
{
	/* ref[1] Eq 2.6 */

	return ( 1125.0 * log( 1.0 + freq/700.0 ) );
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static double  melFreq2freq( double melFreq )
{
	/* ref[1] Eq 6.143 */

	return ( 700.0 * ( exp( melFreq / 1125.0 ) - 1.0 ) ); 
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static int  freq2freqIdx
	(
		const double  freq,
		const double  samplingFreq,
		const int     nFreq
	)
{
	return ( (int)ceil( (2.0 * (nFreq - 1) * freq) / samplingFreq ) );
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static double  freqIdx2freq
	(
		const int     freqIdx,
		const double  samplingFreq,
		const int     nFreq
	)
{
	return ( (samplingFreq * freqIdx) / (2.0 * (nFreq - 1)) );
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void melALLOC
	(
		MelFilterBank **  melFBPtrPtr,
		const double      samplingFreq,
		const int         nFreq
	)
{
	int     nFilt;
	int     filtIdx;
	double  minFreq;
	double  maxFreq;
	double  minMelFreq;
	double  maxMelFreq;
	double  halfBaseMel;

	assert( samplingFreq == 16000.0 );  /* 16 kHz support only */

	/*
	 * Assuming above assertion.
	 */

	nFilt =     40;
	minFreq =  130.0;
	maxFreq = 6800.0;

	*melFBPtrPtr = (MelFilterBank *) malloc( sizeof( MelFilterBank ) );
	assert( *melFBPtrPtr != (MelFilterBank *) NULL );

	(*melFBPtrPtr)->nFilt = nFilt;
	(*melFBPtrPtr)->tab = (MelFilter *) malloc( nFilt * sizeof( MelFilter ) );
	assert( (*melFBPtrPtr)->tab != (MelFilter *) NULL );

	minMelFreq = freq2melFreq( minFreq );
	maxMelFreq = freq2melFreq( maxFreq );
	halfBaseMel = (maxMelFreq - minMelFreq) / (nFilt + 1);

	for ( filtIdx = 0; filtIdx < nFilt; filtIdx ++ )
	{
		double  filtMinMelFreq;
		double  filtMaxMelFreq;
		double  filtCenMelFreq;
		double  filtMinFreq;
		double  filtCenFreq;
		double  filtMaxFreq;
		int     filtMinFreqIdx;
		int     filtCenFreqIdx;
		int     filtMaxFreqIdx;
		double  weight;
		int     freqIdx;

		filtMinMelFreq = minMelFreq + filtIdx * halfBaseMel;
		filtCenMelFreq = filtMinMelFreq + halfBaseMel;
		filtMaxMelFreq = filtCenMelFreq + halfBaseMel;

		filtMinFreq = melFreq2freq( filtMinMelFreq );
		filtCenFreq = melFreq2freq( filtCenMelFreq );
		filtMaxFreq = melFreq2freq( filtMaxMelFreq );

		filtMinFreqIdx = freq2freqIdx( filtMinFreq, samplingFreq, nFreq );
		filtCenFreqIdx = freq2freqIdx( filtCenFreq, samplingFreq, nFreq );
		filtMaxFreqIdx = freq2freqIdx( filtMaxFreq, samplingFreq, nFreq );

		(*melFBPtrPtr)->tab[ filtIdx ].minFreqIdx = filtMinFreqIdx;
		(*melFBPtrPtr)->tab[ filtIdx ].maxFreqIdx = filtMaxFreqIdx;
		(*melFBPtrPtr)->tab[ filtIdx ].nCoeff = nFreq;
		(*melFBPtrPtr)->tab[ filtIdx ].coeffTab
			= (double *) malloc( nFreq * sizeof( double ) );
		assert( (*melFBPtrPtr)->tab[ filtIdx ].coeffTab != (double *) NULL );
		memset
			(
				(void *)((*melFBPtrPtr)->tab[ filtIdx ].coeffTab),
				0,
				nFreq * sizeof( double )
			);
				
		/* assume max of 1 at filtCenFreq */

		weight = 0.0;
		for ( freqIdx = filtMinFreqIdx; freqIdx < filtCenFreqIdx; freqIdx ++ )
		{
			double  freq;
			double  coeff;

			freq = freqIdx2freq( freqIdx, samplingFreq, nFreq );
			coeff = (freq - filtMinFreq) / (filtCenFreq - filtMinFreq);

			(*melFBPtrPtr)->tab[ filtIdx ].coeffTab[ freqIdx ] = coeff;
			weight += coeff;
		}

		for ( freqIdx = filtCenFreqIdx; freqIdx < filtMaxFreqIdx; freqIdx ++ )
		{
			double  freq;
			double  coeff;

			freq = freqIdx2freq( freqIdx, samplingFreq, nFreq );
			coeff = (filtMaxFreq - freq) / ( filtMaxFreq - filtCenFreq );

			(*melFBPtrPtr)->tab[ filtIdx ].coeffTab[ freqIdx ] = coeff;
			weight += coeff;
		}

		assert( weight > 0.0 );
		for ( freqIdx = filtMinFreqIdx; freqIdx < filtMaxFreqIdx; freqIdx ++ )
		{
			(*melFBPtrPtr)->tab[ filtIdx ].coeffTab[ freqIdx ] /= weight;
		}
	}

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void melFREE
	(
		MelFilterBank **  melFBPtrPtr
	)
{
	int  filtIdx;

	for ( filtIdx = 0; filtIdx < (*melFBPtrPtr)->nFilt; filtIdx ++ )
	{
		free( (void *) ((*melFBPtrPtr)->tab[ filtIdx ].coeffTab) );
	}

	free( (void *) ((*melFBPtrPtr)->tab) );

	free( (void *) (*melFBPtrPtr) );

	*melFBPtrPtr = (MelFilterBank *) NULL;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void melPRINT
	(
		const MelFilterBank *  melFBPtr
	)
{
	int  filtIdx;

	for ( filtIdx = 0; filtIdx < melFBPtr->nFilt; filtIdx ++ )
	{
		MelFilter *  melFilterPtr;
		int          coeffIdx;

		melFilterPtr = &(melFBPtr->tab[ filtIdx ]);

		for ( coeffIdx = 1; coeffIdx < melFilterPtr->nCoeff; coeffIdx ++ )
		{
			printf( " %g", melFilterPtr->coeffTab[ coeffIdx ] );
		}
		printf( "\n" );
	}

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void melFILTER
	(
		const double *         spectrumPtr,
		const int              nFreq,
		const MelFilterBank *  melFBPtr,
		double *               melSpectrumPtr,
		double *               spectrumSumPtr,
		double *               spectrumSqrtSumSqPtr
	)
{
	int     filtIdx;
	double  spectrumSumSq;

	*spectrumSumPtr = 0.0;
	spectrumSumSq = 0.0;
	for ( filtIdx = 0; filtIdx < melFBPtr->nFilt; filtIdx ++ )
	{
		MelFilter *  melFilterPtr;
		int          coeffIdx;
		double       melSpectrumVal;

		melFilterPtr = &(melFBPtr->tab[ filtIdx ]);

		melSpectrumVal = 0.0;
		for
			(
				coeffIdx = melFilterPtr->minFreqIdx;
				coeffIdx < melFilterPtr->maxFreqIdx;
				coeffIdx ++
			)
		{
			melSpectrumVal += (melFilterPtr->coeffTab[ coeffIdx ]
				* spectrumPtr[ coeffIdx ]);
		}

		melSpectrumPtr[ filtIdx ] = melSpectrumVal;
		(*spectrumSumPtr) += melSpectrumVal;
		spectrumSumSq += (melSpectrumVal * melSpectrumVal);
	}

	*spectrumSqrtSumSqPtr = sqrt( spectrumSumSq );
 
	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

