/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2007-2009, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Carnegie Mellon University nor the names of its contributors may be used
      to endorse or promote products derived from this software without specific prior written
      permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

18.08.2008 Kornel Laskowski

	Remove dependency on NR/C code.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include "mutils.h"
#include "windowpair.h"

#define PI 3.14159265359
#define DOPREEMPHASIS

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

/*
 * CONSTRUCTOR
 */

void windowpairCREATE
	(
		const double    tint,
		const double    text,
		const double    tsep,
		const double    fs,
		const WinShape  winShape,
		int *           TintPtr,
		int *           TextPtr,
		int *           TsepPtr,
		int *           TsizePtr,
		double **       hLPtr,
		double **       hRPtr,
		int *           NhPtr,
		int *           NhPow2Ptr,
		double **       auxLPtr,
		double **       auxRPtr,
		fftw_plan *     planLPtr,
		fftw_plan *     planRPtr,
		double **       fftLPtr,
		double **       fftRPtr,
		int *           NfftPtr
	)
{
	int        Tint;
	int        Text;
	int        Tsep;
	int        Nh;
	int        NhPow2;
	double     bint;
	double     mint;
	double     bext;
	double     mext;
	double *   hL;
	double *   hR;
	int        TsepHalf;
	int        i;
	int        Nfft;
	double *   auxL;
	double *   auxR;
	fftw_plan  planL;
	fftw_plan  planR;
	double *   fftL;
	double *   fftR;

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Compute window-size parameters in number of samples.
	 */

	Tint = (int) ( tint * fs );
	if ( ( ( tint * fs ) - (double) Tint ) > 0.5 )
	{
		Tint ++;
	}
	assert( Tint > 0 );

	Text = (int) ( text * fs );
	if ( ( ( text * fs ) - (double) Text ) > 0.5 )
	{
		Text ++;
	}
	assert( Text > 0 );

	Tsep = (int) ( tsep * fs );
	if ( ( ( tsep * fs ) - (double) Tsep ) > 0.5 )
	{
		Tsep ++;
	}
	assert( Tsep > 0 );
	assert( ( Tsep % 2 ) == 0 ); 

	assert( Tint <= (Tsep + Text) ); /* can't remember why */

	Nh = Text + Tsep + Text;

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Assign window-shape curve-parameters based on winShape.
	 */

        switch( winShape )
        {
        case WinShape_HAMMING_HAMMING:
                bint = 0.54;
                mint = 0.46;
                bext = 0.54;
                mext = 0.46;
                break;

        case WinShape_HANN_HANN:
                bint = 0.5;
                mint = 0.5;
                bext = 0.5;
                mext = 0.5;
                break;

        case WinShape_HAMMING_HANN:
                bint = 0.5;
                mint = 0.5;
                bext = 0.54;
                mext = 0.46;
                break;

        default:
                assert( 0 );
                break;
        }

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Allocate space for left and right windows.
	 */

	hL = (double *) malloc( Nh * sizeof( double ) );
	assert( hL != (double *) NULL );

	hR = (double *) malloc( Nh * sizeof( double ) );
	assert( hR != (double *) NULL );

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Construct left and right windows based on window-size and window-shape parameters.
	 */

        TsepHalf = Tsep / 2;

	for ( i = 0; i < Nh; i ++ )
	{
		int  t;

		/*
		 * Windows are defined with t=0 in the center; the window arrays will be 0-offset
		 *  for subsequent ease of use. Hence "i" (the 0-offset index) and "t" (the
		 *  variable in the formulas in the paper).
		 */

		t = i - ( Text + TsepHalf );

		/*
		 * Compute hL[0:Nh-1] and hR[0:Nh-1] separately. Not efficient, but clear.
		 */

		if ( t < - ( Text + TsepHalf ) )
		{
			hL[ i ] = 0.0;
		}
		else if ( t < - TsepHalf )
		{
			hL[ i ] = bext + mext * cos( PI * (t + TsepHalf) / Text );
			
		}
		else if ( t <= ( - TsepHalf + Tint ) )
		{
			hL[ i ] = bint + mint * cos( PI * (t + TsepHalf) / Tint );
		}
		else
		{
			hL[ i ] = 0.0;
		}

		if ( t < ( + TsepHalf - Tint ) )
		{
			hR[ i ] = 0.0;
		}
		else if ( t < +TsepHalf )
		{
			hR[ i ] = bint + mint * cos( PI * (t - TsepHalf) / Tint );
		}
		else if ( t < + ( TsepHalf + Text ) )
		{
			hR[ i ] = bext + mext * cos( PI * (t - TsepHalf) / Text );
		}
		else
		{
			hR[ i ] = 0.0;
		}
	}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	/*
	 * Allocate space for the FFTs of the left and right windows.
	 */

	NhPow2 = (int) pow( 2, ceil( log2( Nh ) ) );

	Nfft = NhPow2 + NhPow2;

	auxL = (double *) malloc( NhPow2 * sizeof( double ) );
	assert( auxL != (double *) NULL );

	auxR = (double *) malloc( NhPow2 * sizeof( double ) );
	assert( auxR != (double *) NULL );

	planL = fftw_plan_r2r_1d( NhPow2, auxL, auxL, FFTW_R2HC, FFTW_MEASURE );
	planR = fftw_plan_r2r_1d( NhPow2, auxR, auxR, FFTW_R2HC, FFTW_MEASURE );

	fftL = (double *) malloc( Nfft * sizeof( double ) );
	assert( fftL != (double *) NULL );

	fftR = (double *) malloc( Nfft * sizeof( double ) );
	assert( fftR != (double *) NULL );

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	*TintPtr = Tint;
	*TextPtr = Text;
	*TsepPtr = Tsep;
	*TsizePtr = Nh;
	*hLPtr = hL;
	*hRPtr = hR;
	*NhPtr = Nh;
	*NhPow2Ptr = NhPow2;
	*auxLPtr = auxL;
	*auxRPtr = auxR;
	*planLPtr = planL;
	*planRPtr = planR;
	*fftLPtr = fftL;
	*fftRPtr = fftR;
	*NfftPtr = Nfft;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

/*
 * DESTRUCTOR
 */

void windowpairDESTROY
	(
		double **    hLPtr,
		double **    hRPtr,
		double **    auxLPtr,
		double **    auxRPtr,
		fftw_plan *  planLPtr,
		fftw_plan *  planRPtr,
		double **    fftLPtr,
		double **    fftRPtr
	)
{
	free( (void *) (*hLPtr) );
	*hLPtr = (double *) NULL;

	free( (void *) (*hRPtr) );
	*hRPtr = (double *) NULL;

	free( (void *) (*auxLPtr) );
	*auxLPtr = (double *) NULL;

	free( (void *) (*auxRPtr) );
	*auxRPtr = (double *) NULL;

	fftw_destroy_plan( (*planLPtr) );
	fftw_destroy_plan( (*planRPtr) );

	free( (void *) (*fftLPtr) );
	*fftLPtr = (double *) NULL;

	free( (void *) (*fftRPtr) );
	*fftRPtr = (double *) NULL;

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

/*
 * SIMULTANEOUS FFT OF BOTH WINDOWED VERSIONS OF THE SAME AUDIO
 */

void windowpairTWOFFT
	(
		const signed short int *  audio,
		const int                 nData,
		const double *            hL,    /* length Nh */
		const double *            hR,    /* length Nh */
		const int                 Nh,
		const int                 NhPow2,
		double *                  auxL,  /* length NhPow2 */
		double *                  auxR,  /* length NhPow2 */
		fftw_plan                 planL,
		fftw_plan                 planR,
		double *                  fftL,  /* length Nfft */
		double *                  fftR,  /* length Nfft */
		const int                 Nfft
	)
{
	int  j;
	int  reSrcIdx;
	int  imSrcIdx;
	int  reDstIdx;
	int  imDstIdx;

	assert( ( NhPow2 + NhPow2 ) == Nfft ); 

	for ( j = 0; ( j < nData ) && ( j < Nh ); j ++ )
	{
		double  myAudio;

		#ifdef DOPREEMPHASIS
		{
			if ( j == 0 )
			{
				myAudio = 0.0;
			}
			else
			{
				myAudio = (double)(audio[ j ]) - 0.97 * (double)(audio[ j - 1 ]); 
			}
		}
		#else
		{
			myAudio = audio[ j ];
		}
		#endif

		auxL[ j ] = myAudio * hL[ j ];
		auxR[ j ] = myAudio * hR[ j ];
	}

	if ( nData < Nh )
	{
		for ( j = nData; j < NhPow2; j ++ )
		{
			auxL[ j ] = 0.0;
			auxR[ j ] = 0.0;
		}
	}
	else
	{
		for  ( j = Nh; j < NhPow2; j ++ )
		{
			auxL[ j ] = 0.0;
			auxR[ j ] = 0.0;
		}
	}

	fftw_execute( planL );
	fftw_execute( planR );

	/*
	 * IF THE FFT WERE 8-POINT (IE. NhPow2=8 REAL NUMBERS)
	 *
	 * HAVE (native FFTW):
	 *
	 * input  : [ r0 ][ r1 ][ r2 ][ r3 ][ r4 ][ r5 ][ r6 ][ r7 ]
	 * output : [ r0 ][ r1 ][ r2 ][ r3 ][ r4 ][ i3 ][ i2 ][ i1 ]
	 * (idx)      0     1     2     3     4     5     6     7
	 *
	 * WANT (native NR/C):
	 *
	 * input  : [ r0 ][ 0  ][ r1 ][ 0  ][ r2 ][ 0  ][ r3 ][ 0  ][ r4 ][ 0  ][ r5 ][  0  ][ r6 ][  0  ][ r7 ][  0  ]
	 * output : [ r0 ][ 0  ][ r1 ][ i1 ][ r2 ][ i2 ][ r3 ][ i3 ][ r4 ][ 0  ][ r3 ][ -i3 ][ r2 ][ -i2 ][ r1 ][ -i1 ]
	 * (idx)      0     1     2     3     4     5     6     7     8     9     10     11    12     13    14     15
	 */

	fftL[ 0 ] = auxL[ 0 ];
	fftL[ 1 ] = 0.0;

	fftR[ 0 ] = auxR[ 0 ];
	fftR[ 1 ] = 0.0;

	for
		(
			reSrcIdx = 1, imSrcIdx = NhPow2 - 1, reDstIdx = 2, imDstIdx = 3;
			reSrcIdx < (NhPow2 / 2); 
			reSrcIdx ++, imSrcIdx --, reDstIdx += 2, imDstIdx += 2
		)
	{
		fftL[ reDstIdx ] = auxL[ reSrcIdx ];
		fftL[ imDstIdx ] = auxL[ imSrcIdx ]; 

		fftR[ reDstIdx ] = auxR[ reSrcIdx ];
		fftR[ imDstIdx ] = auxR[ imSrcIdx ];
	}

	fftL[ NhPow2 ] = auxL[ NhPow2 / 2 ];
	fftL[ NhPow2 + 1 ] = 0.0;

	fftR[ NhPow2 ] = auxR[ NhPow2 / 2 ];
	fftR[ NhPow2 + 1 ] = 0.0;

	for
		(
			reSrcIdx = NhPow2 - 2, imSrcIdx = NhPow2 - 1,
				reDstIdx = NhPow2 + 2, imDstIdx = NhPow2 + 3;
			reDstIdx < Nfft;
			reSrcIdx -= 2, imSrcIdx -= 2, reDstIdx += 2, imDstIdx += 2
		)
	{
		fftL[ reDstIdx ] = + fftL[ reSrcIdx ];
		fftL[ imDstIdx ] = - fftL[ imSrcIdx ];

		fftR[ reDstIdx ] = + fftR[ reSrcIdx ];
		fftR[ imDstIdx ] = - fftR[ imSrcIdx ];
	}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

	return;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

