/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <float.h>
#include "tmcount.h"
#include "tm.h"
#include "bit.h"
#include "mpedo_2_0.h"

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void mpedo_2_0ALLOC
	(
		MPEDO_2_0 *  mpedoPtr,
		const int    Kmax
	)
{
	int      Kmax_plus_one;
	int ***  c;

	assert( Kmax > 0 );
	Kmax_plus_one = Kmax + 1;

	c = (int ***) malloc( Kmax_plus_one * sizeof( int ** ) );
	assert( c != (int ***) NULL );

	{
		int  ni;

		for ( ni = 0; ni < Kmax_plus_one; ni ++ )
		{
			int **  c_ni_;
			int     oij;

			c_ni_ = (int **) malloc( Kmax_plus_one * sizeof( int * ) );
			assert( c_ni_ != (int **) NULL );

			for ( oij = 0; oij < Kmax_plus_one; oij ++ )
			{
				int *  c_ni_oij_;

				c_ni_oij_ = (int *) malloc( Kmax_plus_one * sizeof( int ) );
				assert( c_ni_oij_ != (int *) NULL );

				c_ni_[ oij ] = c_ni_oij_;
			}

			c[ ni ] = c_ni_;
		}
	}

	mpedoPtr->Kmax = Kmax;
	mpedoPtr->c = c;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void mpedo_2_0FREE
	(
		MPEDO_2_0 *  mpedoPtr
	)
{
	int      Kmax;
	int ***  c;
	int      Kmax_plus_one;

	Kmax = mpedoPtr->Kmax;
	c = mpedoPtr->c;

	Kmax_plus_one = Kmax + 1;

	{
		int  ni;

		for ( ni = 0; ni < Kmax_plus_one; ni ++ )
		{
			int **  c_ni_;
			int     oij;

			c_ni_ = c[ ni ];

			for ( oij = 0; oij < Kmax_plus_one; oij ++ )
			{
				free( (void *) (c_ni_[ oij ]) );
			}

			free( (void *) c_ni_ );
		} 
	}

	free( (void *) c );

	mpedoPtr->Kmax = 0;
	mpedoPtr->c = (int ***) NULL;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  mpedo_2_0CLEAR
	(
		MPEDO_2_0 *  mpedoPtr
	)
{
	int      Kmax;
	int ***  c;
	int      Kmax_plus_one;

	Kmax = mpedoPtr->Kmax;
	c = mpedoPtr->c;

	Kmax_plus_one = Kmax + 1;

	{
		int  ni;

		for ( ni = 0; ni < Kmax_plus_one; ni ++ )
		{
			int **  c_ni_;
			int     oij;

			c_ni_ = c[ ni ];

			for ( oij = 0; oij < Kmax_plus_one; oij ++ )
			{
				int *  c_ni_oij_;
				int    nj;

				c_ni_oij_ = c_ni_[ oij ];

				for ( nj = 0; nj < Kmax_plus_one; nj ++ )
				{
					c_ni_oij_[ nj ] = 0.0;
				}
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  mpedo_2_0IMPORT
	(
		MPEDO_2_0 *      mpedoPtr,
		const TMCount *  tmCountPtr
	)
{
	int     mpedo_Kmax;
	int *** mpedo_c;
	int     tmcount_K;
	int     tmcount_M;
	int **  tmcount_c;
	int     tmcount_N;

	mpedo_Kmax = mpedoPtr->Kmax;
	mpedo_c = mpedoPtr->c;
	
	tmcount_K = tmCountPtr->K;
	tmcount_M = tmCountPtr->M;
	tmcount_c = tmCountPtr->c;

	assert( tmcount_K > 0 );
	assert( tmcount_M > 1 );

	tmcount_N = (int) pow( (double) tmcount_M, (double) tmcount_K );

	{
		int  Si;

		for ( Si = 0; Si < tmcount_N; Si ++ )
		{
			int *  tmcount_c_Si_;
			int    ni;
			int    Sj;

			tmcount_c_Si_ = tmcount_c[ Si ];

			ni = CNTBIT( Si, tmcount_K );
			if ( ni > mpedo_Kmax )
			{
				ni = mpedo_Kmax;
			}

			for ( Sj = 0; Sj < tmcount_N; Sj ++ )
			{
				int  nj;
				int  oij;
				int  count;

				nj = CNTBIT( Sj, tmcount_K );
				if ( nj > mpedo_Kmax )
				{
					nj = mpedo_Kmax;
				}

				oij = CNTBIT( Si & Sj, tmcount_K );
				if ( oij > ni )
				{
					oij = ni;
				}
				if ( oij > nj )
				{
					oij = nj;
				}

				count = tmcount_c_Si_[ Sj ];

				mpedo_c[ ni ][ oij ][ nj ] += count;
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  mpedo_2_0EXPORT
	(
		const MPEDO_2_0 *  mpedoPtr,
		TM *               tmPtr,
		const double       unkFraction
	)
{
	int        mpedo_Kmax;
	int ***    mpedo_c;
	int        tm_K;
	int        tm_Kmax;
	int        tm_N;
	double **  tm_p;
	int **     tm_fertility;

	mpedo_Kmax = mpedoPtr->Kmax;
	mpedo_c = mpedoPtr->c;

	tm_K = tmPtr->K;
	tm_Kmax = tmPtr->Kmax;
	tm_N = tmPtr->N;
	tm_p = tmPtr->p;

	assert( tm_K > 0 );
	assert( tm_Kmax > 0 );
	assert( tm_N > 1 );

	tm_fertility = (int **) malloc( tm_N * sizeof( int * ) );
	assert( tm_fertility != (int **) NULL );

	/*
	 * Compute fertility.
	 */

	{
		int  Si;

		for ( Si = 0; Si < tm_N; Si ++ )
		{
			int *  tm_fertility_Si_;
			int    ni;
			int    Sj;

			tm_fertility_Si_ = (int *) malloc( tm_N * sizeof( int ) );
			assert( tm_fertility_Si_ != (int *) NULL );

			ni = CNTBIT( Si, tm_K );
			if ( ni > mpedo_Kmax )
			{
				ni = mpedo_Kmax;
			}

			for ( Sj = 0; Sj < tm_N; Sj ++ )
			{
				int  nj;
				int  oij;
				int  Sj2;

				tm_fertility_Si_[ Sj ] = 0;

				nj = CNTBIT( Sj, tm_K );
				if ( nj > mpedo_Kmax )
				{
					nj = mpedo_Kmax;
				}

				oij = CNTBIT( Si & Sj, tm_K );
				if ( oij > ni )
				{
					oij = ni;
				}
				if ( oij > nj )
				{
					oij = nj;
				}

				for ( Sj2 = 0; Sj2 < tm_N; Sj2 ++ )
				{ 
					int  nj2;

					nj2 = CNTBIT( Sj2, tm_K );
					if ( nj2 > mpedo_Kmax )
					{
						nj2 = mpedo_Kmax;
					}

					if ( nj == nj2 )
					{
						int  oij2;

						oij2 = CNTBIT( Si & Sj2, tm_K );
						if ( oij2 > ni )
						{
							oij2 = ni;
						}
						if ( oij2 > nj2 )
						{
							oij2 = nj2;
						}

						if ( oij == oij2 )
						{
							tm_fertility_Si_[ Sj ] ++;
						}
					}
				}
			}

			tm_fertility[ Si ] = tm_fertility_Si_;
		}
	}

	/*
	 * Now export.
	 */

	{
		int  Si;

		for ( Si = 0; Si < tm_N; Si ++ )
		{
			double *  tm_p_Si_;
			int *     tm_fertility_Si_;
			int       ni;

			tm_p_Si_ = tm_p[ Si ];
			tm_fertility_Si_ = tm_fertility[ Si ];

			ni = CNTBIT( Si, tm_K );
			if ( ni > mpedo_Kmax )
			{
				ni = mpedo_Kmax;
			}

			if ( ni > tm_Kmax )
			{
				int  Sj;

				/*
				 * Don't want states with ni > tm_Kmax in the topology:
				 *  set uniform egress probabilities, among allowed next states.
				 */

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					int  nj;

					nj = CNTBIT( Sj, tm_K );
					if ( nj > tm_Kmax )
					{
						tm_p_Si_[ Sj ] = 0.0;
					}
					else
					{
						tm_p_Si_[ Sj ] = 1.0;
					}
				}
			}
			else
			{
				int **  mpedo_c_ni_;
				int     Sj;
				double  sum;
				int     nAllowedButZero;

				mpedo_c_ni_ = mpedo_c[ ni ];

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					int  nj;

					nj = CNTBIT( Sj, tm_K );
					if ( nj > tm_Kmax )
					{
						/*
						 * Don't want states with nj > tm_Kmax in the topology:
						 *  set zero egress probabilities.
						 */

						tm_p_Si_[ Sj ] = 0.0;
					}
					else
					{
						int  oij;

						if ( nj > mpedo_Kmax )
						{
							nj = mpedo_Kmax;
						}

						oij = CNTBIT( Si & Sj, tm_K );
						if ( oij > ni )
						{
							oij = ni;
						}
						if ( oij > nj )
						{
							oij = nj;
						}

						tm_p_Si_[ Sj ] = ((double) (mpedo_c_ni_[ oij ][ nj ]))
							/ ((double) (tm_fertility_Si_[ Sj ]));
					}
				}

				/*
				 * Ensure that sum to unity, priot to mass distribution.
				 */

				sum = 0.0;
				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					sum += tm_p_Si_[ Sj ];
				}

				if ( sum == 0.0 )
				{
					int  Sj;

					/*
					 * Conditioning context must have not occurred in the training data;
					 *  set uniform egress probabilities, among allowed next states.
					 */

					for ( Sj = 0; Sj < tm_N; Sj ++ )
					{
						int  nj;

						nj = CNTBIT( Sj, tm_K );
						if ( nj > tm_Kmax )
						{
							tm_p_Si_[ Sj ] = 0.0;
						}
						else
						{
							tm_p_Si_[ Sj ] = 1.0;
							sum += 1.0;
						}
					}
				}

				assert( sum > 0.0 );

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					tm_p_Si_[ Sj ] /= sum;
				}

				/*
				 * Distribute observed mass to unseen _allowed_ states.
				 */

				nAllowedButZero = 0;

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					int  nj;

					nj = CNTBIT( Sj, tm_K );
					if ( nj > tm_Kmax )
					{
						/* do nothing */
					}
					else
					{
						if ( tm_p_Si_[ Sj ] == 0.0 )
						{
							nAllowedButZero ++;
						}
					}
				}

				if ( nAllowedButZero > 0 )
				{
					for ( Sj = 0; Sj < tm_N; Sj ++ )
					{
						int  nj;

						nj = CNTBIT( Sj, tm_K );
						if ( nj > tm_Kmax )
						{
							/* do nothing */
						}
						else
						{
							if ( tm_p_Si_[ Sj ] == 0.0 )
							{
								tm_p_Si_[ Sj ] = ( unkFraction / nAllowedButZero );
							}
							else
							{
								tm_p_Si_[ Sj ] *= ( 1.0 - unkFraction );
							}
						}
					}
				}
			}

			/*
			 * Ensure that probabilities sum to unity.
			 */

			{
				double  sum;
				int     Sj;

				sum = 0.0;
				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					sum += tm_p_Si_[ Sj ];
				}

				if ( sum == 0.0 )
				{
					int  Sj;

					/*
					 * Conditioning context must have not occurred in the training data;
					 *  set uniform egress probabilities, among allowed next states.
					 */

					for ( Sj = 0; Sj < tm_N; Sj ++ )
					{
						int  nj;

						nj = CNTBIT( Sj, tm_K );
						if ( nj > tm_Kmax )
						{
							tm_p_Si_[ Sj ] = 0.0;
						}
						else
						{
							tm_p_Si_[ Sj ] = 1.0;
							sum += 1.0;
						}
					}
				}

				assert( sum > 0.0 );

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					tm_p_Si_[ Sj ] /= sum;
				}
			}

			/*
			 * Replace zero by almost-zero.
			 */

			{
				int  Sj;

				for ( Sj = 0; Sj < tm_N; Sj ++ )
				{
					if ( tm_p_Si_[ Sj ] == 0.0 )
					{
						tm_p_Si_[ Sj ] = DBL_MIN;
					}
				}
			}
		}
	}

	/*
	 * Deallocate fertility.
	 */

	{
		int  Si;

		for ( Si = 0; Si < tm_N; Si ++ )
		{
			free( (void *) (tm_fertility[ Si ]) );
		}

		free( (void *) tm_fertility );
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

