/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2011, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdio.h>
#include <math.h>
#include <float.h>
#include "spedo_n_nu.h"
#include "q.h"
#include "metric.h"

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void metricCLEAR
	(
		Metric *  metricPtr
	)
{
	metricPtr->reliable = 1;

	metricPtr->sumKNormNegLogProb = 0.0;
	metricPtr->sumT = 0.0;
	metricPtr->sumKGuessStochasticCorrectNum = 0.0;
	metricPtr->sumKGuessStochasticCorrectDen = 0.0;
	metricPtr->sumKGuessDeterministicCorrectNum = 0.0;
	metricPtr->sumKGuessDeterministicCorrectDen = 0.0;

	metricPtr->sumKNormNegLogProbChange = 0.0;
	metricPtr->sumTChange = 0.0;
	metricPtr->sumKGuessStochasticCorrectNumChange = 0.0;
	metricPtr->sumKGuessStochasticCorrectDenChange = 0.0;
	metricPtr->sumKGuessDeterministicCorrectNumChange = 0.0;
	metricPtr->sumKGuessDeterministicCorrectDenChange = 0.0;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void metricACCUM
	(
		Metric *            metricPtr,
		const SPEDO_n_nu *  spedoPtr,
		const EvalType      evalType,
		const int           K,
		const int *         q,
		const int           t
	)
{
	if ( metricPtr->reliable == 1 )
	{
		double  prob;

		/*
		 * Compute the metrics for this instant.
		 */

		prob = spedo_n_nuPROB( spedoPtr, K, q, t, evalType );
		if ( ( prob <= 0.0 ) || ( prob > 1.0 ) || ( isnan( prob ) ) || ( isinf( prob ) ) )
		{
			metricPtr->reliable = 0;
		}
		else
		{
			double  instKNormNegLogProb;
			int     guessStochastic;
			int     guessStochasticKCorrect;
			int     guessDeterministic;
			int     guessDeterministicKCorrect;
			int     changeFlag;

			instKNormNegLogProb = - log2( prob ) / K;

			guessStochastic = spedo_n_nuPREDSTOCHASTIC( spedoPtr, K, q, t, evalType );
			guessStochasticKCorrect = QCompare( guessStochastic, q[ t ], K );

			guessDeterministic = spedo_n_nuPREDDETERMINISTIC( spedoPtr, K, q, t, evalType );
			guessDeterministicKCorrect = QCompare( guessDeterministic, q[ t ], K );

			/*
			 * Accumulate metrics into counter for all instants.
			 */

			metricPtr->sumKNormNegLogProb += instKNormNegLogProb;
			metricPtr->sumT += 1.0;

			metricPtr->sumKGuessStochasticCorrectNum += guessStochasticKCorrect;
			metricPtr->sumKGuessStochasticCorrectDen += K;

			metricPtr->sumKGuessDeterministicCorrectNum += guessDeterministicKCorrect;
			metricPtr->sumKGuessDeterministicCorrectDen += K;

			/*
			 * Accumulate metrics into counter for the change instants.
			 */

			changeFlag = 1;
			if ( t == 0 )
			{
				if ( 0 == q[ 0 ] )
				{
					changeFlag = 0;
				}   
			}
			else
			{
				if ( q[ t - 1 ] == q[ t ] )
				{
					changeFlag = 0;
				}
			}

			if ( changeFlag == 1 )
			{
				metricPtr->sumKNormNegLogProbChange += instKNormNegLogProb;
				metricPtr->sumTChange += 1.0;

				metricPtr->sumKGuessStochasticCorrectNumChange += guessStochasticKCorrect;
				metricPtr->sumKGuessStochasticCorrectDenChange += K;

				metricPtr->sumKGuessDeterministicCorrectNumChange += guessDeterministicKCorrect;
				metricPtr->sumKGuessDeterministicCorrectDenChange += K;
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void metricADD
	(
		Metric *        sumPtr,
		const Metric *  addendPtr
	)
{
	if ( addendPtr->reliable == 1 )
	{
		sumPtr->sumKNormNegLogProb += addendPtr->sumKNormNegLogProb;
		sumPtr->sumT += addendPtr->sumT;
		sumPtr->sumKGuessStochasticCorrectNum += addendPtr->sumKGuessStochasticCorrectNum;
		sumPtr->sumKGuessStochasticCorrectDen += addendPtr->sumKGuessStochasticCorrectDen;
		sumPtr->sumKGuessDeterministicCorrectNum += addendPtr->sumKGuessDeterministicCorrectNum;
		sumPtr->sumKGuessDeterministicCorrectDen += addendPtr->sumKGuessDeterministicCorrectDen;

		sumPtr->sumKNormNegLogProbChange += addendPtr->sumKNormNegLogProbChange;
		sumPtr->sumTChange += addendPtr->sumTChange;
		sumPtr->sumKGuessStochasticCorrectNumChange += addendPtr->sumKGuessStochasticCorrectNumChange;
		sumPtr->sumKGuessStochasticCorrectDenChange += addendPtr->sumKGuessStochasticCorrectDenChange;
		sumPtr->sumKGuessDeterministicCorrectNumChange += addendPtr->sumKGuessDeterministicCorrectNumChange;
		sumPtr->sumKGuessDeterministicCorrectDenChange += addendPtr->sumKGuessDeterministicCorrectDenChange;
	}
	else
	{
		sumPtr->reliable = 0;
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void metricPRINT
	(
		const Metric *  metricPtr
	)
{
	if ( metricPtr->reliable == 1 )
	{
		printf
			(
				"PP(all): %.16g PP(sub): %.16g NLL(all): %.16g NLL(sub): %.16g ES(all): %6.2f ES(sub): %6.2f ED(all): %6.2f ED(sub): %6.2f\n",
				pow( 2.0, ( metricPtr->sumKNormNegLogProb / metricPtr->sumT ) ),
				pow( 2.0, ( metricPtr->sumKNormNegLogProbChange / metricPtr->sumTChange ) ),
				metricPtr->sumKNormNegLogProb / metricPtr->sumT,
				metricPtr->sumKNormNegLogProbChange / metricPtr->sumTChange,
				100.0 * metricPtr->sumKGuessStochasticCorrectNum
					/ metricPtr->sumKGuessStochasticCorrectDen,
				100.0 * metricPtr->sumKGuessStochasticCorrectNumChange
					/ metricPtr->sumKGuessStochasticCorrectDenChange,
				100.0 * metricPtr->sumKGuessDeterministicCorrectNum
					/ metricPtr->sumKGuessDeterministicCorrectDen,
				100.0 * metricPtr->sumKGuessDeterministicCorrectNumChange
					/ metricPtr->sumKGuessDeterministicCorrectDenChange
			);
	}
	else
	{
		printf
			(
				"PP(all): Inf PP(sub): Inf NLL(all): Inf NLL(sub): Inf ES(all): NaN ES(sub): NaN ED(all): NaN ED(sub): NaN\n"
			);
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

double metricOBJFUNC
	(
		const Metric *  metricPtr
	)
{
	double  retVal;

	if ( metricPtr->reliable == 1 )
	{
		retVal = pow( 2.0, ( metricPtr->sumKNormNegLogProb / metricPtr->sumT ) );
	}
	else
	{
		retVal = +DBL_MAX;
	}

	return retVal;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

