/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <getopt.h>
#include "bit.h"
#include "q.h"
#include "sutils.h"
#include "sort.h"
#include "FileName.h"
#include "tmcount.h"
#include "tm.h"
#include "mpedo_2_0.h"

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#define BINARY 2

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

extern char *  optarg;
extern int     optind;
extern int     opterr;
extern int     optopt;

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

char *  progName = "mpedo";

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

void Usage()
{
	fprintf
		(
			stderr,
			"Usage: %s -QDir <dirName>\n",
			progName
		); 

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int main( int argc, char * argv[] ) 
{
	char *             QDirName;
	char **            fileNameTab;
	int                nFiles;
	unsigned long *    sortedIndexTab;
        double             allSumKGuessDeterministicCorrectNum;
        double             allSumKGuessDeterministicCorrectDen;
        double             allSumKGuessDeterministicCorrectNumChange;
        double             allSumKGuessDeterministicCorrectDenChange;

	progName = argv[ 0 ];

	QDirName = (char *) NULL;

	{
		int  argIdx;

		printf( "Command line:" );
		for ( argIdx = 0; argIdx < argc; argIdx ++ )
		{
			printf( " %s", argv[ argIdx ] );
		}
		printf( "\n" );
	}

	while ( 1 )
	{
		int  optChar;
		int  option_index = 0;
		static struct option  long_options[] =
			{
				{ "QDir",         1, 0, 1 }
			};
		optChar = getopt_long_only( argc, argv, "", long_options, &option_index );
		if ( optChar == -1 )
		{
			break;
		}

		switch( optChar )
		{
		case 1: /* QDir */
			QDirName = optarg;
			break;
		case '?':
		default:
			fprintf( stderr, "%s: badly formatted argument list\n", progName );
			Usage();
			exit( -1 );
		}
	}

	if ( QDirName == (char *) NULL )
	{
		fprintf( stderr, "%s: -QDir not specified\n", progName );
		Usage();
		exit( -1 );
	}
	else
	{
		struct stat  statBuf;

		if ( stat( QDirName, &statBuf ) != 0 )
		{
			fprintf
				(
					stderr,
					"%s: Cannot access -QDir value [%s]\n",
					progName,
					QDirName
				);
			exit( -1 );
		}

		if ( S_ISDIR( statBuf.st_mode ) )
		{
			/* ok */
		}
		else
		{
			fprintf
				(
					stderr,
					"%s: -QDir value [%s] exists, but not a directory\n",
					progName,
					QDirName
				);
			exit( -1 );
		}
	}

        FileNameGETLIST
                (
                        QDirName,
                        "Q",
                        &fileNameTab,
                        &nFiles
                );
	sortedIndexTab = (unsigned long *) malloc( nFiles * sizeof( unsigned long ) );
	assert( sortedIndexTab != (unsigned long *) NULL );
	strindexx( nFiles, fileNameTab - 1, sortedIndexTab - 1 ); /* NRC 1-offset */

	allSumKGuessDeterministicCorrectNum = 0.0;
	allSumKGuessDeterministicCorrectDen = 0.0;

	allSumKGuessDeterministicCorrectNumChange = 0.0;
	allSumKGuessDeterministicCorrectDenChange = 0.0;

	{
		int  testFileIdx;

		for ( testFileIdx = 0; testFileIdx < nFiles; testFileIdx ++ )
		{
			char *     inPathName;
			int        T;
			int        K;
			int *      q;
			double     testSumKGuessDeterministicCorrectNum;
			double     testSumKGuessDeterministicCorrectDen;
			double     testSumKGuessDeterministicCorrectNumChange;
			double     testSumKGuessDeterministicCorrectDenChange;

			inPathName = fileNameTab[ sortedIndexTab[ testFileIdx ] - 1 ];
			QReadAndEncode( inPathName, &q, &T, &K );

			testSumKGuessDeterministicCorrectNum = 0.0;
			testSumKGuessDeterministicCorrectDen = 0.0;

			testSumKGuessDeterministicCorrectNumChange = 0.0;
			testSumKGuessDeterministicCorrectDenChange = 0.0;

			{
				int  guessDeterministic;
				int  k;

				guessDeterministic = 0;
				for ( k = 0; k < K; k ++ )
				{
					SETBIT( guessDeterministic, k );
				}

				{
					int  guessDeterministicKCorrect;
					int  t;

					guessDeterministicKCorrect = QCompare( guessDeterministic, q[ 0 ], K );

					testSumKGuessDeterministicCorrectNum += guessDeterministicKCorrect;
					testSumKGuessDeterministicCorrectDen += K;

					if ( 0 != q[ 0 ] )
					{
						testSumKGuessDeterministicCorrectNumChange += guessDeterministicKCorrect;
						testSumKGuessDeterministicCorrectDenChange += K;
					}

					for ( t = 1; t < T; t ++ )
					{
						guessDeterministicKCorrect = QCompare( guessDeterministic, q[ t ], K );

						testSumKGuessDeterministicCorrectNum += guessDeterministicKCorrect;
						testSumKGuessDeterministicCorrectDen += K;

						if ( q[ t - 1 ] != q[ t ] )
						{
							testSumKGuessDeterministicCorrectNumChange += guessDeterministicKCorrect;
							testSumKGuessDeterministicCorrectDenChange += K;
						}
					}
				}
			}

			free( (void *) q );

			printf
				(
					"%s [ED(all): %6.2f] [ED(sub): %6.2f]\n",
					inPathName,
					100.0 *  testSumKGuessDeterministicCorrectNum / testSumKGuessDeterministicCorrectDen,
					100.0 * testSumKGuessDeterministicCorrectNumChange / testSumKGuessDeterministicCorrectDenChange
				);

			allSumKGuessDeterministicCorrectNum += testSumKGuessDeterministicCorrectNum;
			allSumKGuessDeterministicCorrectDen += testSumKGuessDeterministicCorrectDen;

			allSumKGuessDeterministicCorrectNumChange += testSumKGuessDeterministicCorrectNumChange;
			allSumKGuessDeterministicCorrectDenChange += testSumKGuessDeterministicCorrectDenChange;
		}
	}

	printf
		(
			"ALL :      [ED(all): %6.2f] [ED(sub): %6.2f]\n",
			100.0 * allSumKGuessDeterministicCorrectNum / allSumKGuessDeterministicCorrectDen,
			100.0 * allSumKGuessDeterministicCorrectNumChange / allSumKGuessDeterministicCorrectDenChange
		);

	return 0;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

