/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include "sutils.h"
#include "bit.h"
#include "ran1.h"
#include "spedo_n_0.h"

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_n_0PARSEDEFSTRING
	(
		const char *    defString,
		int *           depthPtr,
		SubTreeType **  subTreeTypeTabPtr,
		int **          subTreeDelayTabPtr,
		int *           delayMaxPtr
	)
{
	int            retVal;
	int            defStringLen;
	char *         defStringCopy;
	int            depth;
	const char *   separatorString = ",";
	char *         tokenPtr;
	SubTreeType *  subTreeTypeTab;
	int *          subTreeDelayTab;
	int            delayMax;

	retVal = 0;

	defStringLen = strlen( defString ) + 1;
	defStringCopy = (char *) malloc( defStringLen * sizeof( char ) );
	assert( defStringCopy != (char *) NULL );
	strcpy( defStringCopy, defString );

	depth = 0;
	subTreeTypeTab = (SubTreeType *) NULL;
	subTreeDelayTab = (int *) NULL;
	delayMax = 0;

	if ( ( tokenPtr = strtok( defStringCopy, separatorString ) ) != (char *) NULL )
	{
		int  depthIdx;

		do
		{
			if ( depth == 0 )
			{
				subTreeTypeTab = (SubTreeType *) malloc( sizeof( SubTreeType ) );
				assert( subTreeTypeTab != (SubTreeType *) NULL );

				subTreeDelayTab = (int *) malloc( sizeof( int ) );
				assert( subTreeDelayTab != (int *) NULL );
			}
			else
			{
				int            newDepth;
				SubTreeType *  newSubTreeTypeTab;
				int *          newSubTreeDelayTab;

				newDepth = depth + 1;

				newSubTreeTypeTab = (SubTreeType *) malloc( newDepth * sizeof( SubTreeType ) );
				assert( newSubTreeTypeTab != (SubTreeType *) NULL );

				newSubTreeDelayTab = (int *) malloc( newDepth * sizeof( int ) );
				assert( newSubTreeDelayTab != (int *) NULL );

				for ( depthIdx = 0; depthIdx < depth; depthIdx ++ )
				{
					newSubTreeTypeTab[ depthIdx ] = subTreeTypeTab[ depthIdx ];
					newSubTreeDelayTab[ depthIdx ] = subTreeDelayTab[ depthIdx ];
				}

				free( (void *) subTreeTypeTab );
				free( (void *) subTreeDelayTab );

				subTreeTypeTab = newSubTreeTypeTab;
				subTreeDelayTab = newSubTreeDelayTab;
			}

			switch( tokenPtr[ 0 ] )
			{
			case 'S':
				subTreeTypeTab[ depth ] = SubTreeType_Self;
				break;

			case 'N':
				subTreeTypeTab[ depth ] = SubTreeType_NOthers;
				break;

			default:
				retVal = -1;
				break;
			}

			subTreeDelayTab[ depth ] = str2int( tokenPtr + 1 );

			depth ++;
		}
		while ( ( tokenPtr = strtok( (char *) NULL, separatorString ) ) != (char *) NULL );

		if ( retVal == 0 )
		{
			delayMax = -1;
			for ( depthIdx = 0; depthIdx < depth; depthIdx ++ )
			{
				int  subTreeDelay;

				subTreeDelay = subTreeDelayTab[ depthIdx ];

				if ( subTreeDelay > delayMax )
				{
					delayMax = subTreeDelay;
				}
			}
		}
	}

	free( (void *) defStringCopy );

	if ( retVal == 0 )
	{
		*depthPtr = depth;
		*subTreeTypeTabPtr = subTreeTypeTab;
		*subTreeDelayTabPtr = subTreeDelayTab;
		*delayMaxPtr = delayMax;
	}

	return retVal;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0ALLOC_recursive_helper
	(
		Node *               nodePtr,
		const int            M,
		const int            Kmax,
		const int            depth,
		const int            level,
		const SubTreeType *  subTreeTypeTab,
		const int *          subTreeDelayTab
	)
{
	if ( level < depth )
	{
		int     mySubTreeType;
		int     nAlt;
		Node *  altTab;
		int     altIdx;

		mySubTreeType = subTreeTypeTab[ level ];
		nodePtr->subTreeType = mySubTreeType;
		nodePtr->subTreeDelay = subTreeDelayTab[ level ];
		nodePtr->count = 0;
		nodePtr->prob = 0.0;

		switch( mySubTreeType )
		{
		case SubTreeType_Self:
			nAlt = M;
			break;

		case SubTreeType_NOthers:
			nAlt = Kmax;
			break;

		case SubTreeType_NONE:
		default:
			assert( 0 );
		}

		altTab = (Node *) malloc( nAlt * sizeof( Node ) );
		assert( altTab != (Node *) NULL );

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			spedo_n_0ALLOC_recursive_helper
				(
					&(altTab[ altIdx ]),
					M,
					Kmax,
					depth,
					level + 1,
					subTreeTypeTab,
					subTreeDelayTab
				);
		}

		nodePtr->nAlt = nAlt;
		nodePtr->altTab = altTab;
	}
	else
	{
		nodePtr->subTreeType = SubTreeType_NONE;
		nodePtr->subTreeDelay = 0;
		nodePtr->count = 0;
		nodePtr->prob = 0.0;
		nodePtr->nAlt = 0;
		nodePtr->altTab = (Node *) NULL;
	}

	return;
}

void spedo_n_0ALLOC
	(
		SPEDO_n_0 *    spedoPtr,
		const int      M,
		const int      Kmax,
		const int      depth,
		SubTreeType *  subTreeTypeTab,
		int *          subTreeDelayTab,
		const int      delayMax
	)
{
	Node *         altTab;
	int            m;

	altTab = (Node *) malloc( M * sizeof( Node ) );
	assert( altTab != (Node *) NULL );

	for ( m = 0; m < M; m ++ )
	{
		spedo_n_0ALLOC_recursive_helper
			(
				&(altTab[ m ]),
				M,
				Kmax,
				depth,
				0,
				subTreeTypeTab,
				subTreeDelayTab
			);
	}

	spedoPtr->M = M;
	spedoPtr->Kmax = Kmax;
	spedoPtr->depth = depth;
	if ( depth > 0 )
	{
		SubTreeType *  subTreeTypeTabCopy;
		int *          subTreeDelayTabCopy;
		int            depthIdx;

		subTreeTypeTabCopy = (SubTreeType *) malloc( depth * sizeof( SubTreeType ) );
		assert( subTreeTypeTabCopy != (SubTreeType *) NULL );

		subTreeDelayTabCopy = (int *) malloc( depth * sizeof( int ) );
		assert( subTreeDelayTabCopy != (int *) NULL );

		for ( depthIdx = 0; depthIdx < depth; depthIdx ++ )
		{
			subTreeTypeTabCopy[ depthIdx ] = subTreeTypeTab[ depthIdx ];
			subTreeDelayTabCopy[ depthIdx ] = subTreeDelayTab[ depthIdx ];
		}

		spedoPtr->subTreeTypeTab = subTreeTypeTabCopy;
		spedoPtr->subTreeDelayTab = subTreeDelayTabCopy;
	}
	else
	{
		spedoPtr->subTreeTypeTab = (SubTreeType *) NULL;
		spedoPtr->subTreeDelayTab = (int *) NULL;
	}
	spedoPtr->delayMax = delayMax;
	spedoPtr->count = 0;
	spedoPtr->prob = 0.0;
	spedoPtr->altTab = altTab;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0FREE_recursive_helper
	(
		Node *  nodePtr
	)
{
	int  subTreeType;

	subTreeType = nodePtr->subTreeType;

	if ( subTreeType != SubTreeType_NONE )
	{
		int  nAlt;
		int  altIdx;

		nAlt = nodePtr->nAlt;

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			spedo_n_0FREE_recursive_helper
				(
					&(nodePtr->altTab[ altIdx ])
				);
		}

		free( (void *) (nodePtr->altTab) );
	}

	return;
}

void spedo_n_0FREE
	(
		SPEDO_n_0 *  spedoPtr
	)
{
	int  M;
	int  m;

	free( (void *) (spedoPtr->subTreeTypeTab) );
	free( (void *) (spedoPtr->subTreeDelayTab) );

	M = spedoPtr->M;

	for ( m = 0; m < M; m ++ )
	{

		spedo_n_0FREE_recursive_helper
			(
				&(spedoPtr->altTab[ m ])
			);
	}

	free( (void *) (spedoPtr->altTab) );

	spedoPtr->M = 0;
	spedoPtr->Kmax = 0;
	spedoPtr->depth = 0;
	spedoPtr->subTreeTypeTab = (SubTreeType *) NULL;
	spedoPtr->subTreeDelayTab = (int *) NULL;
	spedoPtr->delayMax = 0;
	spedoPtr->count = 0;
	spedoPtr->prob = 0.0;
	spedoPtr->altTab = (Node *) NULL;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void  spedo_n_0CLEAR_recursive_helper
	(
		Node *  nodePtr
	)
{
	int  subTreeType;

	subTreeType = nodePtr->subTreeType;

	if ( subTreeType != SubTreeType_NONE )
	{
		int  nAlt;
		int  altIdx;

		nAlt = nodePtr->nAlt;

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			spedo_n_0CLEAR_recursive_helper
				(
					&(nodePtr->altTab[ altIdx ])
				);
		}
	}

	nodePtr->count = 0;
	nodePtr->prob = 0.0;

	return;
}

void  spedo_n_0CLEAR
	(
		SPEDO_n_0 *  spedoPtr
	)
{
	int  M;
	int  m;

	M = spedoPtr->M;

	for ( m = 0; m < M; m ++ )
	{
		spedo_n_0CLEAR_recursive_helper
			(
				&(spedoPtr->altTab[ m ])
			);
	}

	spedoPtr->count = 0;
	spedoPtr->prob = 0.0;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void  spedo_n_0ACCUM_recursive_helper
	(
		Node *               nodePtr,
		const int            depth,
		const int            level,
		const SubTreeType *  subTreeTypeTab,
		const int *          subTreeDelayTab,
		const int            KmaxMinusOne,
		const int            K,
		const int *          q,
		const int            t,
		const int            k
	)
{
	if ( level < depth )
	{
		SubTreeType  subTreeType;
		int          subTreeDelay;
		int          tau;
		int          q_tau;
		int          s;
		int          n;
		int          altIdx;

		subTreeType = subTreeTypeTab[ level ];
		subTreeDelay = subTreeDelayTab[ level ];

		tau = t - subTreeDelay;
		if ( tau < 0 )
		{
			q_tau = 0;
		}
		else
		{
			q_tau = q[ tau ];
		}

		s = GETBIT( q_tau, k );
		n = CNTBIT( q_tau, K ) - s;
		if ( n > KmaxMinusOne )
		{
			n = KmaxMinusOne;
		}

		altIdx = -1;
		switch( subTreeType )
		{
		case SubTreeType_Self:
			altIdx = s;
			break;
		case SubTreeType_NOthers:
			altIdx = n;
			break;
		case SubTreeType_NONE:
		default:
			assert( 0 );
		}

		spedo_n_0ACCUM_recursive_helper
			(
				&(nodePtr->altTab[ altIdx ]),
				depth,
				level + 1,
				subTreeTypeTab,
				subTreeDelayTab,
				KmaxMinusOne,
				K,
				q,
				t,
				k
			);
	}

	nodePtr->count ++;

	return;
}

void  spedo_n_0ACCUM
	(
		SPEDO_n_0 *  spedoPtr,
		const int    K,
		const int *  q,
		const int    t
	)
{
	int  N;
	int  delayMax;
	int  tau;
	int  KmaxMinusOne;
	int  k;

	assert( spedoPtr->M == 2 ); /* due to encoding of q */

	N = (int) pow( (double) (spedoPtr->M), (double) K );

	delayMax = spedoPtr->delayMax;
	for ( tau = (t - delayMax); tau <= t; tau ++ )
	{
		if ( tau < 0 )
		{
			/* ignore */
		}
		else
		{
			int  q_tau;

			q_tau = q[ tau ];

			assert( ( q_tau >= 0 ) && ( q_tau < N ) );
		}
	}
 
	KmaxMinusOne = spedoPtr->Kmax - 1;

	for ( k = 0; k < K; k ++ )
	{
		int  s0;

		s0 = GETBIT( q[ t ], k );
		assert( s0 < spedoPtr->M );

		spedo_n_0ACCUM_recursive_helper
			(
				&(spedoPtr->altTab[ s0 ]),
				spedoPtr->depth,
				0,
				spedoPtr->subTreeTypeTab,
				spedoPtr->subTreeDelayTab,
				KmaxMinusOne,
				K,
				q,
				t,
				k
			);

		spedoPtr->count ++;
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0BUILD_recursive_helper
	(
		Node **       nodePtrTab,
		const int     M,
		const double  unkFraction,
		const double  add
	)
{
	int          m;
	SubTreeType  subTreeType;

	for ( m = 0; m < M; m ++ )
	{
		Node *  nodePtr;
		int     count;
		double  sum;
		int     nZero;
		double  sum_loo;
		int     nZero_loo;
		double  augCount;
		double  augCount_loo;
		int     othm;

		nodePtr = nodePtrTab[ m ];
		count = nodePtr->count;

		/*
		 * Compute the numerator (augCount) with all counts, and increment denominator (sum).
		 */

		sum = 0.0;
		nZero = 0;
		if ( count <= 0 )
		{
			augCount = add;
		}
		else
		{
			augCount = count + add;
		}
		if  ( augCount == 0.0 )
		{
			nZero ++;
		}
		sum += augCount;

		/*
		 * Compute the numerator (augCount_loo) as if the observed number of counts were one fewer,
		 *  and increment denominator (sum_loo).
		 */

		sum_loo = 0.0;
		nZero_loo = 0;
		if ( count <= 1 )
		{
			augCount_loo = add;
		}
		else
		{
			augCount_loo = count - 1 + add;
		}
		if ( augCount_loo == 0.0 )
		{
			nZero_loo ++;
		}
		sum_loo += augCount_loo;

		/*
		 * Complete the denominator sum.
		 */

		for ( othm = 0; othm < M; othm ++ )
		{
			if ( othm != m )
			{
				Node *  othNodePtr;
				double  othCount;
				double  augOthCount;

				othNodePtr = nodePtrTab[ othm ];
				othCount = othNodePtr->count;
				augOthCount = othCount + add;

				if ( augOthCount == 0.0 )
				{
					nZero ++;
					nZero_loo ++;
				}

				sum += augOthCount;
				sum_loo += augOthCount;
			}
		}

		if ( sum <= 0.0 )
		{
			nodePtr->prob = 1.0 / ((double) M);
		}
		else if ( augCount <= 0.0 )
		{
			assert( nZero > 0 );

			nodePtr->prob = unkFraction / ((double) nZero);
		}
		else if ( nZero > 0 )
		{
			nodePtr->prob = (1.0 - unkFraction) * ( augCount / sum );
		}
		else
		{
			nodePtr->prob = ( augCount / sum );
		}

		if ( sum_loo <= 0.0 )
		{
			nodePtr->prob_loo = 1.0 / ((double) M);
		}
		else if ( augCount_loo <= 0.0 )
		{
			assert( nZero_loo > 0 );

			nodePtr->prob_loo = unkFraction / ((double) nZero_loo);
		}
		else if ( nZero_loo > 0 )
		{
			nodePtr->prob_loo = (1.0 - unkFraction) * ( augCount_loo / sum_loo );
		}
		else
		{
			nodePtr->prob_loo = ( augCount_loo / sum_loo );
		}
	}

	subTreeType = nodePtrTab[ 0 ]->subTreeType;
	if ( subTreeType != SubTreeType_NONE )
	{
		int  subTreeDelay;
		int  nAlt;
		int  altIdx;

		subTreeDelay = nodePtrTab[ 0 ]->subTreeDelay;
		nAlt = nodePtrTab[ 0 ]->nAlt;

		for ( m = 1; m < M; m ++ )
		{
			Node *  nodePtr;

			nodePtr = nodePtrTab[ m ];

			assert( nodePtr->nAlt == nAlt );
			assert( nodePtr->subTreeType == subTreeType );
			assert( nodePtr->subTreeDelay == subTreeDelay );
		}

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			Node **  subTreeNodePtrTab;

			subTreeNodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
			assert( subTreeNodePtrTab != (Node **) NULL );

			for ( m = 0; m < M; m ++ )
			{
				subTreeNodePtrTab[ m ] = &(nodePtrTab[ m ]->altTab[ altIdx ]);
			}

			spedo_n_0BUILD_recursive_helper
				(
					subTreeNodePtrTab,
					M,
					unkFraction,
					add
				);

			free( (void *) subTreeNodePtrTab );
		}
	}

	return;
}

void spedo_n_0BUILD
	(
		SPEDO_n_0 *   spedoPtr,
		const double  unkFraction,
		const double  add
	)
{
	int      M;
	int      m;
	Node **  nodePtrTab;

	M = spedoPtr->M;

	nodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
	assert( nodePtrTab != (Node **) NULL );

	for ( m = 0; m < M; m ++ )
	{
		nodePtrTab[ m ] = &(spedoPtr->altTab[ m ]);
	}

	spedo_n_0BUILD_recursive_helper
		(
			nodePtrTab,
			M,
			unkFraction,
			add
		);

	free( (void *) nodePtrTab );

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0BUILD_add_recursive_helper
	(
		Node **         nodePtrTab,
		const int       M,
		const EvalType  evalType,
		const double    add
	)
{
	int          m;
	SubTreeType  subTreeType;

	for ( m = 0; m < M; m ++ )
	{
		Node *  nodePtr;
		int     count;
		double  sum;
		double  sum_loo;
		double  augCount;
		double  augCount_loo;
		int     othm;

		nodePtr = nodePtrTab[ m ];
		count = nodePtr->count;

		/*
		 * Compute the numerator (augCount) with all counts, and increment denominator (sum).
		 */

		sum = 0.0;
		if ( count <= 0 )
		{
			augCount = add;
		}
		else
		{
			augCount = count + add;
		}
		sum += augCount;

		/*
		 * Compute the numerator (augCount_loo) as if the observed number of counts were one fewer,
		 *  and increment denominator (sum_loo).
		 */

		sum_loo = 0.0;
		if ( count <= 1 )
		{
			augCount_loo = add;
		}
		else
		{
			augCount_loo = count - 1 + add;
		}
		sum_loo += augCount_loo;

		/*
		 * Complete the denominator sum.
		 */

		for ( othm = 0; othm < M; othm ++ )
		{
			if ( othm != m )
			{
				Node *  othNodePtr;
				double  othCount;
				double  augOthCount;

				othNodePtr = nodePtrTab[ othm ];
				othCount = othNodePtr->count;
				augOthCount = othCount + add;

				sum += augOthCount;
				sum_loo += augOthCount;
			}
		}

		if ( sum <= 0.0 )
		{
			nodePtr->prob = 1.0 / ((double) M);
		}
		else
		{
			nodePtr->prob = augCount / sum;
		}

		if ( sum_loo <= 0.0 )
		{
			nodePtr->prob_loo = 1.0 / ((double) M);
		}
		else
		{
			nodePtr->prob_loo = augCount_loo / sum_loo;
		}
	}

	subTreeType = nodePtrTab[ 0 ]->subTreeType;
	if ( subTreeType != SubTreeType_NONE )
	{
		int  subTreeDelay;
		int  nAlt;
		int  altIdx;

		subTreeDelay = nodePtrTab[ 0 ]->subTreeDelay;
		nAlt = nodePtrTab[ 0 ]->nAlt;

		for ( m = 1; m < M; m ++ )
		{
			Node *  nodePtr;

			nodePtr = nodePtrTab[ m ];

			assert( nodePtr->nAlt == nAlt );
			assert( nodePtr->subTreeType == subTreeType );
			assert( nodePtr->subTreeDelay == subTreeDelay );
		}

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			Node **  subTreeNodePtrTab;

			subTreeNodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
			assert( subTreeNodePtrTab != (Node **) NULL );

			for ( m = 0; m < M; m ++ )
			{
				subTreeNodePtrTab[ m ] = &(nodePtrTab[ m ]->altTab[ altIdx ]);
			}

			spedo_n_0BUILD_add_recursive_helper
				(
					subTreeNodePtrTab,
					M,
					evalType,
					add
				);

			free( (void *) subTreeNodePtrTab );
		}
	}

	return;
}

void spedo_n_0BUILD_add
	(
		SPEDO_n_0 *     spedoPtr,
		const EvalType  evalType,
		const double    add
	)
{
	int      M;
	int      m;
	Node **  nodePtrTab;

	M = spedoPtr->M;

	nodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
	assert( nodePtrTab != (Node **) NULL );

	for ( m = 0; m < M; m ++ )
	{
		nodePtrTab[ m ] = &(spedoPtr->altTab[ m ]);
	}

	spedo_n_0BUILD_add_recursive_helper
		(
			nodePtrTab,
			M,
			evalType,
			add
		);

	free( (void *) nodePtrTab );

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0BUILD_addbo_recursive_helper
	(
		Node **         parentNodePtrTab,
		Node **         nodePtrTab,
		const int       M,
		const EvalType  evalType,
		const double    add
	)
{
	int          m;
	SubTreeType  subTreeType;

	for ( m = 0; m < M; m ++ )
	{
		Node *  nodePtr;
		int     count;
		double  sum;
		double  sum_loo;
		double  augCount;
		double  augCount_loo;
		int     othm;

		nodePtr = nodePtrTab[ m ];
		count = nodePtr->count;

		/*
		 * Compute the numerator (augCount) with all counts, and increment denominator (sum).
		 */

		sum = 0.0;
		if ( count <= 0 )
		{
			augCount = add;
		}
		else
		{
			augCount = count + add;
		}
		sum += augCount;

		/*
		 * Compute the numerator (augCount_loo) as if the observed number of counts were one fewer,
		 *  and increment denominator (sum_loo).
		 */

		sum_loo = 0.0;
		if ( count <= 1 )
		{
			augCount_loo = add;
		}
		else
		{
			augCount_loo = count - 1 + add;
		}
		sum_loo += augCount_loo;

		/*
		 * Complete the denominator sum.
		 */

		for ( othm = 0; othm < M; othm ++ )
		{
			if ( othm != m )
			{
				Node *  othNodePtr;
				double  othCount;
				double  augOthCount;

				othNodePtr = nodePtrTab[ othm ];
				othCount = othNodePtr->count;
				augOthCount = othCount + add;

				sum += augOthCount;
				sum_loo += augOthCount;
			}
		}

		if ( sum <= 0.0 )
		{
			nodePtr->prob = parentNodePtrTab[ m ]->prob;
		}
		else
		{
			nodePtr->prob = augCount / sum;
		}

		if ( sum_loo <= 0.0 )
		{
			nodePtr->prob_loo = parentNodePtrTab[ m ]->prob_loo;
		}
		else
		{
			nodePtr->prob_loo = augCount_loo / sum_loo;
		}
	}

	subTreeType = nodePtrTab[ 0 ]->subTreeType;
	if ( subTreeType != SubTreeType_NONE )
	{
		int  subTreeDelay;
		int  nAlt;
		int  altIdx;

		subTreeDelay = nodePtrTab[ 0 ]->subTreeDelay;
		nAlt = nodePtrTab[ 0 ]->nAlt;

		for ( m = 1; m < M; m ++ )
		{
			Node *  nodePtr;

			nodePtr = nodePtrTab[ m ];

			assert( nodePtr->nAlt == nAlt );
			assert( nodePtr->subTreeType == subTreeType );
			assert( nodePtr->subTreeDelay == subTreeDelay );
		}

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			Node **  subTreeNodePtrTab;

			subTreeNodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
			assert( subTreeNodePtrTab != (Node **) NULL );

			for ( m = 0; m < M; m ++ )
			{
				subTreeNodePtrTab[ m ] = &(nodePtrTab[ m ]->altTab[ altIdx ]);
			}

			spedo_n_0BUILD_addbo_recursive_helper
				(
					nodePtrTab,
					subTreeNodePtrTab,
					M,
					evalType,
					add
				);

			free( (void *) subTreeNodePtrTab );
		}
	}

	return;
}

void spedo_n_0BUILD_addbo
	(
		SPEDO_n_0 *     spedoPtr,
		const EvalType  evalType,
		const double    add
	)
{
	int      M;
	int      m;
	Node **  unigramNodePtrTab;
	Node **  nodePtrTab;

	M = spedoPtr->M;

	unigramNodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
	assert( unigramNodePtrTab != (Node **) NULL );

	for ( m = 0; m < M; m ++ )
	{
		Node *  unigramNodePtr;

		unigramNodePtr = (Node *) malloc( sizeof( Node ) );
		assert( unigramNodePtr != (Node *) NULL );

		unigramNodePtr->subTreeType = SubTreeType_NONE;
		unigramNodePtr->subTreeDelay = 0;
		unigramNodePtr->count = 0;
		unigramNodePtr->prob = 1.0 / ((double) M);
		unigramNodePtr->prob_loo = unigramNodePtr->prob;
		unigramNodePtr->nAlt = 0;
		unigramNodePtr->altTab = (Node *) NULL;

		unigramNodePtrTab[ m ] = unigramNodePtr;
	}

	nodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
	assert( nodePtrTab != (Node **) NULL );

	for ( m = 0; m < M; m ++ )
	{
		nodePtrTab[ m ] = &(spedoPtr->altTab[ m ]);
	}

	spedo_n_0BUILD_addbo_recursive_helper
		(
			unigramNodePtrTab,
			nodePtrTab,
			M,
			evalType,
			add
		);

	for ( m = 0; m < M; m ++ )
	{
		free( (void *) (unigramNodePtrTab[ m ]) );
	}
	free( (void *) unigramNodePtrTab );
	free( (void *) nodePtrTab );

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static void spedo_n_0BUILD_ml_recursive_helper
	(
		Node **       nodePtrTab,
		const int     M,
		const int     evalType
	)
{
	int          m;
	SubTreeType  subTreeType;

	for ( m = 0; m < M; m ++ )
	{
		Node *  nodePtr;
		int     count;
		int     count_loo;
		double  sum;
		double  sum_loo;
		int     othm;

		nodePtr = nodePtrTab[ m ];
		count = nodePtr->count;

		/*
		 * Compute the numerator (count) with all counts, and increment denominator (sum).
		 */

		sum = 0.0;
		sum += count;

		/*
		 * Compute the numerator (count_loo) as if the observed number of counts were one fewer,
		 *  and increment denominator (sum_loo).
		 */

		sum_loo = 0.0;
		if ( count <= 1 )
		{
			count_loo = 0;
		}
		else
		{
			count_loo = count - 1;
		}
		sum_loo += count_loo;

		/*
		 * Complete the denominator sum.
		 */

		for ( othm = 0; othm < M; othm ++ )
		{
			if ( othm != m )
			{
				Node *  othNodePtr;
				double  othCount;

				othNodePtr = nodePtrTab[ othm ];
				othCount = othNodePtr->count;

				sum += othCount;
				sum_loo += othCount;
			}
		}

		nodePtr->prob = count / sum;
		nodePtr->prob_loo = count_loo / sum_loo;
	}

	subTreeType = nodePtrTab[ 0 ]->subTreeType;
	if ( subTreeType != SubTreeType_NONE )
	{
		int  subTreeDelay;
		int  nAlt;
		int  altIdx;

		subTreeDelay = nodePtrTab[ 0 ]->subTreeDelay;
		nAlt = nodePtrTab[ 0 ]->nAlt;

		for ( m = 1; m < M; m ++ )
		{
			Node *  nodePtr;

			nodePtr = nodePtrTab[ m ];

			assert( nodePtr->nAlt == nAlt );
			assert( nodePtr->subTreeType == subTreeType );
			assert( nodePtr->subTreeDelay == subTreeDelay );
		}

		for ( altIdx = 0; altIdx < nAlt; altIdx ++ )
		{
			Node **  subTreeNodePtrTab;

			subTreeNodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
			assert( subTreeNodePtrTab != (Node **) NULL );

			for ( m = 0; m < M; m ++ )
			{
				subTreeNodePtrTab[ m ] = &(nodePtrTab[ m ]->altTab[ altIdx ]);
			}

			spedo_n_0BUILD_ml_recursive_helper
				(
					subTreeNodePtrTab,
					M,
					evalType
				);

			free( (void *) subTreeNodePtrTab );
		}
	}

	return;
}

void spedo_n_0BUILD_ml
	(
		SPEDO_n_0 *     spedoPtr,
		const EvalType  evalType
	)
{
	int      M;
	int      m;
	Node **  nodePtrTab;

	M = spedoPtr->M;

	nodePtrTab = (Node **) malloc( M * sizeof( Node * ) );
	assert( nodePtrTab != (Node **) NULL );

	for ( m = 0; m < M; m ++ )
	{
		nodePtrTab[ m ] = &(spedoPtr->altTab[ m ]);
	}

	spedo_n_0BUILD_ml_recursive_helper
		(
			nodePtrTab,
			M,
			evalType
		);

	free( (void *) nodePtrTab );

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

static double spedo_n_0PROB_recursive_helper
	(
		Node *               nodePtr,
		const int            depth,
		const int            level,
		const SubTreeType *  subTreeTypeTab,
		const int *          subTreeDelayTab,
		const int            KmaxMinusOne,
		const int            K,
		const int *          q,
		const int            t,
		const int            k,
		const EvalType       evalType
	)
{
	double  prob;

	if ( level < depth )
	{
		SubTreeType  subTreeType;
		int          subTreeDelay;
		int          tau;
		int          q_tau;
		int          s;
		int          n;
		int          altIdx;

		subTreeType = subTreeTypeTab[ level ];
		subTreeDelay = subTreeDelayTab[ level ];

		tau = t - subTreeDelay;
		if ( tau < 0 )
		{
			q_tau = 0;
		}
		else
		{
			q_tau = q[ tau ];
		}

		s = GETBIT( q_tau, k );
		n = CNTBIT( q_tau, K ) - s;
		if ( n > KmaxMinusOne )
		{
			n = KmaxMinusOne;
		}

		altIdx = -1;
		switch( subTreeType )
		{
		case SubTreeType_Self:
			altIdx = s;
			break;
		case SubTreeType_NOthers:
			altIdx = n;
			break;
		case SubTreeType_NONE:
		default:
			assert( 0 );
		}

		prob = spedo_n_0PROB_recursive_helper
			(
				&(nodePtr->altTab[ altIdx ]),
				depth,
				level + 1,
				subTreeTypeTab,
				subTreeDelayTab,
				KmaxMinusOne,
				K,
				q,
				t,
				k,
				evalType
			);
	}
	else
	{
		switch( evalType )
		{
		case EvalType_ALL:
			prob = nodePtr->prob;
			break;
		case EvalType_LOO:
			prob = nodePtr->prob_loo;
			break;
		default:
			assert( 0 );
		}
	}

	return prob;
}

double spedo_n_0PROB
	(
		const SPEDO_n_0 *  spedoPtr,
		const int          K,
		const int *        q,
		const int          t,
		const EvalType     evalType
	)
{
	int     N;
	int     delayMax;
	int     KmaxMinusOne;
	int     tau;
	double  prob;
	int     k;

	assert( spedoPtr->M == 2 ); /* due to encoding of q */

	N = (int) pow( (double) (spedoPtr->M), (double) K );

	delayMax = spedoPtr->delayMax;
	for ( tau = (t - delayMax); tau <= t; tau ++ )
	{
		if  ( tau < 0 )
		{
			/* ignore */
		}
		else
		{
			int  q_tau;

			q_tau = q[ tau ];

			assert( ( q_tau >= 0 ) && ( q_tau < N ) );
		}
	}

	KmaxMinusOne = spedoPtr->Kmax - 1;

	prob = 1.0;

	for ( k = 0; k < K; k ++ )
	{
		int     s0;
		double  kProb;

		s0 = GETBIT( q[ t ], k );
		assert( s0 < spedoPtr->M );

		kProb = spedo_n_0PROB_recursive_helper
			(
				&(spedoPtr->altTab[ s0 ]),
				spedoPtr->depth,
				0,
				spedoPtr->subTreeTypeTab,
				spedoPtr->subTreeDelayTab,
				KmaxMinusOne,
				K,
				q,
				t,
				k,
				evalType
			);

		prob *= kProb;
	}
/*
	if ( prob == 0.0 )
	{
		prob = +DBL_MIN;
	}
*/
	return prob;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_n_0PREDSTOCHASTIC
	(
		const SPEDO_n_0 *  spedoPtr,
		const int          K,
		const int *        q,
		const int          t,
		const EvalType     evalType
	)
{
	int  M;
	int  N;
	int  delayMax;
	int  KmaxMinusOne;
	int  tau;
	int  pred_q_0;
	int  k;

	assert( spedoPtr->M == 2 ); /* due to encoding of q */

	M = spedoPtr->M;
	N = (int) pow( (double) M, (double) K );

	delayMax = spedoPtr->delayMax;
	for ( tau = (t - delayMax); tau < t; tau ++ )
	{
		if ( tau < 0 )
		{
			/* ignore */
		}
		else
		{
			int  q_tau;

			q_tau = q[ tau ];

			assert( ( q_tau >= 0 ) && ( q_tau < N ) );
		}
	}

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		int  pred_s0;

		pred_s0 = -1;
		while ( pred_s0 == -1 )
		{
			double  uniformDeviate;
			double  lowerCumulativeBound;
			int     s0;

			uniformDeviate = ran1( idum );

			lowerCumulativeBound = 0.0; 
			for ( s0 = 0; s0 < M; s0 ++ )
			{
				double  upperCumulativeBound;

				upperCumulativeBound = lowerCumulativeBound
					+ spedo_n_0PROB_recursive_helper
						(
							&(spedoPtr->altTab[ s0 ]),
							spedoPtr->depth,
							0,
							spedoPtr->subTreeTypeTab,
							spedoPtr->subTreeDelayTab,
							KmaxMinusOne,
							K,
							q,
							t,
							k,
							evalType
						);

				if ( uniformDeviate < upperCumulativeBound )
				{
					if ( uniformDeviate >= lowerCumulativeBound )
					{
						pred_s0 = s0;
						break;
					}
				}
			}
		}

		/* REMEMBER THAT IF PP IS Inf, THIS IS NOT RELIABLE */
		if ( pred_s0 > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_n_0PREDDETERMINISTIC
	(
		const SPEDO_n_0 *  spedoPtr,
		const int          K,
		const int *        q,
		const int          t,
		const EvalType     evalType
	)
{
	int  M;
	int  N;
	int  delayMax;
	int  KmaxMinusOne;
	int  tau;
	int  pred_q_0;
	int  k;

	M = spedoPtr->M;
	assert( M == 2 ); /* only because of the encoding of q */

	N = (int) pow( (double) M, (double) K );

	delayMax = spedoPtr->delayMax;
	for ( tau = (t - delayMax); tau < t; tau ++ )
	{
		if ( tau < 0 )
		{
			/* ignore */
		}
		else
		{
			int  q_tau;

			q_tau = q[ tau ];

			assert( ( q_tau >= 0 ) && ( q_tau < N ) );
		}
	}

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		double  pred_s0Prob;
		int     pred_s0;
		int     s0;

		pred_s0Prob = 0.0;
		pred_s0 = -1;
		for ( s0 = 0; s0 < M; s0 ++ )
		{
			double  prob;

			prob = spedo_n_0PROB_recursive_helper
				(
					&(spedoPtr->altTab[ s0 ]),
					spedoPtr->depth,
					0,
					spedoPtr->subTreeTypeTab,
					spedoPtr->subTreeDelayTab,
					KmaxMinusOne,
					K,
					q,
					t,
					k,
					evalType
				);

			if ( prob > pred_s0Prob )
			{
				pred_s0Prob = prob;
				pred_s0 = s0;
			} 
		}

		/* -- NO WAY TO TRACK THIS RIGHT NOW -- 
		assert( pred_s0 > -1 );
		*/

		if ( pred_s0 > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

