/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdio.h>

#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <float.h>
#include "bit.h"
#include "ran1.h"
#include "spedo_3_0.h"

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_3_0ALLOC
	(
		SPEDO_3_0 *  spedoPtr,
		const int    M,
		const int    Kmax
	)
{
	double *****  a;
	int           n2;

	assert( Kmax > 0 );
	assert( M > 1 );

	a = (double *****) malloc( Kmax * sizeof( double **** ) );
	assert( a != (double *****) NULL );

	for ( n2 = 0; n2 < Kmax; n2 ++ )
	{
		int  s2;

		a[ n2 ] = (double ****) malloc( M * sizeof( double *** ) );
		assert( a[ n2 ] != (double ****) NULL );

		for ( s2 = 0; s2 < M; s2 ++ )
		{
			int  n1;

			a[ n2 ][ s2 ] = (double ***) malloc( Kmax * sizeof( double ** ) );
			assert( a[ n2 ][ s2 ] != (double ***) NULL );

			for ( n1 = 0; n1 < Kmax; n1 ++ )
			{
				int  s1;

				a[ n2 ][ s2 ][ n1 ] = (double **)
					malloc( M * sizeof( double * ) );
				assert( a[ n2 ][ s2 ][ n1 ] != (double **) NULL );

				for ( s1 = 0; s1 < M; s1 ++ )
				{
					a[ n2 ][ s2 ][ n1 ][ s1 ] = (double *)
						malloc( M * sizeof( double ) );
					assert( a[ n2 ][ s2 ][ n1 ][ s1 ] != (double *) NULL );
				}
			}
		}
	}

	spedoPtr->M = M;
	spedoPtr->Kmax = Kmax;
	spedoPtr->a = a;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_3_0FREE
	(
		SPEDO_3_0 *  spedoPtr
	)
{
	int  Kmax;
	int  M;
	int  n2;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;

	for ( n2 = 0; n2 < Kmax; n2 ++ )
	{
		int  s2;

		for  ( s2 = 0; s2 < M; s2 ++ )
		{
			int  n1;

			for ( n1 = 0; n1 < Kmax; n1 ++ )
			{
				int  s1;

				for ( s1 = 0; s1 < M; s1 ++ )
				{
					free( (void *) (spedoPtr->a[ n2 ][ s2 ][ n1 ][ s1 ]) );
				}

				free( (void *) (spedoPtr->a[ n2 ][ s2 ][ n1 ]) );
			}

			free( (void *) (spedoPtr->a[ n2 ][ s2 ]) );
		}

		free( (void *) (spedoPtr->a[ n2 ]) );
	}

	free( (void *) (spedoPtr->a) );

	return;

	spedoPtr->Kmax = 0;
	spedoPtr->M = 0;
	spedoPtr->a = (double *****) NULL;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  spedo_3_0CLEAR
	(
		SPEDO_3_0 *  spedoPtr
	)
{
	int           Kmax;
	int           M;
	double *****  a;
	int           n2;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;
	a = spedoPtr->a;

	for ( n2 = 0; n2 < Kmax; n2 ++ )
	{
		int  s2;

		for ( s2 = 0; s2 < M; s2 ++ )
		{
			int  n1;

			for ( n1 = 0; n1 < Kmax; n1 ++ )
			{
				int  s1;

				for ( s1 = 0; s1 < M; s1 ++ )
				{
					int  s0;

					for ( s0 = 0; s0 < M; s0 ++ )
					{
						a[ n2 ][ s2 ][ n1 ][ s1 ][ s0 ] = 0.0;
					}
				}
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  spedo_3_0ACCUM
	(
		SPEDO_3_0 *  spedoPtr,
		const int    K,
		const int    q_2,
		const int    q_1,
		const int    q_0
	)
{
	int  N;
	int  KmaxMinusOne;
	int  k;

	assert( spedoPtr->M == 2 );

	N = (int) pow( (double) (spedoPtr->M), (double) K );
	assert( ( q_2 >= 0 ) && ( q_2 < N ) );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	assert( ( q_0 >= 0 ) && ( q_0 < N ) );

	KmaxMinusOne = spedoPtr->Kmax - 1;

	for ( k = 0; k < K; k ++ )
	{
		int  n2;
		int  s2;
		int  n1;
		int  s1;
		int  s0;

		s2 = GETBIT( q_2, k );
		n2 = CNTBIT( q_2, K ) - s2;
		if  ( n2 > KmaxMinusOne )
		{
			n2 = KmaxMinusOne;
		}

		s1 = GETBIT( q_1, k );
		n1 = CNTBIT( q_1, K ) - s1;
		if ( n1 > KmaxMinusOne )
		{
			n1 = KmaxMinusOne;
		}

		s0 = GETBIT( q_0, k );

		spedoPtr->a[ n2 ][ s2 ][ n1 ][ s1 ][ s0 ] += 1.0;
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_3_0BUILD
	(
		SPEDO_3_0 *   spedoPtr,
		const double  unkFraction
	)
{
	int           Kmax;
	int           M;
	double *****  a;
	int           n2;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;
	a = spedoPtr->a;

	for ( n2 = 0; n2 < Kmax; n2 ++ )
	{
		double  ****  a_n2_;
		int           s2;

		a_n2_ = a[ n2 ];

		for ( s2 = 0; s2 < M; s2 ++ )
		{
			double ***  a_n2_s2_;
			int         n1;

			a_n2_s2_ = a_n2_[ s2 ];

			for ( n1 = 0; n1 < Kmax; n1 ++ )
			{
				double **  a_n2_s2_n1_;
				int        s1;

				a_n2_s2_n1_ = a_n2_s2_[ n1 ];

				for ( s1 = 0; s1 < M; s1 ++ )
				{
					double *  a_n2_s2_n1_s1_;
					double    sum;
					int       s0;
					int       nZero;

					a_n2_s2_n1_s1_ = a_n2_s2_n1_[ s1 ];

					sum = 0.0;
					for ( s0 = 0; s0 < M; s0 ++ )
					{
						sum += a_n2_s2_n1_s1_[ s0 ];
					}

					if ( sum == 0.0 )
					{
						for ( s0 = 0; s0 < M; s0 ++ )
						{
							sum += (a_n2_s2_n1_s1_[ s0 ] = 1.0);
						}
					}

					for ( s0 = 0; s0 < M; s0 ++ )
					{
						a_n2_s2_n1_s1_[ s0 ] /= sum;
					}

					nZero = 0;
					for ( s0 = 0; s0 < M; s0 ++ )
					{
						if ( a_n2_s2_n1_s1_[ s0 ] == 0.0 )
						{
							nZero ++;
						}
					} 

					if ( nZero > 0 )
					{
						for ( s0 = 0; s0 < M; s0 ++ )
						{
							if ( a_n2_s2_n1_s1_[ s0 ] == 0.0 )
							{
								a_n2_s2_n1_s1_[ s0 ] = unkFraction / ((double) nZero);
							}
							else
							{
								a_n2_s2_n1_s1_[ s0 ] *= (1.0 - unkFraction);
							}
						}
					}

					sum = 0.0;
					for ( s0 = 0; s0 < M; s0 ++ )
					{
						sum += a_n2_s2_n1_s1_[ s0 ];
					}
					assert( sum > 0.0 );

					for ( s0 = 0; s0 < M; s0 ++ )
					{
						a_n2_s2_n1_s1_[ s0 ] /= sum;
					}
				}
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

double spedo_3_0PROB
	(
		const SPEDO_3_0 *  spedoPtr,
		const int          K,
		const int          q_2,
		const int          q_1,
		const int          q_0
	)
{
	int     N;
	int     KmaxMinusOne;
	double  prob;
	int     k;

	assert( spedoPtr->M == 2 );

	N = (int) pow( (double) (spedoPtr->M), (double) K );
	assert( ( q_2 >= 0 ) && ( q_2 < N ) );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	assert( ( q_0 >= 0 ) && ( q_0 < N ) );

	KmaxMinusOne = spedoPtr->Kmax - 1;

	prob = 1.0;

	for ( k = 0; k < K; k ++ )
	{
		int  n2;
		int  s2;
		int  n1;
		int  s1;
		int  s0;

		s2 = GETBIT( q_2, k );
		n2 = CNTBIT( q_2, K ) - s2;
		if ( n2 > KmaxMinusOne )
		{
			n2 = KmaxMinusOne;
		}

		s1 = GETBIT( q_1, k );
		n1 = CNTBIT( q_1, K ) - s1;
		if ( n1 > KmaxMinusOne )
		{
			n1 = KmaxMinusOne;
		}

		s0 = GETBIT( q_0, k );

		prob *= spedoPtr->a[ n2 ][ s2 ][ n1 ][ s1 ][ s0 ];
	}

	assert( prob > 0.0 );

	return prob;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_3_0PREDSTOCHASTIC
	(
		const SPEDO_3_0 *  spedoPtr,
		const int          K,
		const int          q_2,
		const int          q_1
	)
{
	int           M;
	double *****  a;
	int           N;
	int           KmaxMinusOne;
	int           pred_q_0;
	int           k;

	M = spedoPtr->M;
	assert( M == 2 );
	a = spedoPtr->a;

	N = (int) pow( (double) M, (double) K );
	assert( ( q_2 >= 0 ) && ( q_2 < N ) );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	a = spedoPtr->a;

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		int       n2;
		int       s2;
		int       n1;
		int       s1;
		double *  a_n2_s2_n1_s1_;
		int       pred_s0;

		s2 = GETBIT( q_2, k );
		n2 = CNTBIT( q_2, K ) - s2;
		if ( n2 > KmaxMinusOne )
		{
			n2 = KmaxMinusOne;
		}

		s1 = GETBIT( q_1, k );
		n1 = CNTBIT( q_1, K ) - s1;
		if ( n1 > KmaxMinusOne )
		{
			n1 = KmaxMinusOne;
		}

		a_n2_s2_n1_s1_ = a[ n2 ][ s2 ][ n1 ][ s1 ];

		pred_s0 = -1;
		while ( pred_s0 == -1 )
		{
			double  uniformDeviate;
			double  lowerCumulativeBound;
			int     s0;

			uniformDeviate = ran1( idum );

			lowerCumulativeBound = 0.0; 
			for ( s0 = 0; s0 < M; s0 ++ )
			{
				double  upperCumulativeBound;

				upperCumulativeBound = lowerCumulativeBound + a_n2_s2_n1_s1_[ s0 ];

				if ( uniformDeviate < upperCumulativeBound )
				{
					if ( uniformDeviate >= lowerCumulativeBound )
					{
						pred_s0 = s0;
						break;
					}
				}
			}
		}

		if ( pred_s0 > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}
 
/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_3_0PREDDETERMINISTIC
	(
		const SPEDO_3_0 *  spedoPtr,
		const int          K,
		const int          q_2,
		const int          q_1
	)
{
	int           M;
	double *****  a;
	int           N;
	int           KmaxMinusOne;
	int           pred_q_0;
	int           k;

	M = spedoPtr->M;
	assert( M == 2 );
	a = spedoPtr->a;

	N = (int) pow( (double) M, (double) K );
	assert( ( q_2 >= 0 ) && ( q_2 < N ) );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	a = spedoPtr->a;

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		int       n2;
		int       s2;
		int       n1;
		int       s1;
		double *  a_n2_s2_n1_s1_;
		double    pred_s0Prob;
		int       pred_s0;
		int       s0;

		s2 = GETBIT( q_2, k );
		n2 = CNTBIT( q_2, K ) - s2;
		if ( n2 > KmaxMinusOne )
		{
			n2 = KmaxMinusOne;
		}

		s1 = GETBIT( q_1, k );
		n1 = CNTBIT( q_1, K ) - s1;
		if ( n1 > KmaxMinusOne )
		{
			n1 = KmaxMinusOne;
		}

		a_n2_s2_n1_s1_ = a[ n2 ][ s2 ][ n1 ][ s1 ];

		pred_s0Prob = -DBL_MIN;
		pred_s0 = -1;
		for ( s0 = 0; s0 < M; s0 ++ )
		{
			double  s0Prob;

			s0Prob = a_n2_s2_n1_s1_[ s0 ];

			if ( s0Prob > pred_s0Prob )
			{
				pred_s0Prob = s0Prob;
				pred_s0 = s0;
			}
		}

		assert( pred_s0 > -1 );

		if ( pred_s0 > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

