/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <float.h>
#include "bit.h"
#include "ran1.h"
#include "spedo_2_0.h"

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_2_0ALLOC
	(
		SPEDO_2_0 *  spedoPtr,
		const int    M,
		const int    Kmax
	)
{
	double ***  a;
	int ***     count;
	int         ni;

	assert( Kmax > 0 );
	assert( M > 1 );

	a = (double ***) malloc( Kmax * sizeof( double ** ) );
	assert( a != (double ***) NULL );

	count = (int ***) malloc( Kmax * sizeof( int ** ) );
	assert( count != (int ***) NULL );

	for ( ni = 0; ni < Kmax; ni ++ )
	{
		int  m;

		a[ ni ] = (double **) malloc( M * sizeof( double * ) );
		assert( a[ ni ] != (double **) NULL );

		count[ ni ] = (int **) malloc( M * sizeof( int * ) );
		assert( count[ ni ] != (int **) NULL );

		for ( m = 0; m < M; m ++ )
		{
			a[ ni ][ m ] = (double *) malloc( M * sizeof( double ) );
			assert( a[ ni ][ m ] != (double *) NULL );

			count[ ni ][ m ] = (int *) malloc( M * sizeof( int ) );
			assert( count[ ni ][ m ] != (int *) NULL );
		}
	}

	spedoPtr->M = M;
	spedoPtr->Kmax = Kmax;
	spedoPtr->a = a;
	spedoPtr->count = count;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_2_0FREE
	(
		SPEDO_2_0 *  spedoPtr
	)
{
	int  Kmax;
	int  M;
	int  ni;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;

	for ( ni = 0; ni < Kmax; ni ++ )
	{
		int  m;

		for  ( m = 0; m < M; m ++ )
		{
			free( (void *) (spedoPtr->a[ ni ][ m ]) );
			free( (void *) (spedoPtr->count[ ni ][ m ]) );
		}

		free( (void *) (spedoPtr->a[ ni ]) );
		free( (void *) (spedoPtr->count[ ni ]) );
	}

	free( (void *) (spedoPtr->a) );
	free( (void *) (spedoPtr->count) );

	return;

	spedoPtr->Kmax = 0;
	spedoPtr->M = 0;
	spedoPtr->a = (double ***) NULL;
	spedoPtr->count = (int ***) NULL;

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  spedo_2_0CLEAR
	(
		SPEDO_2_0 *  spedoPtr
	)
{
	int         Kmax;
	int         M;
	double ***  a;
	int         ni;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;
	a = spedoPtr->a;

	for ( ni = 0; ni < Kmax; ni ++ )
	{
		int  m;

		for ( m = 0; m < M; m ++ )
		{
			int  n;

			for ( n = 0; n < M; n ++ )
			{
				a[ ni ][ m ][ n ] = 0.0;
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void  spedo_2_0ACCUM
	(
		SPEDO_2_0 *  spedoPtr,
		const int    K,
		const int    q_1,
		const int    q_0
	)
{
	int  N;
	int  KmaxMinusOne;
	int  k;

	assert( spedoPtr->M == 2 );

	N = (int) pow( (double) (spedoPtr->M), (double) K );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	assert( ( q_0 >= 0 ) && ( q_0 < N ) );

	KmaxMinusOne = spedoPtr->Kmax - 1;

	for ( k = 0; k < K; k ++ )
	{
		int  m;
		int  n;
		int  ni;

		m = GETBIT( q_1, k );
		n = GETBIT( q_0, k );

		ni = CNTBIT( q_1, K ) - m;

		if ( ni > KmaxMinusOne )
		{
			ni = KmaxMinusOne;
		}

		spedoPtr->a[ ni ][ m ][ n ] += 1.0;
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

void spedo_2_0BUILD
	(
		SPEDO_2_0 *   spedoPtr,
		const double  unkFraction
	)
{
	int         Kmax;
	int         M;
	double ***  a;
	int         ni;

	Kmax = spedoPtr->Kmax;
	M = spedoPtr->M;
	a = spedoPtr->a;

	for ( ni = 0; ni < Kmax; ni ++ )
	{
		double **  a_ni_;
		int        m;

		a_ni_ = a[ ni ];

		for ( m = 0; m < M; m ++ )
		{
			double *  a_ni_m_;
			double    sum;
			int       n;
			int       nZero;

			a_ni_m_ = a_ni_[ m ];

			sum = 0.0;
			for ( n = 0; n < M; n ++ )
			{
				spedoPtr->count[ ni ][ m ][ n ] = a[ ni ][ m ][ n ];
				sum += a_ni_m_[ n ];
			}

			if ( sum == 0.0 )
			{
				for ( n = 0; n < M; n ++ )
				{
					sum += ( a_ni_m_[ n ] = 1.0 );
				}
			}

			for ( n = 0; n < M; n ++ )
			{
				a_ni_m_[ n ] /= sum;
			}

			nZero = 0;
			for ( n = 0; n < M; n ++ )
			{
				if ( a_ni_m_[ n ] == 0.0 )
				{
					nZero ++;
				}
			}

			if ( nZero > 0 )
			{
				for ( n = 0; n < M; n ++ )
				{
					if ( a_ni_m_[ n ] == 0.0 )
					{
						a_ni_m_[ n ] = unkFraction / ((double) nZero);
					}
					else
					{
						a_ni_m_[ n ] *= (1.0 - unkFraction);
					}
				}
				
			}

			sum = 0.0;
			for ( n = 0; n < M; n ++ )
			{
				sum += a_ni_m_[ n ];
			}
			assert( sum > 0.0 );

			for ( n = 0; n < M; n ++ )
			{
				a_ni_m_[ n ] /= sum;
			}
		}
	}

	return;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

double spedo_2_0PROB
	(
		const SPEDO_2_0 *  spedoPtr,
		const int          K,
		const int          q_1,
		const int          q_0
	)
{
	int     N;
	int     KmaxMinusOne;
	double  prob;
	int     k;

	assert( spedoPtr->M == 2 );

	N = (int) pow( (double) (spedoPtr->M), (double) K );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	assert( ( q_0 >= 0 ) && ( q_0 < N ) );

	KmaxMinusOne = spedoPtr->Kmax - 1;

	prob = 1.0;

	for ( k = 0; k < K; k ++ )
	{
		int     m;
		int     n;
		int     ni;
		double  kProb;

		m = GETBIT( q_1, k );
		n = GETBIT( q_0, k );

		ni = CNTBIT( q_1, K ) - m;

		if ( ni > KmaxMinusOne )
		{
			ni = KmaxMinusOne;
		}

		kProb = spedoPtr->a[ ni ][ m ][ n ];

		prob *= kProb;
	}

	return prob;
}

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_2_0PREDSTOCHASTIC
	(
		const SPEDO_2_0 *  spedoPtr,
		const int          K,
		const int          q_1
	)
{
	int         M;
	double ***  a;
	int         N;
	int         KmaxMinusOne;
	int         pred_q_0;
	int         k;

	M = spedoPtr->M;
	assert( M == 2 );
	a = spedoPtr->a;

	N = (int) pow( (double) M, (double) K );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	a = spedoPtr->a;

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		int       m;
		int       ni;
		double *  a_ni_m_;
		int       pred_n;

		m = GETBIT( q_1, k );
		ni = CNTBIT( q_1, K ) - m;
		if ( ni > KmaxMinusOne )
		{
			ni = KmaxMinusOne;
		}

		a_ni_m_ = a[ ni ][ m ];

		pred_n = -1;
		while ( pred_n == -1 )
		{
			double  uniformDeviate;
			double  lowerCumulativeBound;
			int     n;

			uniformDeviate = ran1( idum );

			lowerCumulativeBound = 0.0; 
			for ( n = 0; n < M; n ++ )
			{
				double  upperCumulativeBound;

				upperCumulativeBound = lowerCumulativeBound + a_ni_m_[ n ];

				if ( uniformDeviate < upperCumulativeBound )
				{
					if ( uniformDeviate >= lowerCumulativeBound )
					{
						pred_n = n;
						break;
					}
				}
			}
		}

		if ( pred_n > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}
 
/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

int spedo_2_0PREDDETERMINISTIC
	(
		const SPEDO_2_0 *  spedoPtr,
		const int          K,
		const int          q_1
	)
{
	int         M;
	double ***  a;
	int         N;
	int         KmaxMinusOne;
	int         pred_q_0;
	int         k;

	M = spedoPtr->M;
	assert( M == 2 );
	a = spedoPtr->a;

	N = (int) pow( (double) M, (double) K );
	assert( ( q_1 >= 0 ) && ( q_1 < N ) );
	a = spedoPtr->a;

	KmaxMinusOne = spedoPtr->Kmax - 1;

	pred_q_0 = 0;

	for ( k = 0; k < K; k ++ )
	{
		int       m;
		int       ni;
		double *  a_ni_m_;
		double    pred_nProb;
		int       pred_n;
		int       n;

		m = GETBIT( q_1, k );
		ni = CNTBIT( q_1, K ) - m;
		if ( ni > KmaxMinusOne )
		{
			ni = KmaxMinusOne;
		}

		a_ni_m_ = a[ ni ][ m ];

		pred_nProb = -DBL_MIN;
		pred_n = -1;
		for ( n = 0; n < M; n ++ )
		{
			double  nProb;

			nProb = a_ni_m_[ n ];

			if ( nProb > pred_nProb )
			{
				pred_nProb = nProb;
				pred_n = n;
			}
		}

		assert( pred_n > -1 );

		if ( pred_n > 0 )
		{
			SETBIT( pred_q_0, k );
		}
	}

	return pred_q_0;
}

/* =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

