/*
 * =+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+====0
 
Copyright (c) 2003-2010, Kornel Laskowski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted
provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of
      conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of
      conditions and the following disclaimer in the documentation and/or other materials provided
      with the distribution.
    * Neither the name of Sigtactica Research or of the Royal Institute of Technology (KTH) or of
      Carnegie Mellon University nor the names of their contributors may be used to endorse or
      promote products derived from this software without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ===+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+== */

#include <float.h>
#include "ran1.h"

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define RNMX (1.0-DBL_EPSILON) 

/* -+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+-- */

long   idumValue;
long * idum = &idumValue;

/*==+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+===*/

double ran1(long *idum)
	/*
	 * Minimal random number generator of Park and Miller with
	 * Bays-Durham shuffle and added safeguards. Returns a uniform
	 * random deviate between 0.0 and 1.0 (exclusive of the endpoint
	 * values). Call with idum a negative integer to initialize;
	 * thereafter, do not * alter idum between successive deviates in
	 * a sequence. RNMX should approximate the largest floating value
	 * that is less than 1.
	 */
{
	int j;
	long k;
	static long iy=0;
	static long iv[NTAB];
	double temp;

	if (*idum <= 0 || !iy) {           /* Initialize. */
		if (-(*idum) < 1) *idum=1; /* Be sure to prevent idum=0. */
		else *idum = -(*idum);
		for (j=NTAB+7;j>=0;j--) {  /* Load the suffle table */
			k=(*idum)/IQ;      /*  (after 8 warm-ups). */
			*idum=IA*(*idum-k*IQ)-IR*k;
			if (*idum < 0) *idum += IM;
			if (j < NTAB) iv[j] = *idum;
		}
		iy=iv[0];
	}
	k=(*idum)/IQ;               /* Start here when not initializing. */
	*idum=IA*(*idum-k*IQ)-IR*k; /* Compute idum=(IA*idum)%IM without */
	if (*idum < 0) *idum += IM; /*  overflows by Schrage's method. */
	j=iy/NDIV;                  /* Will be in the range 0..NTAB-1. */
	iy=iv[j];                   /* Output previously stored value */
	iv[j] = *idum;              /*  and refill the shuffle table. */
	if ((temp=AM*iy) > RNMX) return RNMX; /* Because users don't */ 
	else return temp;           /*  expect endpoint values. */ 
}

/*==+====1====+====2====+====3====+====4====+====5====+====6====+====7====+====8====+====9====+===*/

