function browse_hits(bounds_hits,im,params)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% Interactive browsing of the hypotheses generated by the detection
%%% process. Use ? for instructions.
%%%
%%% PARAMETERS:
%%%    bounds_hits:  A hit_list of the predicted bounding boxes (usually a
%%%                  return value of detect_objects_in_image)
%%%
%%% OPTIONAL PARAMETERS:
%%%    im:           If the test images are not enrolled (i.e. 
%%%                  do not appear in the global 'im' variable, you will
%%%                  need to create an im for them. See the file
%%%                  demo_poselets.m for how to do this.
%%%    params:       Extra parameters, such as the list of detected
%%%                  poselets, torsos (for the person category), poselet
%%%                  masks and example poselets to help visualization. See
%%%                  demo_poselets.m for more.
%%%
%%% Copyright (C) 2009, Lubomir Bourdev and Jitendra Malik.
%%% This code is distributed with a non-commercial research license.
%%% Please see the license file license.txt included in the source directory.
%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ~exist('im','var') || isempty(im)
   global im; 
end    
if ~exist('params','var') || ~isfield(params,'verbose')
   params.verbose=false; 
end
    
if ~params.verbose
    disp('press ''?'' for instructions');
end

figures=[1];
img_id=1;
mode=1;
ch='b';
while 1
    figure(figures(1)); clf;
    
    switch mode
        case 1
            % browse through images, display bounding boxes of objects
            params.str='visible bounds';
            [ch,img_id] = browse_hit_images(bounds_hits,im,@visualize_hypothesis,[],params,img_id);
        case 2
            % browse through images, display torsos of people, if available            
            params.str='torsos of people';
            [ch,img_id] = browse_hit_images(params.all_torso_hits,im,@visualize_hypothesis,[],params,img_id);
        case 3
            % browse through images, display poselets sorted by score      
            params.str='poselet activations';
            [ch,img_id] = browse_hit_images(params.all_poselet_hits,im,[],@draw_poselet_hits,params,img_id);
        case 4
            % browse through a grid of bounding boxes of objects sorted by score    
            if ~isfield(params,'all_poselet_hits')
                ch = browse_hit_patches(bounds_hits,im);
            else
                params.str='poselet patches';
                ch = browse_hit_patches(bounds_hits,im,@visualize_hypothesis,params);                
            end
    end

    switch ch
        case '?' 
           params.verbose=~params.verbose;
           if ~params.verbose
               disp('Verbose is OFF');
           else
               disp('b      browse images displaying the object bounds');
               disp('t      browse images displaying torsos of people, if available');
               disp('p      browse images displaying the poselet activations, if available');
               disp('q      browse grid of object bounds sorted by score');
               disp('?      suppress instructions');
               disp('ESC    exit browsing');
               disp(' ');
               if mode==1 || mode==2
                   disp('Current mode: Display object hypotheses in images');
                   disp('left/right arrow  move to the previous/next image (if more than one image)'); 
                   disp('g                 go to a specific image'); 
                   disp('up/down arrow     display more/fewer hypotheses, sorted by score'); 
                   disp('click on bounds   zooms on a hypothesis'); 
               elseif mode==3
                   disp('Current mode: Display poselet activations sorted by score. Shows foreground probability contours for the current poselet and examples of it');
                   disp('left/right arrow  move to the previous/next image (if more than one image)'); 
                   disp('g                 go to a specific image'); 
                   disp('up/down arrow     display the next/previous poselet activation'); 
                   disp('click on bounds   zooms on the bounding box');                    
               elseif mode==4
                   disp('Current mode: Display grid of detected objects sorted by score');
                   disp('left/right arrow  move to the previous/next batch of detected objects'); 
                   disp('g                 go to a specific object'); 
                   disp('click on patch    zooms on the patch'); 
               end
           end
        case 'b'
            mode=1;
        case 't'
            if isfield(params,'all_torso_hits')
                mode=2;
            else
               disp('Torsos not available'); 
            end
        case 'p'
            if isfield(params,'all_poselet_hits')
                mode=3;
            else
               disp('Poselet activations not available');                 
            end
        case 'q'
            mode=4;
        case 27
            close(figures(ishandle(figures)))
            return;
    end
end
end

% Zooms on a hypothesis. Displays a grid of all poselet activations within
% the cluster
function visualize_hypothesis(selected_hit, img, params)
global fig3_rect;
if params.verbose
   disp('Click on one or more poselet activations to see where they appear on the hypothesis'); 
   disp('When one activation is selected, Figure 3 displays training examples of the corresponding poselet'); 
   disp('ESC to go back'); 
end
bounds = selected_hit.bounds';
orig_bounds = selected_hit.bounds';
poselet_hits = params.all_poselet_hits.select(selected_hit.src_idx{1});

% The mask is twice as large at half resolution. Make it 1:1 with the patch
for i=1:length(params.masks)
    masks{i} = repmat(get_center_mask(params.masks{i}),[1 1 3]);
end

[srt,srtd]=sort(poselet_hits.score,'descend');
poselet_hits=poselet_hits.select(srtd);

PAD = mean(bounds(3:4))*[0.5 0.5];
bounds = bounds +[-PAD 2*PAD];
img = repmat(rgb2gray(img),[1 1 3]);
img = imcrop(img,bounds);
orig_bounds(1:2) = orig_bounds(1:2)-max(0,bounds(1:2));

tl = max(bounds(1:2),1);
poselet_hits.bounds(1:2,:) = poselet_hits.bounds(1:2,:) - repmat(tl',1,poselet_hits.size);

screen_bounds = [800 800];
scale = min(screen_bounds./[size(img,1) size(img,2)]);
warning off;
img = imresize(img,scale);
poselet_hits.bounds = poselet_hits.bounds * scale;
orig_bounds = orig_bounds*scale;

selected_poselets = false(poselet_hits.size,1);

thumbs = combine_patches_of_different_sizes(masks(poselet_hits.poselet_id));
orig_thumbs = thumbs;


while true
    % Display the image and outlines of selected poselets
    im_draw = img;


    best_hits = poselet_hits.select(selected_poselets);
    best_hits.bounds = round(best_hits.bounds);

    colors = jet(best_hits.size);
    for i=1:best_hits.size
        hm = masks{best_hits.poselet_id(i)};

        mask = imresize(hm,best_hits.bounds([4 3],i)');
        mask = imdilate(edge(double(mask(:,:,1)>0.5)),ones(3));

        img_mask = false(size(img,1),size(img,2));

        y_span=best_hits.bounds(2,i):(best_hits.bounds(2,i)+best_hits.bounds(4,i)-1);
        x_span=best_hits.bounds(1,i):(best_hits.bounds(1,i)+best_hits.bounds(3,i)-1);
        valid_y_span = y_span>0 & y_span<=size(im_draw,1);
        valid_x_span = x_span>0 & x_span<=size(im_draw,2);

        img_mask(y_span(valid_y_span), x_span(valid_x_span)) = mask(valid_y_span,valid_x_span);

        for j=1:3
            colored_mask(:,:,j) = cast(img_mask*colors(i,j)*255,'uint8');
        end
        im_draw(repmat(img_mask,[1 1 3])) = colored_mask(repmat(img_mask,[1 1 3]));
    end
    

    figure(11); clf;
    imshow(im_draw);
    title('Zoomed version of the selected hypothesis');
    hold on;
    rectangle('position',orig_bounds,'edgecolor','r','linewidth',3);
    for i=1:best_hits.size
        best_hits.select(i).draw_bounds(colors(i,:));
    end

    if isfield(params,'poselet_patches')
        if best_hits.size==1
            figure(3);
            if ~isempty(fig3_rect)
               set(3,'position',fig3_rect); 
            end
            montage(params.poselet_patches{best_hits.poselet_id},'Size',[1 3]);
            title('Examples of the selected poselet');
        elseif ishandle(3)
            fig3_rect=get(3,'position');
            close(3);
        end
    end
    
    % Display masks from each poselet in the cluster
    figure(10);

    [hhh,dims]=display_patches(thumbs,num2str(poselet_hits.score));
    title('Masks of the poselets that appear in the selected hypothesis');
    while 1
        [x,y,ch] = ginput(1);
        if isscalar(ch) break; end
    end

    if ch==27 % ESC
       windows = [10 11];
       close(windows(ishandle(windows)));
       figure(1);
       return;
    end

    y = floor(y/(size(thumbs,1)));
    x = floor(x/(size(thumbs,2)));
    idx = y*dims(1) + x+1;

    if idx>0 && idx<=poselet_hits.size
        selected_poselets(idx) = ~selected_poselets(idx);
        colors = jet(sum(selected_poselets));
    
        thumbs = orig_thumbs;
        selected_indices=find(selected_poselets);
        for i=1:length(selected_indices)
            for j=1:3
                thumbs(:,:,j,selected_indices(i)) = thumbs(:,:,j,selected_indices(i))*colors(i,j);
            end
        end
    end
end
end

function draw_poselet_hits(best_hits,params)
    global fig3_rect;
    cf=gcf;
    best_hits.select(1).draw_bounds('r',2,'-',[.7 .9 .7]);

    ch=get(gca,'Children');
    assert(isequal(get(ch(3),'Type'),'image'));
    img=get(ch(3),'CData');
    img=paste_mask(best_hits.select(1),params.masks,[1 0 0],img);
    set(ch(3),'CData',img); 

    if isfield(params,'poselet_patches')
        figure(3);
        if ~isempty(fig3_rect)
           set(3,'position',fig3_rect); 
        end
        montage(params.poselet_patches{best_hits.poselet_id(1)},'Size',[1 3]);
        figure(cf);
    end
end

% Takes an input image
function img=paste_mask(hit, masks, color, img)
    hit.bounds=round(hit.bounds);

    mask = imresize(get_center_mask(masks{hit.poselet_id}),hit.bounds([4 3])');
    mask = imdilate(edge(double(mask(:,:,1)>0.5)) | edge(double(mask(:,:,1)>0.7)) | edge(double(mask(:,:,1)>0.3)),ones(3));

    img_mask = false(size(img,1),size(img,2));

    y_span=hit.bounds(2):(hit.bounds(2)+hit.bounds(4)-1);
    x_span=hit.bounds(1):(hit.bounds(1)+hit.bounds(3)-1);
    valid_y_span = y_span>0 & y_span<=size(img,1);
    valid_x_span = x_span>0 & x_span<=size(img,2);

    img_mask(y_span(valid_y_span), x_span(valid_x_span)) = mask(valid_y_span,valid_x_span);
    for j=1:3
        colored_mask(:,:,j) = cast(img_mask*color(j)*255,'uint8');
    end
    if size(img,3)==1
       img=repmat(img,[1 1 3]); 
    end
    img=max(img,colored_mask);
end

function mask=get_center_mask(mask)
    [H W] = size(mask);
    mask=imresize(mask(round(H/4):round(H*3/4),round(W/4):round(W*3/4)),2);
end
