package KTEditor;

import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import java.util.Enumeration;

/**
 * This action subclass toggles the style attribute which signifies a 
 * particular tag (or effect) across the current selection.  If 
 * all characters in the selection already have the attribute 
 * for this tag set, then this removes the attribute from the selection.
 * We also remove the attributes corresponding to any conflicting tag 
 * types.  Note: if the selection is empty these actions all apply to the 
 * "input attributes" (which control text about to be entered).  
 */
public class ToggleTagAttribute extends StyledEditorKit.StyledTextAction {
    /** A descriptor for the tag (or effect) that we will be toggling. */
    protected SegmentDescriptor tag = null;
    /** Get the descriptor for the tag (or effect) that we will be toggling. */
    public SegmentDescriptor getTag() {return tag;}
    /** Set the descriptor for the tag (or effect) that we will be toggling. */
    public void setTag(SegmentDescriptor tg) {tag = tg;}
    
    /** Construct an action for toggling attributes marking text with the given tag or effect. */
    public ToggleTagAttribute(SegmentDescriptor tg) {
          super(tg.getName()); 
          putValue(Action.NAME, tg.getName()); // already done in super??
          setTag(tg);
    }
    
    /**
     * Remove all segment attributes that conflict with the given tag.  If the he attribute set has 
     * been modified, true is returned.  
     */
    public boolean removeConflictingAttrs(MutableAttributeSet fromAttrs, SegmentDescriptor newAttr)
    { 
        boolean result = false;
        
        // walk down the attributes currently in the set looking for conflics
        for (Enumeration en = fromAttrs.getAttributeNames(); en.hasMoreElements(); ) 
        {
            // is it one of ours (a segment descriptor for its attribute name)
            // (but don't remove the new attribute)
            Object nm = en.nextElement();
            if (nm instanceof SegmentDescriptor && nm != newAttr) 
            {
                // test for a conflict
                SegmentDescriptor otherAttr = (SegmentDescriptor)nm;
                if (otherAttr.conflictsWith(newAttr))
                {
                    // we have a conflict, remove the conflicting attribute
                    fromAttrs.removeAttribute(otherAttr);
                    result = true;
                }
            }
        }
        return result;
    }
    
    /**
     * Actaully carry out the action.  If all of the current selection is not already
     * marked with our attribute, then we mark it.  If all parts of it are already
     * marked then we unmark it.  Any attributes which conflict with our attribute
     * are removed.  In the case of the selection being empty, these actions apply
     * instead to the "input attributes" (i.e., the attributes that will be applied
     * to text typed at the insertion point).
     */
    public void actionPerformed(ActionEvent actEvt) {
        // Pull the editor, document, and editorkit and ensure they exist
        JEditorPane edPane = getEditor(actEvt);
        if (edPane == null) return;
        StyledEditorKit edKit = getStyledEditorKit(edPane);
        if (edKit == null) return;
        DefaultStyledDocument doc = (DefaultStyledDocument)edPane.getDocument();
        if (doc == null) return;
           
        // walk down the selection and see if it is all already set for this attribute
        int selStart = edPane.getSelectionStart();
        int selEnd = edPane.getSelectionEnd();
        boolean allSet; // is this attribute set on all elements of selection already?      
        // if the selection is empty use the EditorKit to look at attributes at Caret
        if (selStart == selEnd) {
            // pull out the attributes for new text
            MutableAttributeSet caretAttrs = edKit.getInputAttributes();
            
            // get the old value (if any)
            Object oldAttrVal = caretAttrs.getAttribute(getTag());
            allSet = (oldAttrVal != null && (oldAttrVal instanceof SegmentDescriptor));
            
            // remove any conflicting attributes 
            // note: this mutates the underlying inputAttributes which we got 
            //       a reference two via getInputAttributes()
            removeConflictingAttrs(caretAttrs, getTag());
        }
        else 
        {
            // selection is not empty so use the document to look at each element in range
            // walk down the selection and see if this attribute is set
            boolean someSet = false;
            allSet = true;  
            for (int pos = selStart; pos < selEnd; ) {
                // pull out the element and look at its attributes
                Element elm = doc.getCharacterElement(pos);
                SimpleAttributeSet elmAttrs = new SimpleAttributeSet(elm.getAttributes());
                Object attrVal = elmAttrs.getAttribute(getTag());
                
                // is our attribute set?
                if (attrVal != null && attrVal.equals(getTag())) 
                {
                    /* so far they are all set */
                    someSet = true;
                }
                else 
                {
                    /* one is not set */
                    allSet = false;
                }
                
                // remove any conflicting attributes and put the result back if changed
                if (removeConflictingAttrs(elmAttrs, getTag()))
                {
                    // clip the element to the selection 
                    int spanStart = Math.max(elm.getStartOffset(), selStart);
                    int spanEnd = Math.min(elm.getEndOffset(), selEnd);
                
                    // set the attributes within that span
                    doc.setCharacterAttributes(spanStart, spanEnd-spanStart, elmAttrs, true);                  
                }

                // move past the chunk 
                pos = elm.getEndOffset();
            }
            
            // at least one had to have been set to count as all
            allSet = allSet && someSet;
        }
 
        
        // if all elements have the attribute already set then we are actually 
        // going to be unsetting, we do that by using the Boolean.FALSE object as 
        // the value
        Object newAttrVal;
        if (!allSet) newAttrVal = getTag(); else newAttrVal = Boolean.FALSE;

        // create an attribute set with that value in it
        SimpleAttributeSet newSet = new SimpleAttributeSet();
        newSet.addAttribute(getTag(), newAttrVal);

        // and put it back replacing only that attribute
        setCharacterAttributes(edPane, newSet, false);
    }  
    
    /** 
     * Produce a debugging dump showing the attributes associated with each chunk of text in the 
     * document associated with the given editor which falls between the given two locations.
     */
    public void dumpDocAttrs(JEditorPane edPane, int startPos, int endPos)
    {
        System.out.println("Document["+startPos+","+endPos+"]");
        if (edPane == null) return;
        StyledEditorKit edKit = getStyledEditorKit(edPane);
        if (edKit == null) return;
        DefaultStyledDocument doc = (DefaultStyledDocument)edPane.getDocument();
        if (doc == null) return;
        
        Element elm;
        for (int pos = startPos; pos<endPos; pos=elm.getEndOffset())
        {
            elm = doc.getCharacterElement(pos);
            System.out.print("(" + pos +","+Math.min(endPos, elm.getEndOffset())+"):");
            System.out.println(new SimpleAttributeSet(elm.getAttributes()));
        }
        System.out.println("End Document["+startPos+","+endPos+"]");
    }
}   
   
