package KTEditor;

import java.awt.*;
import javax.swing.text.*;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * Highlighter subclass that does normal highlights, plus draws a set of
 * highlights corresponding to the tags indicated within attributes on the
 * text itself (without maintaining explicit Highlight objects for these.
 */
public class TagHighlighter extends DefaultHighlighter {
    
    /** Default constructor */
    public TagHighlighter() {super();}
    
    /**
     * Test whether the given element start position is the start of a segment denoted
     * by the given attribute.
     */
    public boolean positionIsStartOfTag(
    StyledDocument inDoc,
    Element thisElm,
    Object thisAttr,
    int pos) {
        // if position is at start of document it must be start of tag
        if (pos <= 0) return true;
        
        // extract the element before us
        Element prevElm = inDoc.getCharacterElement(pos - 1);
        
        // if its the same element we can't be the start
        if (prevElm == thisElm) return false;
        
        // pull the attributes for the previous element
        AttributeSet prevAttrs = prevElm.getAttributes();
        
        // if this tag is not listed as part of those attibutes we are the start
        Object prevV = prevAttrs.getAttribute(thisAttr);
        return ((prevV == null) || !(prevV instanceof SegmentDescriptor));
    }
    
    /**
     * Test whether the given element end position is the end of a segment denoted
     * by the given attribute.
     */
    public boolean positionIsEndOfTag(
    StyledDocument inDoc,
    Element thisElm,
    Object thisAttr,
    int pos) {
        // if position is at end of document it must be end of tag
        if (pos >= inDoc.getLength()) return true;
        
        // extract the element after us
        Element nextElm = inDoc.getCharacterElement(pos + 1);
        
        // if its the same element we can't be the end
        if (nextElm == thisElm) return false;
        
        // pull the attributes for the next element
        AttributeSet nextAttrs = nextElm.getAttributes();
        
        // if this tag is not listed as part of those attibutes we are the end
        Object nextV = nextAttrs.getAttribute(thisAttr);
        return ((nextV == null) || !(nextV instanceof SegmentDescriptor));
    }
    
    /**
     * Method for drawing highlights for one segment of text.  This is
     * overridden to both draw the normal highlights and any tag highlights
     * implied by the tag sttributs in the text.
     */
    public void paintLayeredHighlights(Graphics g, int p0, int p1, Shape viewBounds,
        JTextComponent editor, View view) 
    {
        // Note: we rely on some unstated (but apparently true) facts about the
        // ways this gets called.  In particular, we assume that this is
        // only called for chunks of text which have a uniform set of attributes
        // set, and which do not span lines.  The second assertion must be true
        // for the superclass code to function properly, its not clear that
        // first assertion is as safe.
        
        // have the superclass draw its highlights
        super.paintLayeredHighlights(g,p0,p1,viewBounds,editor,view);
        
        // Now draw highlights for any tag we have covering this segment of text
        
        // get the attribute set for this element
        StyledDocument ourDoc = (StyledDocument)editor.getDocument();
        Element elm = ourDoc.getCharacterElement(p0);
        AttributeSet attrs = elm.getAttributes();
        
        // Note: here we have assumed that the attribues at the start position are
        // sufficient due to uniformity across the segment
        
        // Extract the font this segment is drawn in and determine the basesline offset
        Font thisFont = StyleContext.getDefaultStyleContext().getFont(attrs); 
        FontMetrics thisFontMetrics = g.getFontMetrics(thisFont);
        int baselineOff = thisFontMetrics.getAscent();
        
        // get the rectangle covering this chunk of text
        Rectangle drawRect = null;
        try {
            drawRect = view.modelToView(
                p0,Position.Bias.Forward,
                p1,Position.Bias.Backward, viewBounds).getBounds();
        } catch (BadLocationException ex) {
            // shouldn't happen, but... don't draw anything if we were out of bounds
            return;
        }
        
        // walk across the set of attributes we have in this segment
        // picking out the ones that are ours
        for (Enumeration en = attrs.getAttributeNames(); en.hasMoreElements(); ) 
        {
            // is it one of ours (a segment descriptor for its attribute name)
            Object nm = en.nextElement();
            if (nm instanceof SegmentDescriptor) 
            {
                // we are only interested if it has a SegmentDescriptor value
                Object v = attrs.getAttribute(nm);
                if (v != null && v instanceof SegmentDescriptor) {
                    // set flag for whether we are at ends
                    int endsCode = 0;
                    if (positionIsStartOfTag(ourDoc,elm,nm,p0)) {
                        endsCode |= TagArtist.LEFT_END;
                    }
                    if (positionIsEndOfTag(ourDoc,elm,nm,p1)) {
                        endsCode |= TagArtist.RIGHT_END;
                    }
                    
                    // ask the segment to draw the background highlight for this chunk
                    ((SegmentDescriptor)v).drawHighlightRect(
                        g,drawRect,baselineOff,editor,p0,p1,view,viewBounds,endsCode);
                    
                    // if this was the start, draw start text over that
                    // xx bug this gets clipped at end of segment even though tag may extend
                    //    into another segment, need to search forward for the end of
                    //    segement or end of line here and adjust the drawRect accordingly
                    if ((endsCode & TagArtist.LEFT_END)!= 0) 
                    {
                        ((SegmentDescriptor)v).drawStartText(
                          g,((SegmentDescriptor)v).getName(),drawRect,baselineOff);
                    }
                } // end if (v != ...
            } // end if (nm ...
        } // for
    } // end method
    
    
}
