package KTEditor;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/** more later */
public class SimpleHSlider extends JComponent {
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* management of change listeners */  
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    protected ChangeEvent cachedEvent = null;
    
    /** Add a ChangeListener to our listener set */
    public void addChangeListener(ChangeListener listener) 
    {
        listenerList.add(ChangeListener.class, listener);
    }


    /** Remove a ChangeListener from our set */
    public void removeChangeListener(ChangeListener listener) 
    {
        listenerList.remove(ChangeListener.class, listener);
    }

    /** Return an array with all change listerners currently registered */
    public ChangeListener[] getChangeListeners() 
    {
        return (ChangeListener[])listenerList.getListeners(ChangeListener.class);
    }

    /**
     * Note that a change in one of our values has occured by sending change
     * events to all our listeners.
     */
    protected void fireChangeEvents() 
    {
        // pull the listeners and walk down the list backwards
        Object[] listeners = listenerList.getListenerList();
        for (int l = listeners.length - 2; l >= 0; l -= 2) 
        {
            // is this one of ours
            if (listeners[l] == ChangeListener.class) 
            {
                // make sure our cached event is set up
                if (cachedEvent == null) cachedEvent = new ChangeEvent(this);
                
                // send it to the listener
                ((ChangeListener)listeners[l+1]).stateChanged(cachedEvent);
            }
        }
    }  
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* instance variables */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Value of high end of slider range. */
    protected long maxValue = 1000;
    
    /** Get the value of high end of slider range. */
    public long getMaxValue() {return maxValue;}
    
    /** Set the value of high end of slider range. */
    public void setMaxValue(long val) 
    {
        if (maxValue != val)
        {
            maxValue = val;
            fireChangeEvents();
            repaint();
        }
    }
    
    /** Value of low end of slider range. */
    protected long minValue = 1000;
    
    /** Get the value of low end of slider range. */
    public long getMinValue() {return minValue;}
    
    /** Set the value of low end of slider range. */
    public void setMinValue(long val) 
    {
        if (minValue != val)
        {
            minValue = val;
            fireChangeEvents();
            repaint();
        }
    }
    
    /** Current value of the slider */
    protected long value = 0;
    
    /** Get the current value of the slider */
    public long getValue() {return value;}
    
    /**
     * Set the current value of the slider.  Values outside the
     * range of the slider will be clipped to put them in range.
     */
    public void setValue(long val) {
        /* set the value if its changing */
        if (value != val) {
            /* set the value and damage our image */
            value = val;
            fireChangeEvents();
            repaint();
        }
    }
    
    /**
     * Convenience version of setValue that takes an int 
     */
    public void setValue(int val) {setValue((long)val);}
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* Appearance related */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Width of the thumb */
    protected int thumbW = 5;
    
    /** Height of the thumb */
    protected int thumbH = 15;
    
    /** Height of the track */
    protected int trackH = 6;
    
    /** Offset from the top of the thumb to the track */
    protected int trackOffset = (thumbH - trackH) / 2; // centered
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Redraw the slider in its current configuration */
    public void paint(Graphics g) {
        paintTrack(g);
        paintThumb(g);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    protected void drawShadowRect(Graphics g, int x, int y, int w, int h, boolean inset)
    {
        // do top left
        if (inset)
          g.setColor(Color.black);
        else
          g.setColor(Color.white);
        g.fillRect(x,y,w-1,h-1);
        
        // now bottom right
        if (inset)
            g.setColor(Color.white);
        else
            g.setColor(Color.black);
        g.fillRect(x+1,y+1,w-1,h-1);
        
        // Now the body 
        if (inset)
            g.setColor(Color.darkGray);
        else
            g.setColor(Color.lightGray);
        g.fillRect(x+1,y+1,w-2,h-2);
    }
    
    /** Redraw the track of the slider */
    public void paintTrack(Graphics g) {
        Rectangle bounds = getBounds();
        drawShadowRect(g, 0, trackOffset, (int)bounds.getWidth(), trackH, true);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Compute the screen position of the thumb for the given slider value */
    protected int thumbPosFromValue(long val)
    {
        int ourW = getWidth();
        double range = (double)(getMaxValue()-getMinValue());
        if (range == 0.0) range = Double.MIN_VALUE;
        double posFraction = ((double)(getValue()-getMinValue())) / range;
        return (int)((ourW - thumbW)*posFraction);
    }
   
    /** Compute the value corresponding to the left edge of the thumb being in the given position */
    protected long valueFromThumbPos(int pos)
    {
        int ourW = getWidth();
        int posRange = ourW - thumbW;
        if (pos > posRange) pos = posRange;
        if (pos < 0) pos = 0;
        if (posRange <= 0) posRange = 1;
        double valFraction = ((double)pos)/((double)posRange);
        return getMinValue() + (long)((getMaxValue()-getMinValue())*valFraction);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Redraw the thumb of the slider */
    public void paintThumb(Graphics g) {
        drawShadowRect(g, thumbPosFromValue(getValue()), 0, (int)thumbW, (int)thumbH,false);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */  
    /* Input handling */ 
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
   
    /** determine if the given point is inside the current thumb position */
    protected boolean picksThumb(int x, int y)
    {
        if (y < 0 || y > getHeight()) return false;
        int pos = thumbPosFromValue(getValue());
        if (x < pos || x > pos + thumbW) return false;
        return true;
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** true if we are currently dragging the thumb */
    protected boolean draggingThumb = false;
    
    /** x offset from left edge of the thumb where it was last grabbed */
    protected int grabOffset = 0;
    
    /**
     * Handle mouse press, release, etc. events to implment dragging the
     * thumb.
     */
    protected void processMouseEvent(MouseEvent evt)
    {
        switch(evt.getID())
        {
            case MouseEvent.MOUSE_PRESSED:
                if (!draggingThumb && picksThumb(evt.getX(), evt.getY()))
                {
                    // change to dragging state and remember where inside the thumb we hit
                    draggingThumb = true;
                    grabOffset = evt.getX() - thumbPosFromValue(getValue());
                }
            break;
            case MouseEvent.MOUSE_RELEASED:
                // update value one last time
                setValue(valueFromThumbPos(evt.getX()-grabOffset));
                
                // drop back to non-dragging state
                draggingThumb = false;
                grabOffset = 0;
            break;
            default:
                // ignore everything else
            break;
        }
        // have super class dispatch to any listeners
        super.processMouseEvent(evt);
    }
      
    /** Handle mouse movement events to implement dragging the thumb. */
    protected void processMouseMotionEvent(MouseEvent evt)
    {
        // reposition the thumb if we are dragging 
        if (draggingThumb && evt.getID() == MouseEvent.MOUSE_DRAGGED)
        {
            setValue(valueFromThumbPos(evt.getX()-grabOffset));
        }
        
        // the the superclass have it to handle other listeners
        super.processMouseMotionEvent(evt);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* Constructor */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Construct with full values */
    public SimpleHSlider(long minV, long maxV, long initV, int w) {
        // init instance vars
        setMinValue(minV); setMaxValue(maxV); setValue(initV);
        
        // set a mostly forced size
        Dimension sz = new Dimension(w, thumbH);
        setSize(sz);
        setPreferredSize(sz);
        setMinimumSize(sz);
        setMaximumSize(sz);
        
        // we don't draw everything
        setOpaque(false);
        
        // make sure that we get mouse and mouse motion events
        enableEvents(AWTEvent.MOUSE_EVENT_MASK | AWTEvent.MOUSE_MOTION_EVENT_MASK);
    }
    
    /** Convenience full constructor that takes ints instead of longs */
    public SimpleHSlider(int minV, int maxV, int initV, int w)
    {
        this((long)minV,(long)maxV,(long)initV,w);
    }
    
    /** Construct with default min & max */
    public SimpleHSlider(int w) {
        this(0,1000,0, w);
    }
    
    /** Default width that we make the slider if not told otherwise */
    public static final int DEFAULT_W = 100;
    
    /** Default constructor */
    public SimpleHSlider() {
        this(DEFAULT_W);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
}

