package KTEditor;

import kinetic.Sequence;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import java.text.DecimalFormat;

/**
 *
 * xx more later
 * <p>
 *
 * @author  Scott Hudson
 */
public class PreviewManager {
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* UI instance variables */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** The frame that holds the preview */
    protected JFrame previewFrame = null;
    
    /** The panel that hosts the preview interface */
    protected JComponent  previewTop = null;
    
    /** The canvas that we actually draw previews onto */
    protected JPanel previewCanvas = null;
    
    /** Get the canvas that we actually draw previews onto */
    public JPanel getPreviewCanvas() {return previewCanvas;}
    
    public Dimension getCanvasDimension(){
        Rectangle r = getPreviewCanvas().getBounds();
        return new Dimension(r.width,r.height);
    }
    
    /** A label that displays current frames per second in the preview */
    protected JLabel timeLabel = null;
    
    /** Button for resetting the preview */
    protected JButton resetPreviewButton = null;
    
    /** Button for running/pausing off the preview */
    protected JButton runPreviewButton = null;
    
    /** Preview button text for when preview is not running */
    protected String runLabelWhenPaused = "Run Preview";
    
    /** Preview button text for when preview is running */
    protected String runLabelWhenRunning = "Pause Preview";
    
    /** Checkbox controlling looping status */
    JCheckBox loopCheck = null;
    
    /** Checkbox controlling auto preview status */
    JCheckBox autoCheck = null;
    
    /** Are we currently looping (or in "one-shot" mode) */
    protected boolean looping = true;
    
    /** Are we currently looping the preview animation */
    public boolean isLooping() {return looping;}
    
    /** Set whether we should loop the preview animation */
    public void setLooping(boolean val) {
        looping = val;
        if ((loopCheck != null) && (loopCheck.isSelected() != val)) {
            loopCheck.setSelected(val);
        }
    }
    
    /** Are we allowing automatic restart after edits */
    protected boolean auto = true;
    
    /** Are we allowing automatic restart after edits */
    public boolean isAuto() {return auto;}
    
    /** Set whether we allow automatic restart after edits */
    public void setAuto(boolean val) {
        auto = val;
        if ((autoCheck != null) && (autoCheck.isSelected() != val)) {
            autoCheck.setSelected(val);
        }
    }
    
    /** Slider that tracks and manipulates time */
    protected SimpleHSlider timeSlider = null;
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* UI setup methods */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Build (but do not show) the window containing the preview and
     * related controls.
     */
    public void buildPreviewUI() {
        // create the top level frame for the preview
        previewFrame = new JFrame("Kinetic Typography Preview");
        previewFrame.setLocation(new Point(360,0));
        
        // arrange to shut down the animation when we close
        previewFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        previewFrame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                // bring down the preview window
                hidePreviewUI();
                
                // don't let the autofire bring it back up
                setAuto(false);
            }
        });
        
        // put in the top level pane
        //previewTop = new JPanel(new GridLayout(0,1));
        previewTop = Box.createVerticalBox();
        previewFrame.setContentPane(previewTop);
        
        // put a box with controls at the top of that
        previewTop.add(buildPreviewControlPane());
        previewTop.add(new JSeparator());
        
        // then a blank canvas
        previewCanvas = new JPanel();
        previewCanvas.setBackground(Color.white);
        previewCanvas.setMinimumSize(new Dimension(200,100));
        previewCanvas.setPreferredSize(new Dimension(640,480));
        previewTop.add(previewCanvas);
        
        // finish the frame
        previewFrame.pack();
        previewFrame.setVisible(false);
        
        // get the preview timer set up
        setupPreviewTimer();
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    protected JComponent buildPreviewControlPane() {
        // build the panel to hold the result
        JPanel result = new JPanel(new FlowLayout());
        
        // reset button with handler
        resetPreviewButton = new JButton("Reset Preview");
        resetPreviewButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                resetAnimationTime(0);
            }
        });
        result.add(resetPreviewButton);
        
        // run/pause button with handler
        runPreviewButton = new JButton(runLabelWhenPaused);
        runPreviewButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (runPreviewButton.getText().equals(runLabelWhenRunning))
                    pausePreview();
                else
                    runPreview();
            }
        });
        result.add(runPreviewButton);
        
        // use a little column to stack the check boxes
        JPanel checkBoxes = new JPanel(new GridLayout(0,1));
        
        // looping checkbox with handler
        loopCheck = new JCheckBox("Loop");
        loopCheck.setSelected(isLooping());
        loopCheck.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                JCheckBox onOff = (JCheckBox)e.getSource();
                setLooping(onOff.isSelected());
            }
        });
        checkBoxes.add(loopCheck);
        
        // autofire checkbox with handler
        autoCheck = new JCheckBox("Auto Preview");
        autoCheck.setSelected(isAuto());
        autoCheck.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                JCheckBox onOff = (JCheckBox)e.getSource();
                setAuto(onOff.isSelected());
            }
        });
        checkBoxes.add(autoCheck);
        result.add(checkBoxes);
        
        // a column to stack the time feedback in
        JPanel timeFeedback = new JPanel(new GridLayout(0,1));
        
        // label with feedback about time and FPS
        timeLabel = new JLabel(makeTimeLabel());
        timeFeedback.add(timeLabel);
        
        // slider for time control and feedback
        timeSlider = new SimpleHSlider(0L,Math.max(1000L, getSequenceDuration()), 0L, 150);
        // if the slider is moved, reset the animation time
        timeSlider.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                long sliderTime = ((SimpleHSlider)e.getSource()).getValue();
                if (sliderTime != currentLocalTime)
                    resetAnimationTime(sliderTime);
            }
        });
        timeFeedback.add(timeSlider);
        result.add(timeFeedback);
        
        return result;
    }
        
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    public void showPreviewUI() {
        // if we have a frame and its not already visible, make it so
        if (previewFrame != null && !previewFrame.isVisible())
            previewFrame.setVisible(true);
    }
    
    public void hidePreviewUI() {
        if (previewFrame != null) {
            pausePreview();
            previewFrame.setVisible(false);
        }
    }
    
    public void toFront() {
        if (previewFrame != null) previewFrame.toFront();
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* structures representing and managing the KT engine sequence object for the presentation */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** The Sequence object representing the current preview sequence */
    protected Sequence currentPreviewSequence = null;
    
    /** Get the Sequence object representing the current preview sequence */
    public Sequence getCurrentPreviewSequence() {return currentPreviewSequence;}
    
    /**
     * Set the Sequence object representing the current preview sequence.  The boolean parameter
     * indicates whether the local time is reset (the preview is restarted) or left where it was
     * (which may produce odd results, but might be ok for small edits).
     */
    public void setCurrentPreviewSequence(Sequence seq, boolean resetTime) {
        currentPreviewSequence = seq;
        if (resetTime) restartPreview();
    }
    
    /**
     * Set the Sequence object representing the current preview sequence and restart
     * the presentation for local time 0.
     */
    public void setCurrentPreviewSequence(Sequence seq) {
        setCurrentPreviewSequence(seq,true);
    }
       
    /** Build a new sequence object from the given document */
    public Sequence buildPreviewSequence(StyledDocument fromDoc) {
        EffectSegment curSeg = null;
        
        // start a new presentation
        KTEngineInterface.beginPresentation(previewCanvas);
        
        // walk across document positions and extract each effect segment
        for (int pos = 0; pos < fromDoc.getLength(); pos = curSeg.getEndOff()) {
            // get the segment for the next effect 
            curSeg =  EffectSegment.buildNextSegment(fromDoc, pos);
            
            // if there is no effect apply the default 
            if (curSeg.getEffectInst() == null) 
                curSeg.setEffectInst(new EffectInstanceDescriptor(KTEngineInterface.getDefaultEffect()));
            
            // put it in the presentation
            KTEngineInterface.addToPresentation(curSeg,previewCanvas);
        }
        
        // finish and return the sequence
        return KTEngineInterface.endPresentation(previewCanvas);
    }
    
    /**
     * Build and install a new preview from the given document.  If the boolean argument is
     * true we reset local time to 0, if not we leave local time whereever it was.
     */
    public void installNewPreview(StyledDocument fromDoc, boolean resetTime) {
        setCurrentPreviewSequence(buildPreviewSequence(fromDoc), resetTime);
        setSequenceDuration((long)currentPreviewSequence.getDuration());
    }
    
    /** Build and install a new preview from the given document and reset local time to 0 */
    public void installNewPreview(StyledDocument fromDoc) {
        installNewPreview(fromDoc,true);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* timing and amination driver related stuff */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Timer that schedules rendering of frames for the preview display */
    protected Timer previewTimer = null;
    
    /** Requested frame delay time in ms */
    protected final int frameDelay = 1000/60;
    
    protected void setupPreviewTimer() {
        previewTimer = new Timer(frameDelay, new ActionListener() {
            public void actionPerformed(ActionEvent e) { doFrame(); }
        });
        previewTimer.setInitialDelay(10);
        previewTimer.setCoalesce(true);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Offset from global (wall clock) time in ms to local time.  This corresponds
     * to the wall clock time which counts as 0 in local time.
     */
    protected long localTimeOffset = System.currentTimeMillis();
    
    /**
     * Reset local time offset such that the current time corresponds to the given value
     */
    public void resetAnimationTime(long localTimeValueInMs) {
        // reset the offset as well as stored times
        localTimeOffset = System.currentTimeMillis() - localTimeValueInMs;
        pauseTime = currentLocalTime = localTimeValueInMs;
        
        // set the time label
        timeLabel.setText(makeTimeLabel());
        
        // make sure we have a preview, then render a frame at that time so we are synced
        if (currentPreviewSequence == null)
            KTEdit.getInstance().installNewPreview(false);
        KTEngineInterface.renderFrame(currentPreviewSequence, previewCanvas, currentLocalTime);
    }
    
    /**
     * The local time at which the current (or most recent) frame is being rendered at
     */
    long currentLocalTime = 0;
    
    /**
     * Global time when we started rendering the most recent frame
     */
    protected long lastFrameTime = System.currentTimeMillis();
    
    /**
     * Most recent time from starting rendering a frame until starting a new one
     */
    protected long frameDuration = Long.MAX_VALUE;
    
    /**
     * Duration of the full animation sequence.  If local time goes past this, we
     * either stop time or reset it to 0 depending on whether we are looping.
     */
    protected long sequenceDuration = Long.MAX_VALUE;
    
    /**
     * Get the duration of the full animation sequence (in ms).  If local time goes past this, we
     * either stop time or reset it to 0 depending on whether we are looping.
     */
    public long getSequenceDuration() {return sequenceDuration;}
    
    /**
     * Set the duration of the full animation sequence (in ms).  If local time goes past this, we
     * either stop time or reset it to 0 depending on whether we are looping.  Setting the
     * duration <= 0 makes the duration infinite.
     */
    public void setSequenceDuration(long dur) {
        if (dur <= 0) dur = Long.MAX_VALUE;
        sequenceDuration = dur;
        
        // udpate the time slider with its new end point
        if (timeSlider != null) timeSlider.setMaxValue(dur);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Do the time bookkeeping to start a new frame.  This sets frameDuration to correspond
     * to the time since the last frame, lastFrameTime to the current global time, and
     * currentLocalTime to the current local time.  We also check if we are past the
     * sequence duration here, and will invoke pastSequenceDuration() if that happens.
     */
    protected void newFrameTime() {
        // update frameDuration and currentLocalTime
        long timeNow = System.currentTimeMillis();
        frameDuration = timeNow - lastFrameTime;
        lastFrameTime = timeNow;
        currentLocalTime = timeNow - localTimeOffset;
        
        // if we are past the end of the sequence, handle that
        if (currentLocalTime > sequenceDuration)
            pastSequenceDuration(timeNow);
        
        // update the time label and slider
        timeLabel.setText(makeTimeLabel());
        timeSlider.setValue(currentLocalTime);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Handle time bookkeeping and other actions when we have exceeded the duration of the
     * sequence.  If looping is turned on, this will reset local time to 0.  Otherwise it
     * will stop the animation (and hence stop time).
     */
    protected void pastSequenceDuration(long globalTimeNow) {
        resetAnimationTime(0);
        if (isLooping()) {
            currentLocalTime = globalTimeNow - localTimeOffset;
        }
        else {
            pausePreview();
        }
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** are we currently paused */
    protected boolean paused = true;
    
    /** time at which we resume from a pause (in local time) */
    protected long pauseTime = 0;
    
    /** release the preview animation to run */
    public void runPreview() {
        if (currentPreviewSequence == null)
            KTEdit.getInstance().installNewPreview(false);
        
        // if we are paused start where we left off
        if (paused) resetAnimationTime(pauseTime);
        
        // set state for running and start the frame timer
        paused = false;
        previewTimer.start();
        runPreviewButton.setText(runLabelWhenRunning);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Pause the preview animation */
    public void pausePreview() {
        previewTimer.stop();
        runPreviewButton.setText(runLabelWhenPaused);
        pauseTime = currentLocalTime;
        paused = true;
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Start the preview animation over from the beginning */
    public void restartPreview() {
        resetAnimationTime(0);
        runPreview();
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Formatter for a double suitable for showing frames per second */
    protected final DecimalFormat FPSDigits = new DecimalFormat("#00.0");
    
    /** Formatter for a double suitable for showing current local time */
    protected final DecimalFormat timeDigits = new DecimalFormat("#00.0");
    
    protected String makeTimeLabel() {
        double fps = 1000.0/(double)frameDuration;
        
        return "Time: " +
        timeDigits.format( currentLocalTime/1000.0) + " sec" +
        " (" + FPSDigits.format(fps) + "fps)";
    }
    
    /** Do the work to render a frame */
    public void doFrame() {
        // update frame timing stats and get the current local time
        newFrameTime();
        
        // have the engine render the frame into the preview canvas
        KTEngineInterface.renderFrame(currentPreviewSequence, previewCanvas, currentLocalTime);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
}
