package KTEditor;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.text.DecimalFormat;

/** more later */
public class ParameterSlider extends SimpleHSlider  {
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* appearance related */  
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Font we draw various number displays in */
    protected Font numberFont = new Font("Monospace", Font.PLAIN, 9);
    /** Metrics for the number font */
    protected FontMetrics numberFontMetrics = Toolkit.getDefaultToolkit().getFontMetrics(numberFont);
    /** Height of a number */
    protected int numberH = numberFontMetrics.getHeight();
    /** Ascent of a number */
    protected int numberAscent = numberFontMetrics.getAscent();
    /** Vertical spacing around the number displays */
    protected int numberVPad = 2;
    /** Horizontal spacing around the number displays */
    protected int numberHPad = 2;
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Are we displaying in integer or floating point */
    protected boolean showInt = true;
    
    /** Is this slider currently handling a double value instead of a long */
    public boolean isDoubleValue() {return !showInt;}
    
    /** Formatter to use for small (range <= 2.0) floating point display */
    protected DecimalFormat smallFloatFormat = new DecimalFormat("#0.00");
    
    /** Formatter to use for larger (range > 2.0) floating point display */
    protected DecimalFormat largerFloatFormat = new DecimalFormat("#0.0");
    
    /** Formatter to use for integer display */
    protected DecimalFormat intFormat = new DecimalFormat("#0");
    
    /** Current formatter for number label text */
    protected DecimalFormat numberLabelFormat = intFormat;
    
    /** Scale factor that we use to convert from floating values to the longs we strore */
    protected long doubleScaleFactor = 10000;  // give us 4 decimal places
    
    /** Set the number label format based on showInt setting and dynamic range */
    protected void setupLabelFormat()
    {
        // if we have only seen integers, use interger forma
        if (showInt) 
            numberLabelFormat = intFormat;
        else 
        {
            // pick number of decimals based on dynamic range we are over
            double range = (getMaxValue()-getMinValue()) / (double)doubleScaleFactor;
            if (range <= 2.0)
                numberLabelFormat = smallFloatFormat;
            else
                numberLabelFormat = largerFloatFormat;
        }
    }

    /** Format text for a number label */
    protected String labelStr(long num)
    {
        setupLabelFormat();
        if (showInt)
            return numberLabelFormat.format(num);
        else
            return numberLabelFormat.format(num/(double)doubleScaleFactor);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Redraw the track of the slider */
    public void paintTrack(Graphics g) {
        // track goes along the top of the bound
        Rectangle bounds = getBounds();
        drawShadowRect(g, 0, 0, (int)bounds.getWidth(), trackH, true);
        
        // min value label goes under drawn flush left
        g.setColor(Color.black);
        g.setFont(numberFont);
        g.drawString(labelStr(getMinValue()), 0, trackH+numberHPad+numberAscent);
        
        // max value label goes under drawn flush right
        int maxMarkW = numberFontMetrics.stringWidth(labelStr(getMaxValue()));
        g.drawString(labelStr(getMaxValue()),
            (int)bounds.getWidth()-maxMarkW, trackH+numberHPad+numberAscent);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Rectangle object reused by all calls to thumbRect() */
    private static Rectangle tRectResult = new Rectangle();
    
    /** 
     * Compute the rectangle the thumb should be drawn in for the current value.
     * The size of the thumb is determined by the number label presenting the value.
     * The position of the thumb slides left and right with respect to where the 
     * nominal fixed thumb would have been, such that at the minimum value its 
     * flush with the left edge and at the max its flush with the right.  This keeps
     * the thumb inside the track in all cases.
     *  
     * Note the rectangle returned here is reused on every call, so copy the values 
     * out if they are needed outside a single routine's calculation.
     */
    protected Rectangle thumbRect()
    {
        // figure out where the left edge of a fixed size thumb would have been
        int thumbX = thumbPosFromValue(getValue());
        
        // measure the width needed to display the value label in the thumb
        String valStr = labelStr(getValue());
        int valMarkW = numberFontMetrics.stringWidth(valStr)+2*numberHPad;
        
        // compute the thumb position.  this is an offset from the fixed thumb position
        // which is proportional to the percentage of the range.  at the left end we will
        // be flush with the left side of the fixed thumg.  at the right end we will be 
        // flush with the right side.
        int valNumX = thumbX - ((valMarkW-thumbW) * thumbX)/((int)(getBounds().getWidth()-thumbW)); 
        
        // fill in and return the result
        tRectResult.setBounds(valNumX,0,valMarkW, trackH); 
        return tRectResult;
    }
    
    /** Redraw the thumb of the slider */
    public void paintThumb(Graphics g) {
        // figure out where the thumb is
        Rectangle tRect = thumbRect();
        
        // draw the thumb rectangle, then the text over that
        drawShadowRect(g, (int)tRect.getX(), (int)tRect.getY(), 
                          (int)tRect.getWidth(), (int)tRect.getHeight(), false);
        g.setColor(Color.black);
        g.setFont(numberFont);
        g.drawString(labelStr(getValue()), (int)tRect.getX()+numberHPad, numberVPad+numberAscent);
    }
       
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */  
    /* Input handling */ 
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
   
    /** determine if the given point is inside the current thumb position */
    protected boolean picksThumb(int x, int y)
    {
        // anything inside our bounds vertically counts 
        if (y < 0 || y > getHeight()) return false;
        
        // compute thumb position and size as done when it was drawn
        Rectangle tRect = thumbRect();
        
        // are we inside it
        if (x < (int)tRect.getX() || x > (int)(tRect.getX() + tRect.getWidth())) return false;
        
        return true;
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Set the value of high end of slider range. */
    public void setMaxValue(long val) {showInt = true; super.setMaxValue(val); }
    
    /** Set the value of low end of slider range. */
    public void setMinValue(long val) {showInt = true; super.setMinValue(val); }
   
    /** Set the current value of the slider. */
    public void setValue(long val) {super.setValue(val); }
    
    /** 
     * Set the value of high end of slider range as a double and display doubles. 
     * Min, max, and value must all be set as doubles for this to work.
     */
    public void setMaxValue(double val) 
    {
        showInt = false; 
        super.setMaxValue((long)(val*doubleScaleFactor)); 
    }
    
    /** 
     * Set the value of low end of slider range as a double and display doubles. 
     * Min, max, and value must all be set as doubles for this to work.
     */
    public void setMinValue(double val) 
    {
        showInt = false; 
        super.setMinValue((long)(val*doubleScaleFactor)); 
    }    
    
    /** 
     * Set the current value of the slider as a double and display doubles.      
     * Min, max, and value must all be set as doubles for this to work.
     */
    public void setValue(double val) 
    {
        showInt = false; 
        super.setValue((long)(val*doubleScaleFactor)); 
    }
    
    /** 
     * Get the current value of the slider as a double. This takes care of 
     * applying the internal scale factor if needed (but also works when we 
     * are storing and displaying normal long values.  Note that calling 
     * getValue() when we are displaying doubles will return the internally
     * scaled long value we use to represent the double.
     */
    public double getDoubleValue()
    {
        if (showInt) 
            return (double)getValue();
        else
            return getValue()/(double)doubleScaleFactor; 
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* Constructor */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /** Construct with full values */
    public ParameterSlider(long minV, long maxV, long initV, int w) 
    {
        // init instance vars
        setMinValue(minV); setMaxValue(maxV); setValue(initV);
        
        // we don't draw everything
        setOpaque(false);
        
        // make sure that we get mouse and mouse motion events
        enableEvents(AWTEvent.MOUSE_EVENT_MASK | AWTEvent.MOUSE_MOTION_EVENT_MASK);
        
        // calculate a new thumb and track height,
        thumbH = 2*(numberH+numberVPad*2); 
        trackH = numberH+numberVPad*2;
        trackOffset = 0;
        
        // redo sizes based on new thumbH
        Dimension sz = new Dimension(w, thumbH);
        setSize(sz);
        setPreferredSize(sz);
        setMinimumSize(sz);
        setMaximumSize(sz);
    }
    
    /** Construct with full double values */
    public ParameterSlider(double minV, double maxV, double initV, int w) 
    {
        // set up with dummy values
        this(0,0,100,w);
        // change over to displaying the correct doubles
        setMinValue(minV); setMaxValue(maxV); setValue(initV);
    }
   
    /** Construct with full double values and default width */
    public ParameterSlider(double minV, double maxV, double initV) 
    {
        // set up with dummy values
        this(minV,maxV,initV, DEFAULT_W);
    }
    
    /** Convenience full constructor that takes ints instead of longs */
    public ParameterSlider(int minV, int maxV, int initV, int w)
    {
        this((long)minV,(long)maxV,(long)initV,w);
    }
    
    /** Construct with default min & max (integers) */
    public ParameterSlider(int w) {
        this(0,1000,0, w);
    }
    
    /** Default constructor */
    public ParameterSlider() {
        this(DEFAULT_W);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
}

