package KTEditor;

import java.awt.*;
import java.awt.geom.*;
import javax.swing.*;
import javax.swing.text.*;

/**
 * Object describing an instance of a kinetic typography effect -- that is
 * an actual application of the effect with particular parameter value settings.
 * This object provides access to the same information as an EffectDescriptor
 * i.e., a list of the tags that the effect knows about and a description of 
 * each of the parameters to the effect as a whole.  In addition it provides
 * values for each of the parameters. <p>
 *
 * The conflict testing mechanism xx more.
 *
 * xx more later
 * <p>
 * 
 * @author  Scott Hudson
 */
public class EffectInstanceDescriptor extends EffectDescriptor {

    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* instance variables */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */

    /** Values of the actual parameters */
    protected ParameterValue[] parameterValues = new ParameterValue[0];
    
    /** Get the values of the actual parameters */
    public ParameterValue[] getParameterValues() {return parameterValues;}
    
    /** 
     * Set the values of the actual parameters en mass.  This does 
     * <b>not</b> make a copy of the incoming array.  For everything to
     * operate correctly, the size of this array must be the same as
     * the value returned by getNumParamters() (i.e., the same as the 
     * size of the parameter descriptor array).  However, this is not
     * checked here.
     */
    public void setParameterValues(ParameterValue[] parms) {parameterValues = parms;}
    
    /**
     * Get a particular paramter value.
     */
    public ParameterValue getParameterValue(int indx) {return getParameterValues()[indx];}
    
    /**
     * Set a paraticular parameter value.  Note the parameter array is not
     * expanded to accomodate positions outside its current allocation.
     */
    public void setParameterValue(int indx, ParameterValue val) {getParameterValues()[indx] = val;}

    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */

    /** 
     * The user interface to the parameters of this instance.  This is a
     * container object with pre-build controls linked back to the actual 
     * parameters of this instance.  This will be constructed on demand 
     * the first time it is accessed.
     */
    protected JComponent controlUI = null;
    
     /** 
     * Get the user interface to the parameters of this instance.  This is a
     * container object with pre-build controls linked back to the actual 
     * parameters of this instance.
     */
    public JComponent getControlUI() {
        // if this is the first time out build one from scratch
        if (controlUI == null) controlUI = buildParmBox(this);
        
        return controlUI;
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */ 
    /* methods */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */ 
    
    /** 
     * Build a UI component that holds all the controls for  parameters to the
     * given effect instance.  This can safely be called with null as the 
     * parameter and will create a suitable display for cases where no effect
     * is being edited.
     */
    public static JComponent buildParmBox(EffectInstanceDescriptor forEffect)
    {
        // components we will build the overall box out of
        Box result = Box.createVerticalBox();
        Box topRow = Box.createHorizontalBox();
        Box bottomRow = Box.createHorizontalBox();   
        JPanel bottomColumn = new JPanel(new GridLayout(0,2)); 
        
        // overall we have a column with a row, separator, and another row with a column inside it
        result.add(Box.createVerticalStrut(5));
        result.add(topRow);
        result.add(Box.createVerticalStrut(5));
        result.add(new JSeparator());
        result.add(Box.createVerticalStrut(5));
        result.add(bottomRow);
        bottomRow.add(Box.createVerticalStrut(0));
        bottomRow.add(bottomColumn);
        result.add(Box.createVerticalGlue());  // all extra space at the bottom
        
        // in the top row put a label and a box with tag buttons
        topRow.add(Box.createHorizontalStrut(5));
        Box effectLabels = Box.createVerticalBox();
        JLabel lab = new JLabel("Effect:");
        lab.setFont(lab.getFont().deriveFont(12.0f));
        JLabel effectName;
        if (forEffect == null)
            effectName = new JLabel("No Effect Selected...");
        else
            effectName = new JLabel(forEffect.getName());
        effectName.setFont(effectName.getFont().deriveFont(18.0f));
        effectName.setEnabled(false);
        effectLabels.add(lab); 
        effectLabels.add(effectName);
        topRow.add(effectLabels);
        
        //topRow.add(Box.createHorizontalStrut(10));  //xx goes back once we have buttons
        topRow.add(Box.createHorizontalGlue());
        
        // this is the box for the tag buttons, but not done yet... xx
        // JPanel tagBox = new JPanel();
        // tagBox.add(new JLabel("Mark Selection With:"));
        // tagBox.add(Box.createVerticalStrut(5));
        // JComponent tagList = createTagList(xxTestTags);  //xx later change this to buttons from effect
        // tagBox.add(tagList);
        // tagBox.add(Box.createVerticalStrut(5));
        // topRow.add(tagBox);
        // topRow.add(Box.createHorizontalStrut(0));
        
        // Now we fill up the bottom with the actual parameter controls
        if (forEffect == null || forEffect.getNumParameters() == 0)
        {
            // we either don't have an effect or it has no controls, put up label
            JLabel parmContent = new JLabel("No parameter controls for this effect...");
            parmContent.setHorizontalTextPosition(SwingConstants.LEFT);
            parmContent.setVerticalTextPosition(SwingConstants.TOP);
            parmContent.setEnabled(false);
            
            bottomColumn.add(parmContent);
        }
        else
        {
            // create a control interface for each paramter
            for (int i=0; i < forEffect.getNumParameters(); i++)
            {
                // pull the parameter, its type, and the actual object for its value
                ParameterDescriptor parm = forEffect.getParameter(i);
                ParameterTypeDescriptor parmT = parm.getType();
                ParameterValue modelVal = forEffect.getParameterValue(i);
                
                // build an interface for that and put it in the result;
                bottomColumn.add(parmT.buildInterface(modelVal, parm.getName()));
            }
        }  
        return result;
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    // Spacing constants for drawing 
    
    /** font we use for drawing labels */
    protected Font labelFont = new Font("SansSerif", Font.BOLD, 9);
    /** font metrics for labelFont */
    protected FontMetrics labelMetrics = Toolkit.getDefaultToolkit().getFontMetrics(labelFont); 
    /** spacing above the label */
    protected int labelPadTop = 3;
    /** spacing below the label */
    protected int labelPadBottom = 3;
    /** Spacing before the label */
    protected int labelPadLeft = 2;
    /** height of the higlight bar that we are drawing under the text */
    protected int barH = labelMetrics.getHeight() + labelPadTop + labelPadBottom;
    /** spacing below the baseline */
    protected int basePad = 5;
    /** Rounding radius for rounded tab shape */
    protected int tabRound = 10; //xx ??
   
    /** Custom drawing for effect highlights.  xx more */ 
    public void defaultDrawHighlightRect(
        Graphics g, 
        Rectangle drawRect,
        int baselineOff,
        JTextComponent inComponent, 
        int startPos, int endPos, 
        View forView, 
        Shape viewBounds, 
        int ends) 
    {   
        // adjust the draw rectangle to end at baseline + pad instead of descenders
        drawRect = baselineRect(drawRect, baselineOff+basePad);
        
        // draw in our color 
        g.setColor(getSegmentColor());
        
        // are we drawing at the left end?
        if ((ends & TagArtist.LEFT_END) != 0)
            // do a narrow vertical rectangle within the text 
            g.fillRect((int)drawRect.getX(), (int)drawRect.getY(), 2, (int)drawRect.getHeight());
        
        // are we drawing at the right end?
        if ((ends & TagArtist.RIGHT_END) != 0)
            // do a narrow vertical rectangle within the text 
            g.fillRect((int)drawRect.getMaxX()-2, (int)drawRect.getY(), 2, (int)drawRect.getHeight());
               
        //draw a small rectangle under regular rect using our color
        //g.fillRect((int)drawRect.getX(), (int)drawRect.getMaxY(), (int)drawRect.getWidth(), barH);
        g.fillRect((int)drawRect.getX(), (int)drawRect.getMaxY(), (int)drawRect.getWidth(), labelPadTop);
    }
    
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    
    /**
     * Default drawing for text at the start of a tag.
     *
     * Here we draw the text in a small font within the area of the highlight 
     * bar below the text (drawn in pieces by defaultDrawHighlightRect).
     */
    public void defaultDrawStartText(Graphics g, String text, Rectangle drawRect, int baselineOff) 
    {
        Graphics2D g2 = (Graphics2D)g;
        
        // adjust the draw rectangle to end at baseline instead of descenders
        drawRect = baselineRect(drawRect, baselineOff+basePad);
        

        // computing sizing info
        Rectangle2D textBound = labelFont.getStringBounds(text,g2.getFontRenderContext());
        int tabW = ((int)textBound.getWidth())+labelPadLeft*2;
        
        // set up to draw text for the label xx was: clipped to the bar that will have been already drawn
        g2.setFont(labelFont);
        // Rectangle oldClip = g2.getClipBounds(); 
        // g2.setClip((int)drawRect.getX(), (int)drawRect.getMaxY(), (int)drawRect.getWidth(), barH);
        
        // draw tab rounded on the bottom just under the highlight bar already 
        // drawn by drawHighlightRect
        int x0 = (int)drawRect.getX();
        int y0 = ((int)drawRect.getMaxY())+labelPadTop;
        GeneralPath tabPath = new GeneralPath();
        tabPath.moveTo(x0, y0);
        tabPath.lineTo(x0, y0+barH-tabRound);
        tabPath.quadTo(x0, y0+barH, x0+tabRound, y0+barH);
        tabPath.lineTo(x0+tabW-tabRound, y0+barH);
        tabPath.quadTo(x0+tabW,y0+barH, x0+tabW,y0+barH-tabRound);
        tabPath.lineTo(x0+tabW, y0);
        tabPath.closePath();
        g2.setColor(getSegmentColor());
        g2.fill(tabPath);
        
        // draw text in black shadow, then white
        g2.setColor(Color.black);
        g2.drawString(text, (int)drawRect.getX()+1,
                            (int)drawRect.getMaxY()+labelMetrics.getAscent()+labelPadTop+1);
        g2.setColor(Color.white);
        g2.drawString(text, (int)drawRect.getX()+1,
                            (int)drawRect.getMaxY()+labelMetrics.getAscent()+labelPadTop);
        // restore the clip
        //g2.setClip(oldClip);
    }
       
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
    /* constructors */
    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */

    /** 
     * Construct a descriptor for a kinetic typography effect instance. 
     */
    public EffectInstanceDescriptor(
      String                nm,
      Color                 clr,
      TagDescriptor[]       tagList,
      ParameterDescriptor[] paramList,
      ParameterValue[]              paramValues)
      {
        super(nm,clr,tagList,paramList);
        if (paramValues == null && paramList != null) 
            paramValues = new ParameterValue[paramList.length];
        setParameterValues(paramValues);
      }

    /** 
     * Construct a descriptor for a kinetic typography effect instance.  
     * This descriptor will have no tags, parameters or parameter values.
     */
    public EffectInstanceDescriptor(String nm, Color clr) 
       {this(nm,clr,null,null,null);}    
    
    /** 
     * Construct a descriptor for a kinetic typography effect instance.  
     * This descriptor will no tags, parameters, or parameter values.
     */
    public EffectInstanceDescriptor(String nm) {super(nm);}

    /** 
     * Construct a default descriptor for a kinetic typography effect 
     * instance.  To be practically useful this descriptor will need to 
     * at least be given a name.
     */
    public EffectInstanceDescriptor() {super();}
    
    /** 
     * Construct an effect instance from an effect descriptor.  This copies most
     * information, then creates a fresh set of parameter values.
     */
    public EffectInstanceDescriptor(EffectDescriptor fromEff)
    {
        // initialize parts from the superclass
        super(fromEff.getName(), fromEff.getSegmentColor(), fromEff.getTags(), 
              fromEff.getParameters());
        
        // now allocate space for values and create intial values for all 
        ParameterValue[] pVals = new ParameterValue[getParameters().length];
        for (int i = 0; i < getNumParameters(); i++)
            pVals[i] = getParameter(i).getType().createValue(getParameter(i).defaultValue());
        setParameterValues(pVals);
    }

    /* . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . */
}
