#ifndef _LSYSTEM3D_H_
#define _LSYSTEM3D_H_

#include <string>
#include <iostream>
#include <vector>
#include <map>

#ifdef _WIN32
#include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>

class LSystemNode;

typedef std::vector<LSystemNode*> LSystemNodeArray;
typedef std::map<char, LSystemNode*> LSystemRuleSet;

//////////////////////////////////////////////////////////////////////

/** Class for defining, parsing, and rendering Lindenmayer systems
 *  (L-Systems).
 * 
 *  L-Systems are specified by an initial string or axiom, and a set
 *  of replacement rules which are applied recursively to the string.
 *  The initial axiom and the replacement rules contain commands for
 *  drawing cylinders and/or cones in 3D, commands for rotating the
 *  coordinate system as well as operations to scale the cone length
 *  and radius. It also supports push and pop operations to allow
 *  branching constructions.
 *
 *  Here are the commands which may be placed into an L-System expression:
 *
 *    ##L - scale length (percent)
 *    ##R - scale radius (percent)
 *    ##S - scale length and radius (percent)
 *
 *    ##X - rotate by angle around X-axis (degrees)
 *    ##Y - " Y-axis
 *    ##Z - " Z-axis
 *
 *    [   - push context (including angles, scales, turtle position)
 *    ]   - pop context. Expressions and subexpressions must be balanced.
 *
 *    ##_ - where _ is any other letter - draw a cone along the Y axis
 *          leading number is optional and indicates how much smaller (or larger)
 *          the end of the cone is than the start of the cone
 *
 *    *   - draw a leaf
 *
 *  Initially, the length is set at 1 unit and the radius is set at 0.15 units.
 *  All scaling (for L, R, S, and final cone radius) is done by percent. Using
 *  negative numbers is equivalent to reciprocals (i.e, -50 S scales length
 *  and radius by 200%).
 *
 *  BNF for expressions:
 *  
 *    expr := <empty>
 *    expr := expr expr
 *    expr := [ expr ]
 *    expr := number L
 *    expr := number R
 *    expr := number S
 *    expr := number X
 *    expr := number Y
 *    expr := number Z
 *    expr := number id
 *    expr := id
 *    id   := <any capital letter besides L,R,S,X,Y,Z>
 *
 *  BNF for rules:
 *
 *    rule := id: expr
 *
 *  See the file lsystem_impl.h to understand how the L-System is
 *  actually implemented.
 *
 */


class LSystem {
public:

  /** Constructor. */
  LSystem();

  /** Construct from axiom and rules. */
  LSystem(const std::string& axiom,
	  const std::vector<std::string>& rules,
	  unsigned int maxDepth = 1);

  /** Construct from axiom and rules. */
  LSystem(const char* axiom,
	  const char* const rules[],
	  unsigned int maxDepth = 1);

  /** Destructor. */
  virtual ~LSystem();

  /** Parse an L-System. */
  void parse(const std::string& axiom,
	     const std::vector<std::string>& rules);

  /** Parse an L-System. 
   *  The rules array must be null-terminated (i.e., the last
   *  string in the rules array must be the NULL pointer.
   */
  void parse(const char* axiom,
	     const char* const rules[]);

  /** The maximum recursion depth (default 1) */
  unsigned int maxDepth() const;

  /** Set the maximum depth of recursion. */
  void setMaxDepth(unsigned int d);

  /** The root L-System node (see lsystem_impl.h) */
  const LSystemNode* root() const;

  /** Set of replacement rules. */
  const LSystemRuleSet& rules() const;

  /** Clear the L-System and delete all private data. */
  void clear();

  /** Render the L-System to the currently specified depth
   *  on the current OpenGL context. */
  void renderGL() const;

private:

  LSystemNode* _root;
  LSystemRuleSet _rules;
  mutable GLUquadric* _quadric;
  unsigned int _maxDepth;

  LSystemNode* _parseExpression(std::istream& istr);
  LSystemNode* _parseRule(std::istream& istr, char& id);

};


#endif
