% Gaze concurrence estimation demo
% This script generates random synthetic data (primary gaze rays with two gaze concurrences) and
% estimate the gaze concurrences using the method of Equation (6) in 
% Park, Jain, and Sheikh, "3D Social Saliency from Head-mounted Cameras"
% (NIPS 2012)
% Written by Hyun Soo Park (hypar@seas.upenn.edu)

cone_bandwidth = 1;
merge_threshold = 1e-4;
nInitializedPointsPerRay = 10; % The number of initialized points per primary gaze ray
range = 10; % Initialized point range
nIters = 1000; % The number of maximum meanshift iterations

% Generate synthetic data (gaze concurrence points and primary gaze rays)
% Indices for primary gaze rays
point2cam = [1,2,3,4,5;
             6,7,8,9,10];

for i = 1 : size(point2cam,1)
    % Generate gaze concurrence
    gc(:,i) = 20*i;
    for j = 1 : size(point2cam,2)
        % Set the center of primary gaze ray
        center(:,point2cam(i,j)) = gc(:,i)+20*(rand(3,1)-0.5);
        % Set the direction of primary gaze ray
        direction(:,point2cam(i,j)) = gc(:,i)-center(:,point2cam(i,j));
        % Add noise
        direction(:,point2cam(i,j)) = direction(:,point2cam(i,j)) + 0.1*randn(3,1);
        % Normalization
        direction(:,point2cam(i,j)) = direction(:,point2cam(i,j))/norm(direction(:,point2cam(i,j)));
    end
end
         
% Given primary gaze ray measurement, estimate gaze concurrences
Y = [];
W = [];
GazeConcurrence = [];
% A gaze concurrence point is initialized as a point along the primary gaze
% ray
traj_idx = 1;
for i = 1 : size(center,2)
    % Multiple point initialization per primary gaze ray
    for k = 1 : nInitializedPointsPerRay
        % Gaze concurrence initialization
        x = center(:,i)+range*k/nInitializedPointsPerRay*direction(:,i);
        traj = x;
        isBad = 0;
        while 1
            [x1 weight] = GazeConcurrenceMeanShift(x, center, direction, cone_bandwidth);
                        
            if abs(norm(traj(:,end)-x1)) < 1e-6
                break;
            end            
            traj = [traj x1];
            if (size(traj,2) > nIters)
                isBad = 1;
                break;
            end       
            
            % The gaze concurrence must be reconstructed more than two primary
            % gaze ray
            a = sort(weight);
            if a(end-1)/a(end) < 0.1
                isBad = 1;
                break;
            end
            x = x1;
        end
        if isBad
            continue;
        end
        
        Meanshift_trajectory{traj_idx} = traj;
        traj_idx = traj_idx + 1;
        if (isempty(GazeConcurrence))
            GazeConcurrence = traj(:,end);
        else
            isIn = 0;
            for j = 1 : size(GazeConcurrence,2)
                err = norm(GazeConcurrence(:,j)-traj(:,end));
                if (err < merge_threshold)
                    isIn = 1;
                    break;
                end
            end
            if (~isIn)
                GazeConcurrence(:,end+1) = traj(:,end);
            end
        end       
    end
end

% Plot result
figure(1)
clf;
% Plot primary gaze rays
plot3(center(1,:), center(2,:), center(3,:), 'ko', 'MarkerFaceColor', 'w', 'LineWidth', 2);
for i = 1 : size(center,2)
    hold on
    plot3([center(1,i) center(1,i)+3*direction(1,i)], [center(2,i) center(2,i)+3*direction(2,i)], [center(3,i) center(3,i)+3*direction(3,i)],...
         '-', 'Color', [0.5 0.5 0.5], 'LineWidth', 2);
end

% Plot meanshift trajectories
for i = 1 : length(Meanshift_trajectory)
    hold on
    plot3(Meanshift_trajectory{i}(1,:), Meanshift_trajectory{i}(2,:), Meanshift_trajectory{i}(3,:), 'r-');
end

% Plot gaze concurrences
hold on
plot3(GazeConcurrence(1,:), GazeConcurrence(2,:), GazeConcurrence(3,:), 'ro', 'MarkerFaceColor', 'r', 'LineWidth', 2);

axis equal
grid on



