// test-format.c : unit test for the Common Lisp style output formatter
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with Foobar; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <libstd.h>
#include <string_port.h>
#include <errcodes.h>
#include <libLPC2xxx.h>

#if !defined(TARGET_OSX) && !defined(TARGET_POSIX)
#include <LED_indicators.h>
#include <UART_raw_port.h>
static UART_raw_port console_port;
#endif

#define CHARS 1000
static char string_buffer[CHARS];
string_port strport;

/****************************************************************/
static int 
format_test( char *name, char *format_string, int arg, char *expected )
{
  int err;
  // just reset the port rather than seeking
  string_port_init( &strport, &string_buffer, CHARS, PORT_FLAGS_WRITE );

  // create a C string in the string port
  format( (struct port_t *) &strport, format_string, arg );
  string_port_zero_terminate( &strport );

  // return non-zero if the string doesn't match the expected value
  err = strcmp ( string_buffer, expected );

  // emit the string for the user to read
  format( NULL, "~c: ~a : '~a'\n", (err) ? 'N' : 'y', name, string_buffer );
  
  return err;
}

/****************************************************************/
int run_tests(void)
{
  int errors = 0;

  format( NULL, "test-format (revision " SVN_REVISION ") waking up.\n" );

  if (format_test ("string ", "~a", (int) "this is a string", "this is a string" )) errors++;
  if (format_test ("decimal", "~d", 42,                       "42"               )) errors++;
  if (format_test ("decimal", "~d", 0x7fffffff,               "2147483647"       )) errors++;
  if (format_test ("decimal", "~d", 0x80000000,               "-2147483648"      )) errors++;
  if (format_test ("decimal", "~d", 0x80000001,               "-2147483647"      )) errors++;
  if (format_test ("decimal", "~d", 0xffffffff,               "-1"               )) errors++;
  if (format_test ("hex    ", "~x", 42,                       "0000002A"         )) errors++;
  if (format_test ("percent", "%" , 0,                        "%"                )) errors++;
  if (format_test ("tilde  ", "~~", 0,                        "~"                )) errors++;
  if (format_test ("invalid", "~q", 0,                        "~q"               )) errors++;
  if (format_test ("invalid", "~232q", 0,                     "~q"               )) errors++;
  if (format_test ("truncated", "~", 0,                       "~"                )) errors++;
  if (format_test ("truncated", "~234", 0,                    "~"                )) errors++;

#if LIBSTD_FORMAT_SUPPORTS_BINARY
  if (format_test ("binary ", "~b", 42,          "00000000000000000000000000101010" )) errors++;
  if (format_test ("binary ", "~b", 0xabcd0124,  "10101011110011010000000100100100" )) errors++;
#endif

#if LIBSTD_FORMAT_SUPPORTS_FIXED
  if (format_test ("fixed  ", "~m", FLOAT_TO_FIXED( 0.5  ), "0.500000000" )) errors++;
  if (format_test ("fixed  ", "~m", FLOAT_TO_FIXED( 0.123), "0.122999999" )) errors++;
#endif

  format( NULL, "\nwith field width specifiers:\n");
  if (format_test ("decimal         ", "~4d", 42,                       "42"               )) errors++;
  if (format_test ("decimal zerofill", "~04d", 42,                       "42"               )) errors++;

  // this one differs on each build
  format( NULL, "\nptr    : '~p'\n", run_tests );
  
  format( NULL, "\ntest-format finished with ~d errors\n", errors);
  return errors;
}

/****************************************************************/
#if !defined(TARGET_OSX) && !defined(TARGET_POSIX)
int main (void)
{
  int i;
  UART_raw_port_init( &console_port, &UART0, UART0_DIVISOR, PORT_FLAGS_READ|PORT_FLAGS_WRITE|PORT_FLAGS_CRLF );
  current_output_port = (struct port_t *) &console_port;
  current_input_port  = (struct port_t *) &console_port;

  run_tests();

  while ( 1 ) {

#if CONFIG_CPU_LED_MASK
    for ( i = 0; i < 10000000; i++ ) ;
    CPU_LED_on();
    for ( i = 0; i < 10000000; i++ ) ;
    CPU_LED_off();
#endif
  }

}
#endif

/****************************************************************/
#if defined(TARGET_OSX) || defined(TARGET_POSIX)
int main (int argc, char **argv)
{
  return run_tests();
}
#endif



