// port.c : abstract interface to character devices a la Scheme
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// Generic port access functions.  This offer some protection against
// missing methods and closed ports.
#include <libstd.h>
#include <errcodes.h>
#include <port.h>

/****************************************************************/
// A valid port must be initialized but not closed.
static int port_valid( struct port_t *p) 
{
  return ( p != NULL && 
	   p->methods != NULL &&
	   ((p->flags & (PORT_FLAGS_INITIALIZED | PORT_FLAGS_CLOSED)) == (PORT_FLAGS_INITIALIZED))
	   );
}

/****************************************************************/ 
// Generic functions which may be used by specific port
// implementations.

void
generic_port_init( struct port_t *p, const struct port_class_t *methods, unsigned mode )
{
  p->methods = methods;
  p->flags = mode | PORT_FLAGS_INITIALIZED;
  p->peeked_char = -1;
}

int
generic_port_peek_char( struct port_t *p )
{
  if ( p->peeked_char == -1 ) p->peeked_char = port_read_char( p );
  return p->peeked_char;
}

int
generic_port_write( struct port_t *p, void *buf, size_t nbytes )
{
  unsigned char *data = (unsigned char *) buf;
  int count = 0;

  while ( nbytes-- > 0 ) {
    int err = port_write_char( p, *data++ );
    if ( err ) return err;
    count++;
  }
  return count;
}

/****************************************************************/
// Non-blocking ports will return EOF if no data is available.  For
// this reason non-blocking ports won't work with the Scheme reader.

// I may want to change this so that non-blocking ports return
// -ERRNODATA.  It will affect the implementation of peek_char, etc.

// It seems right to implement peeking here, will it ever need to be
// overridden?

int port_read_char (struct port_t *p)
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( !(p->flags & PORT_FLAGS_READ ) ) return -ERRINVAL;
  if ( p->methods->read_char == NULL ) return EOF;
  else {
    if ( p->peeked_char != -1 ) {
      int c = p->peeked_char;
      p->peeked_char = -1;
      return c;
    } else {
      return p->methods->read_char( p );
    }
  }
}

// Most implementations will use generic_port_peek_char, but some
// might use a more direct approach.
int port_peek_char (struct port_t *p)
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( !(p->flags & PORT_FLAGS_READ ) ) return -ERRINVAL;
  if ( p->methods->peek_char == NULL ) return EOF;
  else return p->methods->peek_char( p );
}

// A true value guarantees that the next read_char will not hang.
// However, EOF counts as a character as far as char_ready is
// concerned.  So if unimplemented, defaults to true, so only blocking
// ports will need to implement it.
int port_char_ready (struct port_t *p)
{
  if ( !port_valid(p) || p->methods->char_ready == NULL ) return 1;
  else return p->methods->char_ready( p );
}

int port_write_char (struct port_t *p, int c)
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( !(p->flags & PORT_FLAGS_WRITE ) ) return -ERRINVAL;  
  if ( p->methods->write_char == NULL ) return ERRNOERROR;

  if ( c == '\n' && (p->flags & PORT_FLAGS_CRLF)) {
    return ( p->methods->write_char( p, '\r' ) ||
	     p->methods->write_char( p, c ) );
  } else {
    return p->methods->write_char( p, c );
  }
}

// Mark the port as closed, which will also make it invalid for future
// operations.  The close method may be unimplemented if no low-level
// close action is required.
int port_close( struct port_t *p )
{
  if ( !port_valid(p) ) return -ERRINVAL;

  p->flags |= PORT_FLAGS_CLOSED;

  if ( p->methods->close == NULL ) return ERRNOERROR;
  else return p->methods->close( p );
}

// Read and write and seek just validate arguments and call the
// method.  Implementations should be careful to handle peek_char
// and PORT_FLAGS_CRLF correctly.

// I may implement a generic method which just repetitively calls
// read_char, etc., for implementations to use.

size_t port_read( struct port_t *p, void *buf, size_t nbytes )
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( !(p->flags & PORT_FLAGS_READ ) ) return -ERRINVAL;
  if ( buf == NULL ) return -ERRINVAL;
  if ( nbytes < 0 ) return -ERRINVAL;
  if ( p->methods->read == NULL ) return -ERRNOSYS;
  return p->methods->read( p, buf, nbytes );
}

size_t port_write( struct port_t *p, void *buf, size_t nbytes )
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( !(p->flags & PORT_FLAGS_WRITE ) ) return -ERRINVAL;
  if ( buf == NULL ) return -ERRINVAL;
  if ( nbytes < 0 ) return -ERRINVAL;
  if ( p->methods->write == NULL ) return generic_port_write( p, buf, nbytes );
  return p->methods->write( p, buf, nbytes );
}

// I may implement a generic method which can seek forward by reading
// the intervening data.
off_t port_seek( struct port_t *p, off_t offset, int mode )
{
  if ( !port_valid(p) ) return -ERRINVAL;
  if ( mode < PORT_SEEK_SET || mode > PORT_SEEK_END ) return -ERRINVAL;
  if ( p->methods->seek == NULL ) return -ERRNOSYS;
  return p->methods->seek( p, offset, mode );
}
