// memalloc.c - a simple heap allocator for a fixed size memory pool.
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// This is not an exact replacement for malloc/free since the blocks
// are limited to a relatively small size, so it uses different
// functions names to avoid conflict.

// Each block in the memory pool has a single word header which has
// double linked list pointers to the adjacent blocks, stored as the
// integer index of the block header in the pool.  The integer indices
// save storage and are suitable for a fixed sized heap.  The double
// linked list allows deallocation in constant time.

// The free list is a separate double linked list which takes
// advantage of the fact that every free block has unused data
// space which can hold the free list pointers.

// The double linked lists are implemented with a sentinel node
// in the beginning of the heap, which simplifies the list updates.

// Typically, the minimum space usage is 8 bytes, then increases in
// four byte increments.  A mild internal fragmentation is possible if
// a block exists which is only four bytes longer than requested,
// since it can't be split, so the returned block includes the extra
// space.

/****************************************************************/
#include <libstd.h>
#include <errcodes.h>

// The heap blocks are chained in a double linked list in storage
// order.  The block size is implied by the offset between adjacent
// blocks.  In principle, a much larger heap (of small blocks) could
// be used by storing just the offset to the adjacent blocks, but at
// the cost of increased complexity in the merging of free blocks, and
// an increase in the size of the free list pointers.

struct mem_block_header {
  unsigned prev: 15;  // the heap position of the previous block
  unsigned next: 15;  // the heap position of the next block
  unsigned free:  1;  // true if the block is unallocated, and thus contains free list pointers
  unsigned mark:  1;  // extra bit 
};

// The free blocks are chained into a separate free list.  A
// free_block_header always immediately follows the mem_block_header
// in a free block.  The "heap position" values indicate the offset
// within the heap of the complete block, i.e., the position of the
// mem_block_header for another free block.

struct free_block_header {
  unsigned prev: 15;    // the heap position of the previous free block
  unsigned next: 15;    // the heap position of the next free block
  unsigned padding :2;  // unused bits
};

// Each unit of heap memory is structured in one of these formats.
union heap_unit {
  unsigned u;
  struct mem_block_header  blkhdr;
  struct free_block_header freehdr;
};

// The minimum allocatable unit, typically four bytes.
#define MIN_BLOCK_SIZE (sizeof(union heap_unit))

// Retrieve the heap size from the heap header.
#define HEAP_POOLSIZE(heap) (heap[2].u)

// Define a static pointer to the heap area specified by the user.
static union heap_unit *heap = NULL;

/****************************************************************/
// Initialize the user-provided memory area as a heap.  The pool_size
// is specified in bytes.  A pointer is retained for subsequent
// allocation operations.

int
init_memalloc( void *memory, size_t pool_size )
{
  heap = (union heap_unit *) memory;

  // structure the first block as the root of the block and free lists
  heap[0].blkhdr.prev  = 3;   // create a circular double-linked list
  heap[0].blkhdr.next  = 3;   
  heap[0].blkhdr.free  = 0;   // the root block is not a free block
  heap[1].freehdr.prev = 3;   // create a circular double-linked list
  heap[1].freehdr.next = 3;
  heap[2].u            = pool_size / MIN_BLOCK_SIZE;

  // structure the remainder as a large free block
  heap[3].blkhdr.prev  = 0;
  heap[3].blkhdr.next  = 0;
  heap[3].blkhdr.free  = 1;
  heap[4].freehdr.prev = 0;
  heap[4].freehdr.next = 0;

  return ERRNOERROR;
}

/****************************************************************/
// Comparable to malloc, but operating from a fixed heap with a modest
// block size limit.

void *
memalloc( size_t size )
{
  unsigned idx;

  // Round the requested allocation size up to the unit size, convert
  // from bytes to the number of heap units, and add a unit for the header
  // to get the full block size required.

  unsigned units_needed = ((size + (MIN_BLOCK_SIZE-1)) / MIN_BLOCK_SIZE) + 1;
  
  // walk the free list looking for a block large enough
  idx = heap[1].freehdr.next;

  // the end of the circular free list is indicated by the initial sentinel at offset 0
  while ( idx != 0 ) {
    // compute the size of a free list block
    unsigned next_block = heap[idx].blkhdr.next;  // position of the next adjacent block
    unsigned block_size;

    // if the next block is the sentinel, this is the last block in the heap
    if ( next_block < idx) block_size = HEAP_POOLSIZE(heap) - idx;
    else block_size = next_block - idx; // else the raw size is just the offset

    // select the first block which is large enough to use, splitting it as needed
    if ( block_size >= units_needed ) {

      // prepare to modify the free list by retaining the offsets of the neighboring free blocks
      unsigned prev_free = heap[idx+1].freehdr.prev;
      unsigned next_free = heap[idx+1].freehdr.next;

      // mark the block as non-free
      heap[idx].blkhdr.free = 0;

      // If the block is nearly the right size, return as-is.  If the
      // block is just one unit longer than requested, it cannot be
      // split, since the fragment wouldn't be large enough to be a
      // free block.

      if ( block_size <= units_needed+1) {
	heap[prev_free+1].freehdr.next = next_free;  // remove block from the free list
	heap[next_free+1].freehdr.prev = prev_free;

	// return a pointer to the block data area, one unit past the mem_block_header
	return (void *) &heap[idx+1];

      } else {	
	// Else split the block, returning the first part and creating
	// a new free block from the second part.

	unsigned new_hdr = idx + units_needed;   // position of the header for the remaining free fragment

	// insert the newly created fragment into the sequential block list
	heap[idx].blkhdr.next = new_hdr;           // the free fragment is after the allocated block
	heap[next_block].blkhdr.prev = new_hdr;
	heap[new_hdr].blkhdr.prev = idx;
	heap[new_hdr].blkhdr.next = next_block;

	heap[new_hdr].blkhdr.free = 1;             // mark the new fragment block as free

	// Add the split remnant back into the free list.
	heap[new_hdr+1].freehdr.prev = prev_free;
	heap[new_hdr+1].freehdr.next = next_free;
	heap[prev_free+1].freehdr.next = new_hdr;
	heap[next_free+1].freehdr.prev = new_hdr;

	// return a pointer to the block data area, one unit past the mem_block_header	
	return (void *) &heap[idx+1];	
      }
    }

    // if the block wasn't large enough, advance to the next block in the free list
    idx = heap[idx+1].freehdr.next;
  }

  // if no block was large enough, return a null pointer
  return NULL;
}

/****************************************************************/
// De-allocate a block obtained from memalloc.  This will fail
// horribly if given some other memory.

void 
memfree( void *p )
{
  unsigned header, prev_head, next, prev, nextfree, prevfree;
  union heap_unit *block_header;

  // minimal error check
  if ( p == NULL || heap == NULL ) return;

  // find the position of the header for the given block
  block_header = ((union heap_unit *) p) - 1;
  if ( block_header < heap ) return;  // sanity check
  header = block_header - heap;       // else find the integer index of the block header

  // Mark the block as free.
  heap[header].blkhdr.free = 1;

  // Splice the newly free block into the head of the free list, after
  // the sentinel node.  This makes the heap consistent, if not
  // optimally merged.
  prev_head = heap[1].freehdr.next;           // save previous head of the free list
  heap[1].freehdr.next = header;              // make the newly freed block the new head of the free list
  heap[prev_head+1].freehdr.prev = header;    // update back pointer for old head
  heap[header+1].freehdr.prev = 0;            // set back pointer for new head
  heap[header+1].freehdr.next = prev_head;    // set forward pointer for new head

  // Now merge as many adjacent blocks as possible to reduce
  // fragmentation.  There won't be any problem with block wraparound
  // because the sentinel node (before the first block/after the last
  // block) is considered non-free and thus unmergeable.  If the heap
  // is consistent this will do at most two merges, since the adjacent
  // free blocks shouldn't also have free neighbors.

  for (;; ) {
    next = heap[header].blkhdr.next;  // adjacent blocks
    prev = heap[header].blkhdr.prev;

    // if the previous block is also a free block, it can be merged with the current free block
    if ( heap[prev].blkhdr.free ) {

      // just by splicing this block out of the block list it implicitly joins the previous block
      heap[prev].blkhdr.next = next;
      heap[next].blkhdr.prev = prev;

      // same to splice it out of the free list
      nextfree = heap[header+1].freehdr.next;
      prevfree = heap[header+1].freehdr.prev;
      heap[prevfree+1].freehdr.next = nextfree;
      heap[nextfree+1].freehdr.prev = prevfree;

      // then continue merging from the newly enlarged block
      header = prev;
      continue;

    } else if ( heap[next].blkhdr.free ) {
      // if the following block is free, just start again there, which
      // will back merge with this block
      header = next;
      continue;

    } else return;
  }
}
