// libstd.h : configuration file for libstd, a libc replacement using a Scheme compatible API
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// A source-code configurable library of standard machine-independent functions.
// Some are trivial implementations of standard C library functions.

#ifndef __LIBSTD_H_INCLUDED__
#define __LIBSTD_H_INCLUDED__

// Include a user-defined source-code configuration file.
#include <config.h> 

// Define the variable arguments macros.
#include <stdarg.h>

/****************************************************************/
// Types and standard constants.

#ifndef LIBSTD_USE_SYSTEM_C_DEFINITIONS
#define LIBSTD_USE_SYSTEM_C_DEFINITIONS 0
#endif


#if LIBSTD_USE_SYSTEM_C_DEFINITIONS

#include <stdlib.h>
#include <unistd.h>

#else // !LIBSTD_USE_SYSTEM_C_DEFINITIONS

typedef int size_t;
typedef int off_t;
#define NULL ((void *) 0)

#endif // LIBSTD_USE_SYSTEM_C_DEFINITIONS

#ifndef EOF
#define EOF  ((int) -1)
#endif

/****************************************************************/
// A simple port abstraction.  Every port has a class pointer to a
// private structure which holds the methods for the port.

struct port_class_t;  // an opaque class definition

struct port_t {
  const struct port_class_t *methods;
  unsigned short flags;
  short peeked_char;
};

#define PORT_FLAGS_READ        1
#define PORT_FLAGS_WRITE       2
#define PORT_FLAGS_INITIALIZED 4
#define PORT_FLAGS_CLOSED      8
#define PORT_FLAGS_CRLF        16  // if true, write_char expands newlines to cr-lf sequences

// additional flags to specify port opening behavior, not necessarily stored in the port flags
#define PORT_FLAGS_CREATE      (0x00010000)     // create file if it doesn't exist
#define PORT_FLAGS_TRUNCATE    (0x00020000)     // set file length to zero; implied by create

// The current ports are the default source or destination for data.
// Any routine which temporarily changes these should restore the
// previous value when complete.
extern struct port_t *current_output_port;
extern struct port_t *current_input_port;

// Generic port access functions.  Not all port types implement all operations.  
extern int port_read_char (struct port_t *p);
extern int port_peek_char (struct port_t *p);
extern int port_char_ready (struct port_t *p);
extern int port_write_char (struct port_t *p, int c);
extern int port_close( struct port_t *p );
extern size_t port_read( struct port_t *p, void *buf, size_t nbytes );
extern size_t port_write( struct port_t *p, void *buf, size_t nbytes );

extern off_t port_seek( struct port_t *p, off_t offset, int mode );
#define PORT_SEEK_SET 1  // the possible seek modes
#define PORT_SEEK_CUR 2
#define PORT_SEEK_END 3

// Some specialized types of ports.

typedef struct string_port_t string_port;
extern int string_port_init( struct string_port_t *p, void *buffer, size_t chars, unsigned mode );

typedef struct port_t null_port;
extern int null_port_init( null_port *p );

// Standard ports, in standard_ports.c, which use the C putchar() and getchar() for I/O.
extern struct port_t standard_output_port;
extern struct port_t standard_input_port;

// By default, the current I/O ports are not mapped since there is no
// standard choice of I/O configuration, but this allows the use of
// the putchar()/getchar() interface by default.
#ifndef LIBSTD_STANDARD_PORTS_DEFAULT
#define LIBSTD_STANDARD_PORTS_DEFAULT 0
#endif

/****************************************************************/
// Standard libc functions.
extern void  *memcpy  (void *dst, const void *src, size_t len);
extern int    memcmp  (const void *b1, const void *b2, size_t len);
extern void  *memset  (void *b, int c, size_t len);
extern int    strcmp  (const char *s1, const char *s2);
extern int    strncmp (const char *s1, const char *s2, size_t len);
extern size_t strlen  (const char *s);
extern int    atoi    (const char *nptr );

// character types
extern int isspace(int c);
extern int isalpha(int c);
extern int isdigit(int c);
extern int toupper(int c);

// bad random numbers
#define RAND_MAX 0x7fffffff
extern int rand(void);
extern void srand(unsigned seed);

/****************************************************************/
// A partial implementation of Common Lisp style formatted output.
// If the first argument is NULL, output goes to current_output_port.
extern int format  ( struct port_t *p, const char *format_string, ... );

#ifndef LIBSTD_FORMAT_SUPPORTS_BINARY
#define LIBSTD_FORMAT_SUPPORTS_BINARY 0
#endif

#ifndef LIBSTD_FORMAT_FIXED_PRECISION
#define LIBSTD_FORMAT_FIXED_PRECISION 9
#endif

#ifndef LIBSTD_FORMAT_SUPPORTS_FIXED
#define LIBSTD_FORMAT_SUPPORTS_FIXED 0
#endif

#ifndef LIBSTD_FORMAT_SUPPORTS_FLOAT
#define LIBSTD_FORMAT_SUPPORTS_FLOAT 0
#endif

/****************************************************************/
// Fixed point arithmetic support.  The default format requires 32
// bits and can represent numbers on the open set (-1.0, 1.0).
typedef int fixed_point_t;       // should be a 32 bit integer

#define NAN_FIXED_POINT ((fixed_point_t)  0x80000000)  // would be -1.0, but is reserved for an invalid value
#define MIN_FIXED_POINT ((fixed_point_t)  0x80000001)  // approx -0.9999999995
#define MAX_FIXED_POINT ((fixed_point_t)  0x7fffffff)  // approx  0.9999999995
#define EPS_FIXED_POINT ((fixed_point_t)  0x00000001)  // approx  0.0000000005

// Convert to and from the default 0.31 format fixed point, which represents
// signed numbers over the open interval (-1,1).
#define FLOAT_TO_FIXED(f) ((fixed_point_t)(2147483648.0*(f)))
#define FIXED_TO_FLOAT(f) (((float) (f)) / 2147483648.0)

// Convert to and from m.n format fixed point, where m+n == 31
#define FLOAT_TO_FIXED_N(f,n) ((fixed_point_t)((float)(1<<(n))*(f)))
#define FIXED_N_TO_FLOAT(f,n) (((float)(f)) / ((float)(1<<(n))*(f)))

// Fixed point operations for the default 0.31 format.
extern fixed_point_t fixed_mul( fixed_point_t i1, fixed_point_t i2 );
extern fixed_point_t fixed_add( fixed_point_t i1, fixed_point_t i2 );  // this isn't really necessary

// Fixed point multiplication for mantissa lengths m0 and m1, producing
// a fixed point mantissa length m2.
extern fixed_point_t fixed_mul_n( fixed_point_t i1, fixed_point_t i2, int m0, int m1, int m2 );

// Faster fixed point multiplication for mantissa lengths m0 and m1,
// producing a fixed point mantissa length m2, using only normal 32
// bit multiplication but at a substantial loss of accuracy.
extern fixed_point_t fast_fixed_mul_n( fixed_point_t i1, fixed_point_t i2, int m0, int m1, int m2 );

/****************************************************************/
// User-provided external interface functions.
extern int    putchar( int c );
extern int    getchar( void );

/****************************************************************/
// A simple memory allocator in memalloc.c
extern int   init_memalloc( void *memory, size_t pool_size );
extern void *memalloc( size_t size );
extern void  memfree( void *p );

/****************************************************************/
#endif //__LIBSTD_H_INCLUDED__

