// libc.c : simple implementations of standard libc functions
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// This file has implementations of standard libc functions chosen for
// small size and clarity over efficiency.  For example, the usual
// word optimizations are not included in memcpy, so it is relatively
// slow, although the small size of the typical memory on a LPC2xxx
// targets makes that less of an issue.

// These functions should conform to the libc API in all cases;
// libc-like functions which vary from the standard have been given
// different names and don't appear in this file.

#include <libstd.h>

/****************************************************************/
int 
atoi ( const char *nptr )
{
  int sign = 0;
  long result = 0;

  // eat initial spaces
  while ( *nptr == ' ' ) nptr++;

  // look for a sign
  if ( *nptr == '+' ) {
    sign = 1;
    nptr++;
  } else if ( *nptr == '-') {
    sign = -1;
    nptr++;
  }

  // read digits
  while ( *nptr >= '0' && *nptr <= '9' ) {
    result *= 10;
    result +=  *nptr - '0';
    nptr++;
  }

  if ( sign == -1 ) return -result;
  else return result;
}

/*****************************************************************/
int isspace(int c)
{
  return (c == ' ' || c == '\n' || c == '\r' || c == '\t');
}
int isalpha(int c)
{
  return ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'));
}
int isdigit(int c)
{
  return (c >= '0' && c <= '9');
}

int toupper(int c)
{
  if ( c >= 'a' && c <= 'z' ) return ( 'A' + c - 'a' );
  else return c;
}

/****************************************************************/
int memcmp ( const void *b1, const void *b2, size_t len )
{
  const unsigned char *p1 = (const unsigned char *) b1;
  const unsigned char *p2 = (const unsigned char *) b2;

  while ( len-- ) {
    if ( *p1 != *p2 ) {
      return *p1 - *p2;
    }
    p1++;
    p2++;
  }
  return 0;
}

/****************************************************************/
void *memcpy(void *dst, const void *src, size_t len)
{
  unsigned char *d       = (unsigned char *)       dst;
  const unsigned char *s = (const unsigned char *) src;
  while ( len-- ) *d++ = *s++;
  return dst;
}

/****************************************************************/
void *memset ( void *b, int c, size_t len )
{
  unsigned char *d = (unsigned char *) b;
  while ( len-- ) *d++ = c;
  return b;
}

/****************************************************************/
int strcmp (const char *s1, const char *s2)
{
  while ( (*s1) && (*s1 == *s2) ) {
    s1++;
    s2++;
  }
  return (unsigned char)*s1 - (unsigned char)*s2;
}

/****************************************************************/
size_t strlen(const char *s)
{
  const char *start = s;
  while (*s) s++;
  return (s - start);
}

/****************************************************************/
int strncmp (const char *s1, const char *s2, size_t len)
{
  if ( len == 0 ) return 0;

  while ( (*s1) && (*s1 == *s2) ) {
    if ( --len == 0 ) break;
    s1++;
    s2++;
  }
  return (unsigned char)*s1 - (unsigned char)*s2;
}

/****************************************************************/
// srand() and rand() were borrowed from newlib-1.14.0
unsigned long long _rand_next = 1;

void
srand( unsigned int seed )
{
  _rand_next = seed;
}

int
rand(void)
{
  /* This multiplier was obtained from Knuth, D.E., "The Art of
     Computer Programming," Vol 2, Seminumerical Algorithms, Third
     Edition, Addison-Wesley, 1998, p. 106 (line 26) & p. 108 */
  _rand_next *= 6364136223846793005LL + 1;
  return (int) ((_rand_next >> 32) & RAND_MAX);
}

/****************************************************************/
