// format.c : partial implementation of Common Lisp style formatted output (in lieu of printf)
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libstd.h>

#if LIBSTD_FORMAT_SUPPORTS_SCM
#include <scm_functions.h>
#endif

/****************************************************************/
// output a 32bit unsigned hex digit
static void print_hex_integer( struct port_t *p, int input )
{
  int i;
  char tmp[8];
  for ( i = 0; i < 8; i++ ) {
    int c = input & 0x0f;
    tmp[i] = (c > 9) ? (c + 'A' - 10) : (c + '0');
    input >>= 4;
  }
  // output the digits
  for (i = 7; i >= 0; i-- ) port_write_char ( p,  tmp[i] );
}

/****************************************************************/
// output a signed decimal number
static void print_decimal_integer( struct port_t *p, int input )
{
  int i;
  char tmp[10];  // max int is ~4E9
  unsigned value;

  // Avoid any problems with the most negative number, which has no
  // positive counterpart, by doing the conversion with unsigned
  // arithmetic.

  if ( input < 0 ) {
    port_write_char(p, '-');
    value = -input;
  } else value = input;

  // the digits are created in reverse order
  for ( i = 0; i < 10; i++ ) {
    tmp[i] = '0' + value % 10;  // generate one digit
    value /= 10;
    if ( value == 0) break;
  }
  // output the digits
  while ( i >= 0 ) port_write_char(p,  tmp[i--] );
}

/****************************************************************/
#if LIBSTD_FORMAT_SUPPORTS_FIXED

// Output a signed fixed point number.
static void print_fixed_point ( struct port_t *p, fixed_point_t input )
{
  int i;
  long long fraction; 
  int integer;

  // There is no -1.
  if ( input == NAN_FIXED_POINT ) {
    port_write_char(p, 'N'); port_write_char(p, 'a'); port_write_char(p, 'N');
    return;
  }

  // Convert to an unsigned long long value.
  if ( input < 0 ) {
    port_write_char(p, '-');
    fraction = -input;
  } else {
    fraction = input;
  }

  // Since the input number is only 31 bits once the sign has been
  // stripped off, move it up to align with the half-long-word
  // boundary.
  fraction <<= 1;

  // The number always begins "0.", since it is defined on an open set.
  port_write_char(p, '0'); port_write_char(p, '.');

  // Emit each digit of the fraction up to the compiled precision.
  for ( i = 0; i < LIBSTD_FORMAT_FIXED_PRECISION; i++ ) {
    fraction *= 10;
    integer = fraction >> 32;
    port_write_char(p,  '0' + integer);  // generate one digit
    fraction &= 0xffffffff;
  }
}
#endif // LIBSTD_FORMAT_SUPPORTS_FIXED

/****************************************************************/
#if LIBSTD_FORMAT_SUPPORTS_BINARY
// Output a 32bit unsigned binary number.  The field width specifies
// how many of the least-significant bits to include, and if 0
// defaults to 32.
static void print_binary_integer( struct port_t *p, int field_width, unsigned input )
{
  unsigned mask;

  if (field_width == 0) field_width = 32;
  mask = 1 << (field_width-1);

  while (mask) {
    port_write_char ( p, (input & mask) ? '1' : '0');
    mask >>=1;
  }
}
#endif // LIBSTD_FORMAT_SUPPORTS_BINARY

/****************************************************************/
// A simple implementation of the Common Lisp/Scheme format function.
// Returns an error code.

int format( struct port_t *p, const char *format_string, ... )
{
  const char *c;
  va_list args;
  
  if ( p == NULL ) p = current_output_port;
  
  va_start( args, format_string );

  c = format_string;
  while (*c) {

    if ( *c != '~' ) {
      port_write_char(p,  *c++ ); // default is to emit the character

    } else {// else beginning a control sequence
      unsigned field_width = 0;
      c++; // advance past the tilde

      // Parse the format arguments.  This should be generalized to handle
      // multiple arguments separated by commas.

      // For now, just parse the field width.
      while ( *c >= '0' && *c <= '9' ) {
	field_width *= 10;
	field_width += (*c - '0');
	c++;
      }

      // Interpret the format character.
      switch (*c) {
      case 0:
	// If the end of string is reached, the type specifier is
	// missing, so just emit the tilde, but don't advance the
	// pointer, so this will break out of the while loop.
	port_write_char( p, '~');
	break;

      case '~': 
	port_write_char ( p, '~');
	c++;
	break;

      case '%': 
	port_write_char ( p, '\n');
	c++;
	break;

      case 'd': 
	print_decimal_integer( p, va_arg( args,  int ));
	c++;
	break;

      case 'a': 
	{
	  char *str = va_arg( args, char *);
	  while (*str) port_write_char ( p, *str++);
	}
	c++;
	break;

      case 'c': 
	port_write_char( p, va_arg( args, int ));
	c++;
	break;

      case 'x': 
	print_hex_integer( p, va_arg( args, int));
	c++;
	break;

      case 'p':  // this isn't standard, but is a convenient nod to printf
	print_hex_integer( p, (unsigned) (va_arg( args, void *)));
	c++;
	break;

#if LIBSTD_FORMAT_SUPPORTS_BINARY
      case 'b':
	print_binary_integer( p, field_width, (unsigned) (va_arg( args, void *)));
	c++;
	break;
#endif

#if LIBSTD_FORMAT_SUPPORTS_FIXED

#if !LIBSTD_FORMAT_SUPPORTS_FLOAT
	// this is deprecated; until floating point support is really added this will
	// continue to work for fixed point, but all instances of ~f for fixed point
	// should be changed to ~m.
      case 'f':
#endif	
	// non-standard: fixed point output. 'm' stands for
	// mantissa. ('f' was already taken).  eventually the
	// arguments will allow specifying the fixed point format, but
	// for now the default 0.31 signed fixed point is assumed.
      case 'm':
	print_fixed_point( p, va_arg( args, fixed_point_t ) );
	c++;
	break;
#endif

#if LIBSTD_FORMAT_SUPPORTS_SCM
      case 's': 
	// SCM is by default an unsigned short, which doesn't work
	// well in va_arg, so this hardcodes an int instead.  This
	// should be fine since the SCM type is fundamentally an
	// index, not a pointer, so it should never be larger than an
	// int.
	scm_write_port( (SCM) va_arg( args, int ), p );
	c++;
	break;
#endif

	// The default is to emit the control sequence itself.
      default:
	port_write_char(p,  '~' );
	port_write_char(p,  *c );
	c++;
	break;
      }
    }
  }

  va_end( args );

  return 0;
}

