// watchdog_interrupt.c : interrupt handler for the hardware watchdog to be used while debugging
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// Watchdog interrupt handler.  This is primarily useful while
// debugging the watchdog; ordinarily it generates a reset.

#include <libLPC2xxx.h>
#include <interrupt_handlers.h>

// Permissible values for the "interrupt" attribute parameter are:
// IRQ, FIQ, SWI, ABORT and UNDEF. This behavior changed with GCC 3.4.4.

// Watchdog timer ISR.  Useful when debugging and the watchdog
// generates an interrupt instead of reset

volatile unsigned int watchdog_interrupts = 0;

#define TRACELENGTH 32
volatile unsigned int watchdog_stack_trace_length = 0;
volatile unsigned int watchdog_stack_trace[TRACELENGTH];
volatile unsigned int watchdog_stack_base;
volatile unsigned int watchdog_interrupt_return_address;

extern char _IRQ_stack_top;  // defined by linker

void __attribute__((interrupt ("IRQ"))) watchdog_interrupt_handler(void)
{
  unsigned int count;
  unsigned int *stackptr;
  register unsigned int link_reg asm ("r8");

  // signal a problem
  watchdog_interrupts++;

  // save the return address, since it is not guaranteed to be on the stack
  asm("mov r8, lr"); // this could be improved
  watchdog_interrupt_return_address = link_reg;
  
  // save a stack trace 
  stackptr = &count;
  watchdog_stack_base = (unsigned int) stackptr;
  watchdog_stack_trace_length = 0;

  for ( count = 0; count < TRACELENGTH; count++ ) {
    watchdog_stack_trace[ count ] = *stackptr++;
    watchdog_stack_trace_length++;
    if ( (void *) stackptr > (void *) &_IRQ_stack_top + 8) break;
  }

  // This must now disable itself or it will recur.
  VIC.IntEnClear = VIC_WDT_MASK;

  // A write must be performed to the VIC Vector Address Register to
  // update the VIC priority hardware.
  VIC.VectAddr = 0;

}
/****************************************************************/
