// libLPC2xxx.h : configuration file for the libLPC2xxx microcontroller support library
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// This file specifies the default configuration of the hardware and
// software environment for compiling the body of code for supporting
// my LPC2xxx projects.  This allows compiling the library code for
// different targets and customizations.  If any settings get promoted
// to the required user configuration file config.h, then the setting
// here should be bracketed in ifndef/endif.

// Some of the more rarely used options could be migrated to
// individual header files; the most common are here for convenience.

// It is best if the options take a definite value and to use the #if
// directive rather than #ifdef; that way, all defaults are explicitly
// specified and the compiler can help catch typos.

// The policy of the library is to default to a relatively minimal
// configuration, requiring each application to specifically enable
// resources it will need.  This should help reduce contention between
// different applications.  In addition, the individual makefile for
// the target will only try to compile driver code for the hardware it
// implements, which should cut down on problems with undefined
// symbols.

// It is preferable to express the configuration in the choice
// of code instead of options wherever possible.  The linker omits
// unused code, so just avoiding unnecessary references to functions
// or objects will minimize the included code.

// It may also be desirable to follow a naming convention where
// user-configuration symbols all begin CONFIG_.  And computed values
// might begin LIBLPC_, although that is not currently followed.

#ifndef __LIBLPC2XXX_H_INCLUDED__
#define __LIBLPC2XXX_H_INCLUDED__

// Include the required user-defined source-code configuration file.
// This should include one of the part variant files (e.g. LPC2138.h)
// to define the exact hardware available in this configuration.

#include <config.h> 

/******************* CPU clock PLL configuration *****************************/
#ifndef FOSC
#define FOSC              (14745600)       // crystal oscillator frequency
#endif

#ifndef CLOCKMULT
#define CLOCKMULT         (4)              // PLL multiplier rate; CCLK = FOSC * CLOCKMULT
#endif

#define CCLK              (CLOCKMULT*FOSC) // CPU clock
#define PCLK              CCLK             // VPB clock

/*******************  LED indicator configuration **************************/
#ifndef CONFIG_CPU_LED_MASK	// typical single LED indicator 
#define CONFIG_CPU_LED_MASK  0
#endif

#ifndef CONFIG_CPU_LED_ACTIVE_LOW    // define if the CPU LED is active-low instead of active-high
#define CONFIG_CPU_LED_ACTIVE_LOW 1  // default is active-low
#endif

#ifndef CONFIG_LED_MASK		// general purpose indicator array
#define CONFIG_LED_MASK 0
#endif

#ifndef CONFIG_LED_ACTIVE_LOW_MASK    // define which lamps are active-low instead of active-high
#define CONFIG_LED_ACTIVE_LOW_MASK 0  // default is active-high
#endif

/******************  UART configuration ********************************/
#ifndef CONFIG_CONSOLE_BAUD
#define CONFIG_CONSOLE_BAUD (57600)
#endif

// By default, UART0 is assumed to be the console.
#ifndef UART0_BAUD
#define UART0_BAUD        (CONFIG_CONSOLE_BAUD)
#endif

// It is safe to always compute valid divisors for UART0, even if not used.
#define UART0_BAUD_CLOCK  (16 * UART0_BAUD)
#define UART0_DIVISOR     (PCLK/UART0_BAUD_CLOCK)

#ifndef CONFIG_UART0_DEFAULT_INIT
#define CONFIG_UART0_DEFAULT_INIT 0  // if true, initialize UART0 in chipinit.c
#endif

// Similar for UART1
#ifdef CONFIG_UART1_BAUD
#define UART1_BAUD        (CONFIG_UART1_BAUD)
#endif

#define UART1_BAUD_CLOCK  (16 * UART1_BAUD)
#define UART1_DIVISOR     (PCLK/UART1_BAUD_CLOCK)

// Define the default buffer sizes for the buffered serial port support.
#ifndef CONFIG_UART0_OUTPUT_BUFLEN
#define CONFIG_UART0_OUTPUT_BUFLEN 100
#endif

#ifndef CONFIG_UART0_INPUT_BUFLEN
#define CONFIG_UART0_INPUT_BUFLEN 100
#endif

#ifndef CONFIG_UART1_OUTPUT_BUFLEN
#define CONFIG_UART1_OUTPUT_BUFLEN 100
#endif

#ifndef CONFIG_UART1_INPUT_BUFLEN
#define CONFIG_UART1_INPUT_BUFLEN 100
#endif

// UART_port.c can support either or both of UART0 and UART1.
#ifndef CONFIG_UART_PORT_SUPPORTS_UART0
#define CONFIG_UART_PORT_SUPPORTS_UART0 1
#endif

#ifndef CONFIG_UART_PORT_SUPPORTS_UART1
#define CONFIG_UART_PORT_SUPPORTS_UART1 0
#endif

/****************************************************************/
// The console provides a simple API like a trivial libc stdio. This
// configures which functions it uses to provide I/O.
#ifndef CONSOLE_USES_UNBUFFERED_SERIAL
#define CONSOLE_USES_UNBUFFERED_SERIAL 0
#endif

#ifndef CONSOLE_USES_SERIAL_FIFOS
#define CONSOLE_USES_SERIAL_FIFOS 0
#endif

// True to enable simple "cooked" console input.
// #ifndef CONSOLE_INPUT_LINE_BUFFERED
// #define CONSOLE_INPUT_LINE_BUFFERED 1
// #endif

// #ifndef CONSOLE_INPUT_LINE_BUFFER_SIZE 
// #define CONSOLE_INPUT_LINE_BUFFER_SIZE 120
// #endif

/****************************************************************/
#ifndef CONFIG_WATCHDOG_INTERVAL
#define CONFIG_WATCHDOG_INTERVAL 0    // watchdog timeout period in seconds, zero to disable
#endif

// True if the watchdog timer generates an interrupt rather than a reset.
#define CONFIG_WATCHDOG_GENERATES_INTERRUPT 0

/****************************************************************/
// The timer interrupt can be used to poll lots of devices in addition
// to keeping track of elapsed time.

#ifndef CONFIG_TIMER_INTERRUPT_RATE
#define CONFIG_TIMER_INTERRUPT_RATE 0   // zero indicates off, else interrupt rate in Hz
#endif

// If the timer interrupt is enabled, then the default is to standard handler.
#if CONFIG_TIMER_INTERRUPT_RATE
#ifndef CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT
#define CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT 1
#endif
#endif

// The timerclock updates once per timer interrupt by default.
#define TIMERCLOCK_RATE  (CONFIG_TIMER_INTERRUPT_RATE)
#define TIMER_INTERRUPT_DIVISOR (PCLK / CONFIG_TIMER_INTERRUPT_RATE)

// These macros work up to intervals a little longer than a day.  They are only good
// to use for constants, to avoid using floating point in the code.
#define SECONDS_TO_TICKS(sec)  ((int)((float)(sec)*TIMERCLOCK_RATE))
#define MSEC_TO_TICKS(msec)    ((int)(((float)(msec)*TIMERCLOCK_RATE*0.001)))

// More approximate, but integer-only.
#define INT_MSEC_TO_TICKS(msec)  ((int)((msec)*(TIMERCLOCK_RATE/1000)))

/****************************************************************/
// Many polling operations can be enabled within the timer interrupt.
// The advantage of polling is complete control over the relative
// timing of I/O, and no worries about the jitter created by competing
// interrupts.

// True to enable basic load reporting from within the timer
// interrupt, useful for checking whether it is taking too long.
#ifndef CONFIG_TIMER_INTERRUPT_REPORTS_LOAD
#define CONFIG_TIMER_INTERRUPT_REPORTS_LOAD 0
#endif

// True if timer interrupt should managed buffered serial I/O.
#ifndef TIMER_INTERRUPT_UPDATES_POLLED_SERIAL_FIFOS
#define TIMER_INTERRUPT_UPDATES_POLLED_SERIAL_FIFOS 0
#endif

// True if the timer interrupt sends samples to the external audio output D/A
#ifndef TIMER_INTERRUPT_UPDATES_AUDIO_OUTPUT
#define TIMER_INTERRUPT_UPDATES_AUDIO_OUTPUT 0
#endif

#ifndef AUDIO_BUFLEN
#define AUDIO_BUFLEN 8192  // must be a power of two
#endif

// AD5220 digital potentiometer for volume control, on GPIO1 outputs
#ifndef TIMER_INTERRUPT_POLLS_VOLUME_CONTROL
#define TIMER_INTERRUPT_POLLS_VOLUME_CONTROL 0
#endif

// QT117 capacitance-to-digital converter
#ifndef TIMER_INTERRUPT_POLLS_QT117
#define TIMER_INTERRUPT_POLLS_QT117 0
#endif

// built-in A/D converter
#ifndef TIMER_INTERRUPT_POLLS_ANALOG_INPUTS
#define TIMER_INTERRUPT_POLLS_ANALOG_INPUTS 0
#endif

/****************************************************************/
#define ADC_CLOCK         (4500000)        // A/D converter clock, max 4.5 MHz
#define ADC_CLOCK_DIVISOR (PCLK/ADC_CLOCK) // actual division is (CLKDIV+1), so this rounds down

/****************************************************************/
// The SPI0 clock rate is PCLK/SPI0_DIVISOR.
#define SPI0_DIVISOR      (8) // must be even, and between 8 and 254 inclusive

// The SPI1 prescaler value must be even, and between 2 and 254.  The
// SPI1 bit frequency is PCLK / ( SSPCPSR * (SCR + 1)).  Since SCR =
// 0, the slow rate yields about 400kHz for the rate during card
// setup.  The fast rate is about 3.9 mHz for data transfer.  The fast
// rate is limited by the maximum timer interrupt duration allowing
// the SPI FIFO to go dry.
#define SPI1_SLOW_PRESCALE  (148)     // rate for MMC card initialization
#define SPI1_FAST_PRESCALE   (15)     // rate for data

/****************************************************************/

// Not actually defined by this; defined by code in analog_input.c
#ifndef MICROPHONE_SAMPLING_RATE
#define MICROPHONE_SAMPLING_RATE (TIMERCLOCK_RATE/8)
#endif

#ifndef CONFIG_MICROPHONE_MONITORS_AUDIO_FILE_PLAYER
#define CONFIG_MICROPHONE_MONITORS_AUDIO_FILE_PLAYER 0
#endif

/****************************************************************/
// The SD/MMC card is interface is configured read-only by default for safety.
#ifndef CONFIG_INCLUDE_MMC_WRITE_BLOCK
#define CONFIG_INCLUDE_MMC_WRITE_BLOCK 0
#endif

#endif //  __LIBLPC2XXX_H_INCLUDED__
