// fifo.c : non-blocking FIFO implementation suitable for IPC between interrupt and normal contexts
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libstd.h>
#include <libLPC2xxx.h>
#include <fifo.h>
#include <errcodes.h>

// Inline primitives to manipulate and examine the ring buffer
// pointers.  In general these are not thread safe by themselves, 
// they should only used in critical regions.

static inline void buffer_data_added( fifo *p, unsigned bytes )
{
  unsigned new_space = p->space + bytes;

  // Wrap the index around without using the modulo operator.
  while (new_space >= p->buffer_size) new_space -= p->buffer_size;

  p->space = new_space;
}
static inline void buffer_data_removed( fifo *p, unsigned bytes )
{
  unsigned new_data = p->data + bytes;
  while (new_data >= p->buffer_size) new_data -= p->buffer_size;
  p->data = new_data;
}
static inline unsigned buffer_contiguous_space( fifo *p )
{
  if ( p->space >= p->data ) return p->buffer_size - p->space;
  else return (p->data - p->space - 1);
}
static inline unsigned buffer_contiguous_data( fifo *p )
{
  if ( p->data > p->space ) return p->buffer_size - p->data;
  else return (p->space - p->data);
}

/****************************************************************/
unsigned int
fifo_space_available( fifo *p )
{
  int diff = p->buffer_size + p->space - p->data;
  while (diff >= p->buffer_size) diff -= p->buffer_size;
  return p->buffer_size - 1 - diff;
}

unsigned int
fifo_data_available( fifo *p )
{
  int diff = p->buffer_size + p->space - p->data;
  while (diff >= p->buffer_size) diff -= p->buffer_size;
  return diff;
}

/****************************************************************/
// Private functions for copying data taking account of the ring wraparound.
static void 
memcpy_to_buffer( fifo *p, void *data, unsigned length )
{
// Decide how to copy the data.
  unsigned contiguous_space = buffer_contiguous_space( p );

  if ( length <= contiguous_space ) {
    // simple case, contiguous space available
    memcpy( p->buffer + p->space, data, length );

  } else {
    // else the block wraps around the end
    memcpy( p->buffer + p->space, data, contiguous_space );
    memcpy( p->buffer, data + contiguous_space, length - contiguous_space );
  }
}
static void 
memcpy_from_buffer( fifo *p, void *data, unsigned length )
{
  // Decide how to copy the data.
  unsigned contiguous_data = buffer_contiguous_data( p );

  if ( length <= contiguous_data ) {
    // simple case, contiguous data available
    memcpy( data, p->buffer + p->data, length );

  } else {
    // else the block wraps around the end
    memcpy( data, p->buffer + p->data, contiguous_data );
    memcpy( data + contiguous_data, p->buffer, length - contiguous_data );
  }
}

/****************************************************************/
//! Initialize a fifo structure.
fifo *
init_fifo( fifo *p, void *buffer, int buffer_size  )
{
  if ( p == NULL) return NULL;

  p->data = 0;
  p->space = 0;
  p->buffer_size = buffer_size;
  p->buffer = buffer;
  return p;
}
/****************************************************************/
//! Writes the specified number of bytes into the FIFO from the
//! supplied buffer.  Returns zero on success.  On failure returns a
//! negative error code.  In particular, returns -ERRAGAIN if sufficient
//! space isn't available.

int
fifo_try_put( fifo *p, void *data, unsigned int length )
{
  if ( p == NULL ) {
    return -ERRINVAL;
  }
  if ( fifo_space_available (p) < length ) {
    return -ERRAGAIN;
  }
  memcpy_to_buffer( p, data, length );
  buffer_data_added( p, length );
  return 0;
}
/****************************************************************/
//! Read the specified number of bytes from the FIFO into the supplied
//! buffer.  Returns zero on success.  On failure returns a negative
//! error code on failure.  In particular, returns -ERRAGAIN if
//! sufficient data isn't available.

int
fifo_try_get( fifo *p, void *data, unsigned int length )
{
  if ( p == NULL ) {
    return -ERRINVAL;
  }
  if ( fifo_data_available (p) < length ) {
    return -ERRAGAIN;
  }
  memcpy_from_buffer( p, data, length );
  buffer_data_removed( p , length );
  return 0;
}

/****************************************************************/
//! Read up to the specified number of bytes from the FIFO into the
//! supplied buffer. Returns the number of bytes read on success.  On
//! failure returns a negative error code.

int
fifo_try_read( fifo *p, void *data, unsigned int length )
{
  unsigned data_available;

  if ( p == NULL ) {
    return -ERRINVAL;
  }
  if ( ( data_available = fifo_data_available (p)) == 0 ) {
    return -ERRAGAIN;
  }

  if ( data_available < length ) length = data_available;

  memcpy_from_buffer( p, data, length );
  buffer_data_removed( p , length );
  return length;
}

/****************************************************************/
//! Read the specified number of bytes from the FIFO into the supplied
//! buffer without actually removing them from the queue.  Returns
//! zero on success.  On failure returns a negative
//! error code.  In particular, returns -ERRAGAIN if the data
//! isn't available.  Note that this release the lock again, so 
//! if there multiple readers this data might not remain available.

int
fifo_peek( fifo *p, void *data, unsigned int length )
{
  if ( p == NULL ) {
    return -ERRINVAL;
  }
  if ( fifo_data_available(p) < length ) {
    return -ERRAGAIN;
  }
  memcpy_from_buffer( p, data, length );
  return 0;
}

/****************************************************************/
//! Put just one character into the fifo.  Returns 0 on success, or -ERRAGAIN if full.
int 
fifo_putchar( fifo *p, unsigned char byte)
{
  if ( p == NULL )  return -ERRINVAL;

  if ( fifo_space_available (p) == 0 )  return -ERRAGAIN;

  ((unsigned char *) p->buffer) [p->space] = byte;
  buffer_data_added( p, 1 );
  return 0;
}

//! Try getting one character from the fifo.  Returns a character, or EOF if nothing is available.
int 
fifo_getchar( fifo *p )
{
  unsigned char data;

  if ( p == NULL || (p->data == p->space) ) return EOF;

  data = ((unsigned char *) p->buffer) [ p->data ];
  buffer_data_removed( p, 1 );

  return data;
}
