// UART_port.c : character device interface to interrupt driven serial ports
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// An abstract character device interface to the UART0 and UART1
// interrupt handler FIFOS. An unbuffered polled version is in
// UART_raw_port.c.

// A simpler, more-libc-like interface is in console.c. 

#include <libstd.h>
#include <port.h>
#include <libLPC2xxx.h>
#include <UART_port.h>
#include <errcodes.h>

/****************************************************************/
#if CONFIG_UART_PORT_SUPPORTS_UART0

#include <UART0_interrupt_handler.h>

// A trivial bit of glue to match the port abstraction to the
// underlying interrupt driven buffers.

static int UART0_write_char( struct port_t *port_ptr, int c)
{
  return UART0_fifo_write_char( c );
}
static int UART0_char_ready( struct port_t *port_ptr )
{
  return UART0_fifo_read_ready();
}
static int UART0_read_char (struct port_t *port_ptr)
{
  return UART0_fifo_read_char();
}
const static struct port_class_t UART0_class = {
  UART0_read_char,    // int  (*read_char)(struct port_t *p);
  NULL,               // int  (*peek_char)(struct port_t *p);
  UART0_char_ready,   // int  (*char_ready)(struct port_t *p);
  UART0_write_char,   // int  (*write_char)(struct port_t *p, int c);
  NULL, 	      // size_t  (*read)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	      // size_t (*write)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	      // off_t   (*seek)(struct port_t *p, off_t offset, int whence);
  NULL  	      // int    (*close)(struct port_t *p);
};
#endif // CONFIG_UART_PORT_SUPPORTS_UART0

/****************************************************************/
#if CONFIG_UART_PORT_SUPPORTS_UART1

#include <UART1_interrupt_handler.h>

// A trivial bit of glue to match the port abstraction to the
// underlying interrupt driven buffers.

static int UART1_write_char( struct port_t *port_ptr, int c)
{
  return UART1_fifo_write_char( c );
}
static int UART1_char_ready( struct port_t *port_ptr )
{
  return UART1_fifo_read_ready();
}
static int UART1_read_char (struct port_t *port_ptr)
{
  return UART1_fifo_read_char();
}
const static struct port_class_t UART1_class = {
  UART1_read_char,    // int  (*read_char)(struct port_t *p);
  NULL,               // int  (*peek_char)(struct port_t *p);
  UART1_char_ready,   // int  (*char_ready)(struct port_t *p);
  UART1_write_char,   // int  (*write_char)(struct port_t *p, int c);
  NULL, 	      // size_t  (*read)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	      // size_t (*write)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	      // off_t   (*seek)(struct port_t *p, off_t offset, int whence);
  NULL  	      // int    (*close)(struct port_t *p);
};
#endif // CONFIG_UART_PORT_SUPPORTS_UART1

/****************************************************************/
int 
UART_port_init( UART_port *p, struct UART_registers_t *uart, unsigned divisor, unsigned mode )
{
#if CONFIG_UART_PORT_SUPPORTS_UART0
  if ( uart == &UART0 ) generic_port_init( &p->interface, &UART0_class, mode );
#endif

#if CONFIG_UART_PORT_SUPPORTS_UART1
  if ( uart == &UART1 ) generic_port_init( &p->interface, &UART1_class, mode );
#endif

  // There is no test of whether the uart value is invalid.

  // p->uart = uart;

  // Initialize the hardware port.
  // initialize UART
  uart->FCR = 0x7;   // Enable and reset fifos (0x7); keep Rx Trigger Level at 0, to interrupt 
                     // after one character is received
  uart->LCR = 0x83;  // 8 bits; enable divisor latches
  uart->DLL = (divisor & 0xff);  // LSB of divider for baud clock
  uart->DLM = (divisor >> 8);    // MSB of divider for baud clock
  uart->LCR = 0x3;   // 8 bits; 1 stop bit; no parity; odd parity; disable break; disable divisor latches

  // enable UART pins after the UART is configured and install the interrupt handler
#if CONFIG_UART_PORT_SUPPORTS_UART0
  if ( uart == &UART0 ) {
    if ( mode & PORT_FLAGS_WRITE )  PCB.PINSEL0 |= PINSEL0_MASK( 0, 1 );  // P0.0 is UART0 TxD 
    if ( mode & PORT_FLAGS_READ )   PCB.PINSEL0 |= PINSEL0_MASK( 1, 1 );  // P0.1 is UART0 RxD.
    UART0_install_interrupt();
  }
#endif

#if CONFIG_UART_PORT_SUPPORTS_UART1
  if ( uart == &UART1 ) {
    if ( mode & PORT_FLAGS_WRITE )  PCB.PINSEL0 |= PINSEL0_MASK( 8, 1 );  // P0.8 is UART1 TxD 
    if ( mode & PORT_FLAGS_READ )   PCB.PINSEL0 |= PINSEL0_MASK( 9, 1 );  // P0.9 is UART1 RxD.
    UART1_install_interrupt();
  }
#endif

  // enable either or both the read and write interrupts
  uart->IER = ((mode & PORT_FLAGS_READ) ? 0x01 : 0) | ((mode & PORT_FLAGS_WRITE) ? 0x02 : 0);
  
  return ERRNOERROR;
}
