// UART0_interrupt_handler.c : interrupt handler for the first serial port
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// This is hardcoded for a particular UART for efficiency, so that all
// hardware references can be compiled as constants.

#include <libLPC2xxx.h>
#include <UART0_interrupt_handler.h>
#include <fifo.h>
#include <errcodes.h>

#define UART_INTERRUPT_NUMBER 6
#define UART_INTERRUPT_MASK 0x40  // the UART0 interrupt bit

/****************************************************************/
// Define the UART I/O buffers.

static unsigned char UART0_output_buffer[ CONFIG_UART0_OUTPUT_BUFLEN ];
static fifo UART0_output_fifo = { &UART0_output_buffer, CONFIG_UART0_OUTPUT_BUFLEN, 0, 0 };

static unsigned char UART0_input_buffer[ CONFIG_UART0_INPUT_BUFLEN ];
static fifo UART0_input_fifo = { &UART0_input_buffer, CONFIG_UART0_INPUT_BUFLEN, 0, 0 };

/****************************************************************/
// A simple user-side API to the interrupt driven character fifos.

int UART0_fifo_read_ready(void)
{
  return (fifo_data_available( &UART0_input_fifo ) > 0);
}

int UART0_fifo_write_ready(void)
{
  return (fifo_space_available( &UART0_output_fifo ) > 0);
}

int UART0_fifo_write_char(unsigned char c)
{
  if (!UART0_fifo_write_ready()) return -ERRNOSPACE;

  fifo_putchar( &UART0_output_fifo, c );

  // If the transmitter is empty, generate a soft UART0 interrupt to start
  // sending data.
  if ( UART0.LSR & LSR_THRE_MASK ) VIC.SoftInt = UART_INTERRUPT_MASK;  // the UART interrupt bit
  return ERRNOERROR;
}

int UART0_fifo_read_char(void)
{
  return fifo_getchar( &UART0_input_fifo );
}

/****************************************************************/
// UART0 ISR

static void __attribute__((interrupt ("IRQ"))) UART0_interrupt_handler(void)
{
  
  unsigned char IIR_value;
  unsigned char LSR_value;
  unsigned char data;

  // Read the IIR, which will also clear the interrupt.
  IIR_value = UART0.IIR;

  // Clear any possible soft interrupt.
  VIC.SoftIntClear = UART_INTERRUPT_MASK;  // the UART interrupt bit

  // Check the combination of the interrupt pending flag (bit 0, active low) and the interrupt source.
  switch (IIR_value & 0x0f) {

  case 6: // Rx Line Status / Error.  OE or PE or FE or BI is set.
    LSR_value = UART0.LSR;  // this will reset the interrupt
    break;

  case 4:  // Rx Data Available
  case 12: // Character Timeout Indication.  This shouldn't happen if the FIFO trigger level is set at 1.
    {
      fifo *input = &UART0_input_fifo;  // this is the software fifo buffer for storing input

      // This will try putting new data into the buffer, and ignore any overrun.
      do {
	unsigned new_space_index = input->space + 1;

	data = UART0.RBR;  // read a new datum; this will reset the interrupt

	if (new_space_index >= input->buffer_size) new_space_index = 0;

	// if the buffer isn't full, store the character
	if ( new_space_index != input->data ) {
	  ((unsigned char *) input->buffer) [input->space] = data;
	  input->space = new_space_index;
	}
      } while (UART0.LSR & LSR_RDR_MASK ); // loop as long as data is available in the hardware input FIFO
    }
    break;

  default:
    // This will be reached in these cases:
    //  no UART interrupt is pending, which is most likely a soft interrupt to trigger sending,
    //     but could be a spurious interrupt;
    //  one of the "reserved" interrupt types is reported.

    // Either way, it's always appropropriate to fall through and check the transmitter.

  case 2: // THRE, transmitter holding register empty. This interrupt is reset by a IIR read or THR write.

    // check transmitter empty flag; loop as long as space is available in the hardware output FIFO and data to be sent
    while ( UART0.LSR & LSR_THRE_MASK ) {
      fifo *output = &UART0_output_fifo;  // this is the software fifo buffer for storing output

      // if the buffer isn't empty, send a character
      if ( output->space != output->data ) {

	// write a character to the output
	UART0.THR = ((unsigned char *) output->buffer) [output->data]; 
	if (++output->data >= output->buffer_size) output->data = 0;

      } else break; // leave the FIFO-filling loop if no more data to send
    }
    break;
  }

  VIC.VectAddr = 0; // clear VIC by writing to it
}

/****************************************************************/
void UART0_install_interrupt(void)
{
  // Set interrupt service routine addresses for each vectored slot.
  VIC.VectAddr4   = (unsigned long) UART0_interrupt_handler;   // ISR address for slot 4

  // Initialize slot in the vectored interrupt response table. UART0 is interrupt channel 6.
  VIC.VectCntl4 = 0x20 | UART_INTERRUPT_NUMBER; 

  // Enable UART interrupt in VIC.  Each 1 bit enables the corresponding interrupt, 0 bits have no effect.
  VIC.IntEnable = UART_INTERRUPT_MASK; 
}

/****************************************************************/
