// LPC2xxx.h : register definitions for hardware modules common to multiple LPC2xxx variants
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// Hardware module definitions shared across the line of Philips
// LPC2xxx microcontrollers with ARM7TDMI core.

// All unspecified bits should be written as zeros.

#ifndef __LPC2xxx_H_INCLUDED__
#define __LPC2xxx_H_INCLUDED__

#include <integers.h>   //  define the specified-width integer types

/****************************************************************/
// System Control Block

struct SCB_registers_t {
  VUINT32 MEMMAP;                 // 0xE01FC040  Memory mapping control.
  VUINT32 pad0[ 15 ];

  VUINT32 PLLCON;              	  // 0xE01FC080  PLL control register.
#define PLLCON_PLLE_MASK  1    	  //   PLL Enable
#define PLLCON_PLLC_MASK  2    	  //   PLL Connect
      
  VUINT32 PLLCFG;              	  // 0xE01FC084  PLL configuration register.
#define PLLCFG_MSEL_MASK 0x1f  	  //   PLL Multiplier value.
#define PLLCFG_PSEL_MASK 0x60  	  //   PLL Divider value.

  VUINT32 PLLSTAT;                // 0xE01FC088  PLL status register, read-only.
#define PLLSTAT_MSEL_MASK  0x01f  //   read-back for the PLL Multiplier value
#define PLLSTAT_PSEL_MASK  0x060  //   read-back for the PLL Divider value
#define PLLSTAT_PLLE_MASK  0x100  //   read-back for the PLL Enable bit 
#define PLLSTAT_PLLC_MASK  0x200  //   read-back for the PLL Connect bit 
#define PLLSTAT_PLOCK_MASK 0x400  //   reflects the PLL Lock status.

  VUINT8 PLLFEED;                 // 0xE01FC08C  PLL feed register.   
  VUINT8 unused0[3];
  VUINT32 pad1[ 12 ];
  VUINT32 PCON;                   // 0xE01FC0C0  Power control register.
  VUINT32 PCONP;                  // 0xE01FC0C4  Power control for peripherals.
  VUINT32 pad2[ 14 ];
  VUINT32 VPBDIV;                 // 0xE01FC100  VPB divider control.
  VUINT32 pad3[ 15 ];
  VUINT32 EXTINT;                 // 0xE01FC140  External interrupt flag register.       
  VUINT32 EXTWAKE;                // 0xE01FC144  External interrupt wakeup register.

#if LPC_PART==2148
  // These would need to be worked into the structure at multiple points and padding added.
  // PLL1CON                      // 0xE01FC0A0  PLL1 (USB) Control Register
  // PLL1CFG                      // 0xE01FC0A4  PLL1 (USB) Configuration Register
  // PLL1STAT                     // 0xE01FC0A8  PLL1 (USB) Status Register
  // PLL1FEED                     // 0xE01FC0AC  PLL1 (USB) Feed Register
  // EXTMODE;                     // 0xE01FC148  External interrupt mode register.
  // EXTPOLAR;                    // 0xE01FC14C  External interrupt polarity register.
  // RSID                         // 0xE01FC180  Reset Source Identification Register
  // CSPR                         // 0xE01FC184  Code Security Protection Register
  // SCS                          // 0xE01FC1A0  System Controls and Status
#endif
};
/*****************************************************************/
// Memory Accelerator Module (MAM)
struct MAM_registers_t {
  VUINT8 CR;              // 0xE01FC000  MAM Control register
  VUINT8 unused0[3];
  VUINT8 TIM;             // 0xE01FC004  MAM Timing register
};

/****************************************************************/
// Vectored Interrupt Controller

// All registers are word registers, and byte and short reads and
// writes are not supported.

struct VIC_registers_t {

  VUINT32 IRQStatus;      // 0xFFFFF000  Each bit indicates an enabled and asserted IRQ interrupt
  VUINT32 FIQStatus;      // 0xFFFFF004  Each bit indicates an enabled and asserted FIQ interrupt
  VUINT32 RawIntr;        // 0xFFFFF008  Raw Interrupt status, each bit indicates an interrupt request state.
  VUINT32 IntSelect;      // 0xFFFFF00C  Each bit selects between assignment to FIQ (1) or IRQ (0).
  VUINT32 IntEnable;      // 0xFFFFF010  Interrupt Enable; on write each 1 enables an interrupt request
  VUINT32 IntEnClear;     // 0xFFFFF014  Interrupt Enable Clear; on write each 1 disables an interrupt request.
  VUINT32 SoftInt;        // 0xFFFFF018  Software Interrupt; each bit forces a particular interrupt request.
  VUINT32 SoftIntClear;   // 0xFFFFF01C  Software Interrupt Clear; each bit clears a bit in SoftInt
  VUINT32 Protection;     // 0xFFFFF020  

  VUINT32 pad0[ 3 ];

  VUINT32 VectAddr;       // 0xFFFFF030  
  VUINT32 DefVectAddr;    // 0xFFFFF034  
  
  VUINT32 pad1[ 50 ];

  VUINT32 VectAddr0;      // 0xFFFFF100    address of ISR for interrupt slot 0, the highest priority slot
  VUINT32 VectAddr1;      // 0xFFFFF104  
  VUINT32 VectAddr2;      // 0xFFFFF108  
  VUINT32 VectAddr3;      // 0xFFFFF10C  
  VUINT32 VectAddr4;      // 0xFFFFF110  
  VUINT32 VectAddr5;      // 0xFFFFF114  
  VUINT32 VectAddr6;      // 0xFFFFF118  
  VUINT32 VectAddr7;      // 0xFFFFF11C  
  VUINT32 VectAddr8;      // 0xFFFFF120  
  VUINT32 VectAddr9;      // 0xFFFFF124  
  VUINT32 VectAddr10;     // 0xFFFFF128  
  VUINT32 VectAddr11;     // 0xFFFFF12C  
  VUINT32 VectAddr12;     // 0xFFFFF130  
  VUINT32 VectAddr13;     // 0xFFFFF134  
  VUINT32 VectAddr14;     // 0xFFFFF138  
  VUINT32 VectAddr15;     // 0xFFFFF13C  

  VUINT32 pad2[ 48 ];

  VUINT32 VectCntl0;      // 0xFFFFF200  
  VUINT32 VectCntl1;      // 0xFFFFF204  
  VUINT32 VectCntl2;      // 0xFFFFF208  
  VUINT32 VectCntl3;      // 0xFFFFF20C  
  VUINT32 VectCntl4;      // 0xFFFFF210  
  VUINT32 VectCntl5;      // 0xFFFFF214  
  VUINT32 VectCntl6;      // 0xFFFFF218  
  VUINT32 VectCntl7;      // 0xFFFFF21C  
  VUINT32 VectCntl8;      // 0xFFFFF220  
  VUINT32 VectCntl9;      // 0xFFFFF224  
  VUINT32 VectCntl10;     // 0xFFFFF228  
  VUINT32 VectCntl11;     // 0xFFFFF22C  
  VUINT32 VectCntl12;     // 0xFFFFF230  
  VUINT32 VectCntl13;     // 0xFFFFF234  
  VUINT32 VectCntl14;     // 0xFFFFF238  
  VUINT32 VectCntl15;     // 0xFFFFF23C  

#define VIC_VECTCNTL_ENABLED_MASK (0x20)

  VUINT32 pad3[ 48 ];

  VUINT32 ITCR;           // 0xFFFFF300  
  VUINT32 ITIP1;          // 0xFFFFF304  
  VUINT32 ITIP2;          // 0xFFFFF308  
  VUINT32 ITOP1;          // 0xFFFFF30C  
  VUINT32 ITOP2;          // 0xFFFFF310  

  VUINT32 pad[ 819 ];

  VUINT32 PeriphID0;      // 0xFFFFFFE0  
  VUINT32 PeriphID1;      // 0xFFFFFFE4  
  VUINT32 PeriphID2;      // 0xFFFFFFE8  
  VUINT32 PeriphID3;      // 0xFFFFFFEC  
};
// Define the interrupt request bitfields used by many of the
// interrupt control registers.  (partial list)
#define VIC_WDT_MASK     0x00000001
#define VIC_SWI_MASK     0x00000002  // reserved for software interrupts
#define VIC_TIMER0_MASK  0x00000010
#define VIC_TIMER1_MASK  0x00000020
#define VIC_UART0_MASK   0x00000040
#define VIC_UART1_MASK   0x00000080
#define VIC_SPI0_MASK    0x00000400
#define VIC_SPI1_MASK    0x00000800

#define VIC_WDT_CHANNEL  0
#define VIC_TIMER1_CHANNEL  5

/****************************************************************/
// Watchdog Timer

struct WDT_registers_t {
  VUINT8 WDMOD;                  // 0xe0000000  Watchdog Mode
  VUINT8 unused0[3];
#define WDMOD_WDEN_MASK     0x01 //   watchdog enable
#define WDMOD_WDRESET_MASK  0x02 //   reset enable (else an interrupt is generated)
#define WDMOD_WDTOF_MASK    0x04 //   time-out flag, true after watchdog-triggered reset
#define WDMOD_WDINT_MASK    0x08 //   interrupt flag

  VUINT32 WDTC;                  // 0xe0000004  Watchdog Timer Constant

  // VUINT8 WDFEED;                 // 0xe0000008  Watchdog Feed
  // VUINT8 unused1[3];
  VUINT32 WDFEED;                 // 0xe0000008  Watchdog Feed

  VUINT32 WDTV;                  // 0xe000000c  Watchdog Timer Value
};

/*****************************************************************/
// PCB - Pin Connect Block

struct PCB_registers_t {

  // These registers contain fields of 2-bit values that select one of
  // four functions for each of the I/O pins or groups of pins.

  VUINT32 PINSEL0;      // 0xE002C000  Pin function Selection register 0
  VUINT32 PINSEL1;      // 0xE002C004  Pin function Selection register 1
  VUINT32 unused0[3];
  VUINT32 PINSEL2;      // 0xE002C014  Pin function Selection register 2
};

extern void PCB_configure_pin( int pin, int function );

#define PINSEL0_MASK(pin, value) ((value) << (2*(pin)))
#define PINSEL1_MASK(pin, value) ((value) << (2*((pin) - 16)))


// Define names for the I/O pins. 
enum { PIN0_0 = 0,   PIN0_1,       PIN0_2,       PIN0_3,
       PIN0_4,       PIN0_5,       PIN0_6,       PIN0_7,
       PIN0_8,       PIN0_9,       PIN0_10,      PIN0_11,
       PIN0_12,      PIN0_13,      PIN0_14,      PIN0_15,
       PIN0_16,      PIN0_17,      PIN0_18,      PIN0_19,
       PIN0_20,      PIN0_21,      PIN0_22,      PIN0_23,
       PIN0_24,      PIN0_25,      PIN0_26,      PIN0_27,
       PIN0_28,      PIN0_29,      PIN0_30,      PIN0_31
};

/****************************************************************/
// General purpose I/O module.

#define GPIO_PIN_MASK(pin) (1 << (pin))

struct GPIO_registers_t {

  // This is a read-only register to read the GPIO pin states.
  // Actually, it could be written to set all pins states
  // simultaneously, but IOSET and IOCLR are less vulnerable to race
  // conditions.

  VUINT32 IOPIN;   // 0xE0028000 

  // Output set: a 1 produces a HIGH level for each GPIO pin
  // configured for output.  A 0 has no effect.  It is a readable
  // register.
  VUINT32 IOSET;   // 0xE0028004

  // Direction control bits for GPIO; 0 is input, 1 is output.
  VUINT32 IODIR;   // 0xE0028008

  // Output set: writing a 1 produces a LOW level for each GPIO pin
  // configured for output, and clears the corresponding bit in IOSET.
  // Writing a 0 has no effect.  This is a write-only register.
  VUINT32 IOCLR;   // 0xE002800C
  
};

/****************************************************************/
// UART0 and UART1 serial interfaces.  UART1 is identical to UART0, 
// with the addition of a modem interface.

struct UART_registers_t {

  // Reads as the Receiver Buffer Register, writes as THR, the Transmit Holding Register.
  // Also the DLL Divisor LSB Latch when DLAB is set.
  VUINT8 RBR;                // 0xe00XX000
  VUINT8 unused0[3];
#define THR RBR              // the UART receive/transmit data register
#define DLL RBR              // the UART clock divider LSB

  // Interrupt Enable Register. Also the DLM Divisor MSB Latch when DLAB is set.
  VUINT8 IER;                // 0xe00XX0004
  VUINT8 unused1[3];
#define IER_RBRIE_MASK  1    // RBR Interrupt Enable
#define IER_THREIE_MASK 2    // THRE Interrupt Enable
#define IER_RXLSIE_MASK 4    // RX Line Status Interrupt Enable
#define DLM IER              // the UART clock divider MSB

  // Reads as the Interrupt ID Register, writes as FCR, the FIFO Control Register
  VUINT8 IIR;                // 0xe00XX0008
  VUINT8 unused2[3];
#define IIR_IP_MASK    0x01  // Interrupt pending: 0 == at least one interrupt pending, 1 == no pending interrupts.
#define IIR_IID_MASK   0x0e  // Interrupt Identification.
#define IIR_FIFOE_MASK 0xc0  // Fifo Enable, equivalent to the FCR register available at this write address
#define FCR IIR              // the UART FIFO control register

  // Line Control Register
  VUINT8 LCR;                // 0xe00XX000c
  VUINT8 unused3[3];
  VUINT32 unused4;           // 0xe00XX0010

  // Line Status Register
  VUINT8 LSR;                // 0xe00XX0014
  VUINT8 unused5[3];
#define LSR_RDR_MASK   0x01  // Receiver Data Ready
#define LSR_OE_MASK    0x02  // Overrun Error
#define LSR_PE_MASK    0x04  // Parity Error
#define LSR_FE_MASK    0x08  // Framing Error
#define LSR_BI_MASK    0x10  // Break Interrupt
#define LSR_THRE_MASK  0x20  // Transitter Holding Register Empty
#define LSR_TEMT_MASK  0x40  // Transmitter Empty
#define LSR_RXFE_MASK  0x80  // Rx FIFO Error

  //  This only exists on UART1:
  VUINT8 MSR;                // 0xe000c018  Modem Status Register
  VUINT8 unused6[3];

  // Scratch Pad Register
  VUINT8 SCR;                // 0xe00XX001c
  VUINT8 unused7[19];

  // This doesn't exist on the LPC2106:
  VUINT8 TER;                // 0xe00XX0030 Transmit Enable Register
#define TER_TXEN_MASK  0x80  // true to enable

};
/****************************************************************/
// SPI (Serial Peripheral Interface).  The LPC2138 has an extended
// version of the SPI device in the LPC2106.

struct SPI_registers_t {
  VUINT16 SPCR;                // 0xE0020000  SPI Control Register
  VUINT16 unused0;
#define SPCR_BE_MASK   0x0004  //   bit enable, enables BITS field to select transfer length
#define SPCR_CPHA_MASK 0x0008  //   Clock Phase Control
#define SPCR_CPOL_MASK 0x0010  //   Clock Polarity Control
#define SPCR_MSTR_MASK 0x0020  //   Master mode select
#define SPCR_LSBF_MASK 0x0040  //   LSB First
#define SPCR_SPIE_MASK 0x0080  //   Serial Peripheral Interrupt Enable
#define SPCR_BITS_MASK 0x0f00  //   select the number of bits per transfer

  VUINT8 SPSR;                 // 0xE0020004  SPI Status Register
  VUINT8 unused1[3];
#define SPSR_ABRT_MASK  0x08   //   Slave Abort
#define SPSR_MODF_MASK  0x10   //   Mode Fault
#define SPSR_ROVR_MASK  0x20   //   Read Overrun
#define SPSR_WCOL_MASK  0x40   //   Write Collision
#define SPSR_SPIF_MASK  0x80   //   Serial Transfer Complete Flag

  VUINT16 SPDR;                // 0xE0020008  SPI Data Register
  VUINT16 unused2;

  VUINT8 SPCCR;                // 0xE002000C  SPI Clock Counter Register. An even value >= 8
  VUINT8 unused3[3];

  VUINT32 unused4[3];          // 0xE0020010 - 0xE002001B

  VUINT8 SPINT;                // 0xE002001C
#define SPINT_SPII_MASK 1      //   SPI interrupt flag, cleared by writing a 1 to this bit.

};

// Define names and function codes for configuring the SPI pins in the
// Pin Connect Block. 

#define SPI0_SCK_PIN          PIN0_4  // in PINSEL0
#define SPI0_MISO_PIN         PIN0_5
#define SPI0_MOSI_PIN         PIN0_6
#define SPI0_SSEL_PIN         PIN0_7

#define SPI0_SCK_FUNCTION     1
#define SPI0_MISO_FUNCTION    1
#define SPI0_MOSI_FUNCTION    1
#define SPI0_SSEL_FUNCTION    1

/****************************************************************/
// Synchronous Serial Port.  This can operate as SPI, 4-wire SSI, or
// Microwire.  It is a different architecture than the SPI module,
// it has a FIFO and more flexible transfer lengths.

struct SSP_registers_t {
  VUINT16 SSPCR0;               // 0xE0068000  SSP Control Register 0
  VUINT16 unused0;
#define SSPCR0_DSS_MASK  0x000f //   data size select
#define SSPCR0_FRF_MASK  0x0030 //   frame format
#define SSPCR0_CPHA_MASK 0x0040 //   clock phase control
#define SSPCR0_CPOL_MASK 0x0080 //   clock polarity control
#define SSPCR0_SCR_MASK  0xff00 //   serial clock rate; the bit frequency is PCLK / ( SSPCPSR * (SCR + 1))
#define SSPCR0_SCR_SHIFT 8

  VUINT8 SSPCR1;                // 0xE0068004  SSP Control Register 1
  VUINT8 unused1[3];
#define SSPCR1_LBM_MASK  0x01   //   loop back mode
#define SSPCR1_SSE_MASK  0x02   //   SSP enable
#define SSPCR1_MS_MASK   0x04   //   master/slave mode select
#define SSPCR1_SOD_MASK  0x08   //   slave output disable

  VUINT16 SSPDR;                // 0xE0068008  SSP Data Register
  VUINT16 unused2;

  VUINT8 SSPSR;                 // 0xE006800c  SSP Status Register
  VUINT8 unused3[3];
#define SSPSR_TFE_MASK   0x01   //   transmit fifo empty
#define SSPSR_TNF_MASK   0x02   //   transmit fifo not full
#define SSPSR_RNE_MASK   0x04   //   receive fifo not empty
#define SSPSR_RFF_MASK   0x08   //   receive fifo full
#define SSPSR_BSY_MASK   0x10   //   SSP active

  VUINT8 SSPCPSR;               // 0xE0068010  SSP Clock Prescale Register.  Even number >= 2.
  VUINT8 unused4[3];

  VUINT8 SSPIMSC;               // 0xE0068014  SSP Interrupt Mask Set and Clear Register
  VUINT8 unused5[3];
#define SSPIMSC_RORIM_MASK 0x01   // enable receive overrun interrupt
#define SSPIMSC_RTIM_MASK  0x02   // enable receive timeout interrupt
#define SSPIMSC_RXIM_MASK  0x04   // enable receive FIFO half-full interrupt
#define SSPIMSC_TXIM_MASK  0x08   // enable transmit FIFO half-empty interrupt

  VUINT8 SSPRIS;                // 0xE0068018  SSP Raw Interrupt Status Register
  VUINT8 unused6[3];
#define SSPRIS_RORRIS_MASK 0x01   // receiver overrun flag
#define SSPRIS_RTRIS_MASK  0x02   // receive timeout flag
#define SSPRIS_RXRIS_MASK  0x04   // receive FIFO half-full flag
#define SSPRIS_TXRIS_MASK  0x08   // transmit FIFO half-empty flag

  VUINT8 SSPMIS;                // 0xE006801c  SSP Masked Interrupt Status Register
  VUINT8 unused7[3];
#define SSPMIS_RORMIS_MASK 0x01   // receiver overrun flag
#define SSPMIS_RTMIS_MASK  0x02   // receive timeout flag
#define SSPMIS_RXMIS_MASK  0x04   // receive FIFO half-full flag
#define SSPMIS_TXMIS_MASK  0x08   // transmit FIFO half-empty flag

  VUINT8 SSPICR;                // 0xE0068020  SSP Interrupt Clear Register
#define SSPICR_RORIC_MASK  0x01 //  clear receiver overrun flag
#define SSPICR_RTIC_MASK   0x02 //  clear receiver timeout flag

};

// Define names and function codes for configuring the SPI pins in the
// Pin Connect Block. 

#define SPI1_SCK_PIN          PIN0_17 // in PINSEL1 
#define SPI1_MISO_PIN         PIN0_18
#define SPI1_MOSI_PIN         PIN0_19
#define SPI1_SSEL_PIN         PIN0_20

#define SPI1_SCK_FUNCTION     2
#define SPI1_MISO_FUNCTION    2
#define SPI1_MOSI_FUNCTION    2
#define SPI1_SSEL_FUNCTION    2

/****************************************************************/
// The general purpose timer module.  
//  Timer0 and Timer1 are 32 bit timers.
//  On some variants, Timer3 and Timer4 are 16 bit timers following the same layout.

struct TIMER_registers_t {
  VUINT32 IR;                   // 0xE000x000  Interrupt Register

  // each bit identifies an interrupt source, and can be written to clear the interrupt
#define TIMER_IR_MR0_MASK 0x01  // interrupt flag for match channel 0
#define TIMER_IR_MR1_MASK 0x02  // interrupt flag for match channel 1
#define TIMER_IR_MR2_MASK 0x04  // interrupt flag for match channel 2
#define TIMER_IR_MR3_MASK 0x08  // interrupt flag for match channel 3
#define TIMER_IR_CR0_MASK 0x10  // interrupt flag for capture channel 0 event
#define TIMER_IR_CR1_MASK 0x20  // interrupt flag for capture channel 1 event
#define TIMER_IR_CR2_MASK 0x40  // interrupt flag for capture channel 2 event
#define TIMER_IR_CR3_MASK 0x80  // interrupt flag for capture channel 3 event

  VUINT32 TCR;                  // 0xE000x004  Timer Control Register
#define TIMER_TCR_ENABLE_MASK  0x01
#define TIMER_TCR_RESET_MASK   0x02

  VUINT32 TC; 			   // 0xE000x008  Timer Counter. This is a 32 bit counter.          
  VUINT32 PR; 			   // 0xE000x00C  Prescale Register. This is a 32 bit divisor.      
  VUINT32 PC; 			   // 0xE000x010  Prescale Counter.  This is a 32 bit counter.      

  VUINT32 MCR;                     // 0xE000x014  Match Control Register
#define TIMER_MCR_IMR0_MASK 0x001  // flag to enable interrupt when MR0 matches TC
#define TIMER_MCR_RMR0_MASK 0x002  // flag to enable TC reset when MR0 matches TC
#define TIMER_MCR_SMR0_MASK 0x004  // flag to enable stopping TC and PC when MR0 matches TC

#define TIMER_MCR_IMR1_MASK 0x008  // flag to enable interrupt when MR1 matches TC
#define TIMER_MCR_RMR1_MASK 0x010  // flag to enable TC reset when MR1 matches TC
#define TIMER_MCR_SMR1_MASK 0x020  // flag to enable stopping TC and PC when MR1 matches TC

#define TIMER_MCR_IMR2_MASK 0x040  // flag to enable interrupt when MR2 matches TC
#define TIMER_MCR_RMR2_MASK 0x080  // flag to enable TC reset when MR2 matches TC
#define TIMER_MCR_SMR2_MASK 0x100  // flag to enable stopping TC and PC when MR2 matches TC

#define TIMER_MCR_IMR3_MASK 0x200  // flag to enable interrupt when MR3 matches TC
#define TIMER_MCR_RMR3_MASK 0x400  // flag to enable TC reset when MR3 matches TC
#define TIMER_MCR_SMR3_MASK 0x800  // flag to enable stopping TC and PC when MR0 matches TC

  VUINT32 MR0;  		   // 0xE000x018  Match Register 0
  VUINT32 MR1;  		   // 0xE000x01C  Match Register 1
  VUINT32 MR2;  		   // 0xE000x020  Match Register 2
  VUINT32 MR3;  		   // 0xE000x024  Match Register 3


  VUINT32 CCR;                     // 0xE000x028  Capture Control Register
#define TIMER_CCR_CC0R_MASK 0x001  // flag to enable CR0 capture (loading CR0 with TC) on capture[0] rising edge
#define TIMER_CCR_CC0F_MASK 0x002  // flag to enable CR0 capture (loading CR0 with TC) on capture[0] falling edge
#define TIMER_CCR_ICR0_MASK 0x004  // flag to enable interupt on a CR0 capture event

#define TIMER_CCR_CC1R_MASK 0x008  // flag to enable CR1 capture (loading CR1 with TC) on capture[1] rising edge
#define TIMER_CCR_CC1F_MASK 0x010  // flag to enable CR1 capture (loading CR1 with TC) on capture[1] falling edge
#define TIMER_CCR_ICR1_MASK 0x020  // flag to enable interupt on a CR1 capture event

#define TIMER_CCR_CC2R_MASK 0x040  // flag to enable CR2 capture (loading CR2 with TC) on capture[2] rising edge
#define TIMER_CCR_CC2F_MASK 0x080  // flag to enable CR2 capture (loading CR2 with TC) on capture[2] falling edge
#define TIMER_CCR_ICR2_MASK 0x100  // flag to enable interupt on a CR2 capture event

      // The following capture[3] definition is only valid on timer 1:
#define TIMER_CCR_CC3R_MASK 0x200  // flag to enable CR3 capture (loading CR3 with TC) on capture[3] rising edge
#define TIMER_CCR_CC3F_MASK 0x400  // flag to enable CR3 capture (loading CR3 with TC) on capture[3] falling edge
#define TIMER_CCR_ICR3_MASK 0x800  // flag to enable interupt on a CR3 capture event

  VUINT32 CR0;   		   // 0xE000x02C  Capture Register 0
  VUINT32 CR1;   		   // 0xE000x030  Capture Register 1
  VUINT32 CR2;   		   // 0xE000x034  Capture Register 2
  VUINT32 CR3;   		   // 0xE000x038  Capture Register 3

  VUINT32 EMR;                     // 0xE000x03C  External Match Register
#define TIMER_EMR_EM0_MASK 0x001   // external match 0, state of output MATn.0
#define TIMER_EMR_EM1_MASK 0x002   // external match 1, state of output MATn.1
#define TIMER_EMR_EM2_MASK 0x004   // external match 2, state of output MATn.2
#define TIMER_EMR_EM3_MASK 0x008   // external match 3, state of output MATn.3
      // For each of the following the value is defined as follows:
      //  0 do nothing
      //  1 clear  corresponding external match output
      //  2 set    corresponding external match output
      //  3 toggle corresponding external match output
#define TIMER_EMR_EMC0_MASK 0x030  // external match control 0
#define TIMER_EMR_EMC1_MASK 0x0c0  // external match control 1
#define TIMER_EMR_EMC2_MASK 0x300  // external match control 2
#define TIMER_EMR_EMC3_MASK 0xc00  // external match control 3
#define TIMER_EMR_EMC_MASK(pin,value) ((value)<<(4+(2*(pin))))

  VUINT32 pad1[12];                // 0xE000x040 through 0xE000x06c

  // This register is included on at least the following chips    : LPC2138, LPC2103
  VUINT32 CTCR;			   // 0xE000x070  Count Control Register

  // This register is included on at least the following chips    : LPC2103
  // This register is not included on at least the following chips: LPC2138
#define TIMER_PWMCON_PWMEN0_MASK 0x00000001 // PWM enable for MATn.0
#define TIMER_PWMCON_PWMEN1_MASK 0x00000002 // PWM enable for MATn.1
#define TIMER_PWMCON_PWMEN2_MASK 0x00000004 // PWM enable for MATn.2
#define TIMER_PWMCON_PWMEN3_MASK 0x00000008 // PWM enable for MATn.3

  VUINT32 PWMCON;		   // 0xE000x074  PWM Control Register
};
/****************************************************************/
// PWM module.  This is very similar to Timer0 or Timer1, but is not identical.
struct PWM_registers_t {
  VUINT32 IR;                   // 0xE001x000  Interrupt Register

  // each bit identifies an interrupt source, and can be written to clear the interrupt
#define PWM_IR_MR0_MASK 0x0001  // interrupt flag for match channel 0
#define PWM_IR_MR1_MASK 0x0002  // interrupt flag for match channel 1
#define PWM_IR_MR2_MASK 0x0004  // interrupt flag for match channel 2
#define PWM_IR_MR3_MASK 0x0008  // interrupt flag for match channel 3
#define PWM_IR_MR4_MASK 0x0100  // interrupt flag for match channel 4
#define PWM_IR_MR5_MASK 0x0200  // interrupt flag for match channel 5
#define PWM_IR_MR6_MASK 0x0400  // interrupt flag for match channel 6

  VUINT32 TCR;                  // 0xE001x004  Timer Control Register
#define PWM_TCR_ENABLE_MASK  0x01
#define PWM_TCR_RESET_MASK   0x02
#define PWM_TCR_PWM_ENABLE   0x08

  VUINT32 TC; 			   // 0xE001x008  Timer Counter. This is a 32 bit counter.          
  VUINT32 PR; 			   // 0xE001x00C  Prescale Register. This is a 32 bit divisor.      
  VUINT32 PC; 			   // 0xE001x010  Prescale Counter.  This is a 32 bit counter.      

  VUINT32 MCR;                     // 0xE001x014  Match Control Register
#define PWM_MCR_IMR0_MASK 0x000001  // flag to enable interrupt when MR0 matches TC
#define PWM_MCR_RMR0_MASK 0x000002  // flag to enable TC reset when MR0 matches TC
#define PWM_MCR_SMR0_MASK 0x000004  // flag to enable stopping TC and PC when MR0 matches TC

#define PWM_MCR_IMR1_MASK 0x000008  // flag to enable interrupt when MR1 matches TC
#define PWM_MCR_RMR1_MASK 0x000010  // flag to enable TC reset when MR1 matches TC
#define PWM_MCR_SMR1_MASK 0x000020  // flag to enable stopping TC and PC when MR1 matches TC

#define PWM_MCR_IMR2_MASK 0x000040  // flag to enable interrupt when MR2 matches TC
#define PWM_MCR_RMR2_MASK 0x000080  // flag to enable TC reset when MR2 matches TC
#define PWM_MCR_SMR2_MASK 0x000100  // flag to enable stopping TC and PC when MR2 matches TC

#define PWM_MCR_IMR3_MASK 0x000200  // flag to enable interrupt when MR3 matches TC
#define PWM_MCR_RMR3_MASK 0x000400  // flag to enable TC reset when MR3 matches TC
#define PWM_MCR_SMR3_MASK 0x000800  // flag to enable stopping TC and PC when MR0 matches TC

#define PWM_MCR_IMR4_MASK 0x001000  // flag to enable interrupt when MR4 matches TC
#define PWM_MCR_RMR4_MASK 0x002000  // flag to enable TC reset when MR4 matches TC
#define PWM_MCR_SMR4_MASK 0x004000  // flag to enable stopping TC and PC when MR0 matches TC

#define PWM_MCR_IMR5_MASK 0x008000  // flag to enable interrupt when MR5 matches TC
#define PWM_MCR_RMR5_MASK 0x010000  // flag to enable TC reset when MR5 matches TC
#define PWM_MCR_SMR5_MASK 0x020000  // flag to enable stopping TC and PC when MR0 matches TC

#define PWM_MCR_IMR6_MASK 0x040000  // flag to enable interrupt when MR6 matches TC
#define PWM_MCR_RMR6_MASK 0x080000  // flag to enable TC reset when MR6 matches TC
#define PWM_MCR_SMR6_MASK 0x100000  // flag to enable stopping TC and PC when MR0 matches TC

  VUINT32 MR0;  		   // 0xE001x018  Match Register 0
  VUINT32 MR1;  		   // 0xE001x01C  Match Register 1
  VUINT32 MR2;  		   // 0xE001x020  Match Register 2
  VUINT32 MR3;  		   // 0xE001x024  Match Register 3
  VUINT32 pad1[ 6 ];
  VUINT32 MR4;  		   // 0xE001x040  Match Register 4
  VUINT32 MR5;  		   // 0xE001x044  Match Register 5
  VUINT32 MR6;  		   // 0xE001x048  Match Register 6

  VUINT32 PCR;                     // 0xE001x04C  PWM Control Register
#define PWM_PCR_PWMSEL2_MASK 0x0004
#define PWM_PCR_PWMSEL3_MASK 0x0008
#define PWM_PCR_PWMSEL4_MASK 0x0010
#define PWM_PCR_PWMSEL5_MASK 0x0020
#define PWM_PCR_PWMSEL6_MASK 0x0040
#define PWM_PCR_PWMENA1_MASK 0x0200
#define PWM_PCR_PWMENA2_MASK 0x0400
#define PWM_PCR_PWMENA3_MASK 0x0800
#define PWM_PCR_PWMENA4_MASK 0x1000
#define PWM_PCR_PWMENA5_MASK 0x2000
#define PWM_PCR_PWMENA6_MASK 0x4000

  VUINT32 LER;                     // 0xE001x050  PWM Latch Enable Register
#define PWM_LER_ENABLE_LATCH0   0x01
#define PWM_LER_ENABLE_LATCH1   0x02
#define PWM_LER_ENABLE_LATCH2   0x04
#define PWM_LER_ENABLE_LATCH3   0x08
#define PWM_LER_ENABLE_LATCH4   0x10
#define PWM_LER_ENABLE_LATCH5   0x20
#define PWM_LER_ENABLE_LATCH6   0x40

};


/****************************************************************/
// A/D Converter
struct ADC_registers_t {
  VUINT32 ADCR;                       // 0xe00XX000  A/D Control Register
#define ADCR_SEL_MASK     0x000000ff  //   bitmask to select which inputs are converted
#define ADCR_CLKDIV_MASK  0x0000ff00  //   divider for PCLK to produce the A/D conversion clock (max 4.5 MHz)
#define ADCR_BURST_MASK   0x00010000  //   enables automatically repeated conversions
#define ADCR_CLKS_MASK    0x000e0000  //   select number of conversion clocks, i.e. accuracy
#define ADCR_PDN_MASK     0x00200000  //   1 to active converter, 0 to power down
#define ADCR_START_MASK   0x07000000  //   define trigger conditions
#define ADCR_EDGE_MASK    0x08000000  //   define edge trigger conditions for external trigger inptus

  VUINT32 ADDR;                       // 0xe00XX004  A/D Data Register
#define ADDR_DATA_MASK    0x0000ffc0  //   10 bit result value; 0 == VSSA, 0x3ff == VREF
#define ADDR_DATA_SHIFT   6           //   offset of LSB from bit 0
#define ADDR_CHN_MASK     0x07000000  //   the source channel of the converted data
#define ADDR_OVERRUN_MASK 0x40000000  //   in burst mode indicates if a result was not read
#define ADDR_DONE_MASK    0x80000000  //   indicates a conversion is complete
#define ADC_MAX_VALUE      0x03ff
#define ADC_MIN_VALUE      0x0000

  // The ADGSR is shared between both A/D converters and exists at AD0.ADGSR but not AD1.ADGSR.
  VUINT8 ADGSR;                       // 0xe0034008  A/D Global Start Register
#define ADGSR_BURST_MASK  0x00010000  //   enable repeated conversions on both A/Ds
#define ADGSR_START_MASK  0x07000000  //   define trigger conditions for both A/Ds
#define ADGSR_EDGE_MASK   0x08000000  //   define edge trigger conditions for both A/Ds

};

/****************************************************************/
// D/A Converter.  It's only one register, but it's put in a structure for consistency.
struct DAC_registers_t {
  VUINT32 DACR;                       // 0xE006C000   DAC Register
#define DACR_DATA_MASK    0x0000ffc0  //   10 bit output value; 0 == VSSA, 0x3ff == VREF
#define DACR_DATA_SHIFT   6           //   offset of LSB from bit 0
#define DACR_BIAS_MASK    0x00010000  //   0 for high current and fast settling time; 1 for low current, slow settling
};

// PCB values
#define DAC0_PIN       PIN0_25
#define DAC0_FUNCTION  2

/****************************************************************/
// Real Time Clock

// For simplicity, all the registers are declared as VUINT32, although
// very few are that wide, since all fall on four byte boundaries.

struct RTC_registers_t {
  //      register       width  description				mode	address  

  // Miscellaneous Register Group
  VUINT32 ILR;          // 2	Interrupt Location Register		R/W  	0xE002 4000 
  VUINT32 CTC;		// 15	Clock Tick Counter			RO  	0xE002 4004 
  VUINT32 CCR;		// 4	Clock Control Register			R/W  	0xE002 4008 
  VUINT32 CIIR;		// 8	Counter Increment Interrupt Register	R/W  	0xE002 400C 
  VUINT32 AMR;		// 8	Alarm Mask Register			R/W  	0xE002 4010 
  VUINT32 CTIME0;	// 32	Consolidated Time Register 0		RO  	0xE002 4014 
  VUINT32 CTIME1;	// 32	Consolidated Time Register 1		RO  	0xE002 4018 
  VUINT32 CTIME2;	// 32	Consolidated Time Register 2		RO  	0xE002 401C 

#define CCR_CLKEN  0x00000001
#define CCR_CTCRST 0x00000002
#define CCR_CTTEST 0x0000000c
#define CCR_CLKSRC 0x00000010

#define CTIME0_SEC    0x0000003f
#define CTIME0_MIN    0x00003f00
#define CTIME0_HOUR   0x001f0000
#define CTIME0_DOW    0x07000000  // day of week
#define CTIME1_DOM    0x0000001f  // day of month
#define CTIME1_MONTH  0x00000f00
#define CTIME1_YEAR   0x0fff0000 
#define CTIME2_DOY    0x00000fff  // day of year

  // Time Counter Group
  VUINT32 SEC;		// 6	Seconds Counter				R/W  	0xE002 4020 
  VUINT32 MIN;		// 6	Minutes Register			R/W  	0xE002 4024 
  VUINT32 HOUR;		// 5	Hours Register				R/W  	0xE002 4028 
  VUINT32 DOM;		// 5	Day of Month Register			R/W  	0xE002 402C 
  VUINT32 DOW;		// 3	Day of Week Register			R/W  	0xE002 4030 
  VUINT32 DOY;		// 9	Day of Year Register			R/W  	0xE002 4034 
  VUINT32 MONTH;	// 4	Months Register				R/W  	0xE002 4038 
  VUINT32 YEAR;		// 12	Years Register				R/W  	0xE002 403C 

  VUINT32 unused0[8];   //                                                      0xE002 4040 - 0xE002 405C

  // Alarm Register Group
  VUINT32 ALSEC;	// 6	Alarm value for Seconds			R/W  	0xE002 4060 
  VUINT32 ALMIN;	// 6	Alarm value for Minutes			R/W  	0xE002 4064 
  VUINT32 ALHOUR;	// 5	Alarm value for Seconds			R/W  	0xE002 4068 
  VUINT32 ALDOM;	// 5	Alarm value for Day of Month		R/W  	0xE002 406C 
  VUINT32 ALDOW;	// 3	Alarm value for Day of Week		R/W  	0xE002 4070 
  VUINT32 ALDOY;	// 9	Alarm value for Day of Year		R/W  	0xE002 4074 
  VUINT32 ALMON;	// 4	Alarm value for Months			R/W  	0xE002 4078 
  VUINT32 ALYEAR;	// 12	Alarm value for Year			R/W 	0xE002 407C 

  // Reference Clock Divider
  VUINT32 PREINT;	// 13	Prescaler value, integer portion	R/W 	0xE002 4080 
  VUINT32 PREFRAC;	// 15	Prescaler value, integer portion	R/W 	0xE002 4084

};

#endif // __LPC2xxx_H_INCLUDED__
